package com.segway.robot.algo.tf;

import android.os.Parcel;
import android.os.Parcelable;

/**
 * Created by Yusen.QIN on 2017/10/16.
 */

public class Quaternion implements Parcelable {
    public float x;
    public float y;
    public float z;
    public float w;

    protected Quaternion(Parcel in) {
        x = in.readFloat();
        y = in.readFloat();
        z = in.readFloat();
        w = in.readFloat();
    }

    public static final Creator<Quaternion> CREATOR = new Creator<Quaternion>() {
        @Override
        public Quaternion createFromParcel(Parcel in) {
            return new Quaternion(in);
        }

        @Override
        public Quaternion[] newArray(int size) {
            return new Quaternion[size];
        }
    };

    public Quaternion set (float x, float y, float z, float w) {
        this.x = x;
        this.y = y;
        this.z = z;
        this.w = w;
        return this;
    }

    public Quaternion (float x, float y, float z, float w) {
        this.set(x, y, z, w);
    }

    public Quaternion set (Quaternion quaternion) {
        return this.set(quaternion.x, quaternion.y, quaternion.z, quaternion.w);
    }

    /** Get the if there has any pole of the gimbal lock.
     * reference : https://en.wikipedia.org/wiki/Gimbal_lock
     * @return 1: north pole, -1:outh pole, 0:no gimbal lock */
    public int getGimbalPole () {
        final float t = y * x + z * w;
        if (t > 0.49999f){
            return 1;
        }
        if(t < -0.49999f){
            return -1;
        }
        return 0;
    }

    /** Get the roll euler angle in radians, which is the rotation around the z axis. Requires that this quaternion is normalized.
     * Get
     * @return the rotation around the z axis in radians (between -PI and +PI) */
    public float getRollRad () {
        final int pole = getGimbalPole();
        if (pole == 0){
            return MathUtils.atan2(2f * (w * z + y * x), 1f - 2f * (x * x + z * z));
        } else {
            return (float)pole * 2f* MathUtils.atan2(y, w);
        }
    }

    /** Get the pitch euler angle in radians, which is the rotation around the x axis. Requires that this quaternion is normalized.
     * @return the rotation around the x axis in radians (between -(PI/2) and +(PI/2)) */
    public float getPitchRad () {
        final int pole = getGimbalPole();
        return pole == 0 ? (float)Math.asin(MathUtils.clamp(2f * (w * x - z * y), -1f, 1f)) : (float)pole * MathUtils.PI * 0.5f;
    }

    /** Get the yaw euler angle in radians, which is the rotation around the y axis. Requires that this quaternion is normalized.
     * @return the rotation around the y axis in radians (between -PI and +PI) */
    public float getYawRad () {
        return getGimbalPole() == 0 ? MathUtils.atan2(2f * (y * w + x * z), 1f - 2f * (y * y + x * x)) : 0f;
    }

    /**
     * get sample with roll = getDegreeFromRadians(q.getRollRad())
     * @param degree
     * @return degree
     */
    public float getDegreeFromRadians(float degree){
        return degree * MathUtils.radiansToDegrees;
    }

    public Quaternion setEulerRad (float yaw, float pitch, float roll) { //XYZ axis
        final float halfRoll = roll * 0.5f;
        final float sHalfRoll = (float)Math.sin(halfRoll);
        final float cHalfRoll = (float)Math.cos(halfRoll);
        final float halfPitch = pitch * 0.5f;
        final float sHalfPitch = (float)Math.sin(halfPitch);
        final float cHalfPitch = (float)Math.cos(halfPitch);
        final float halfYaw = yaw * 0.5f;
        final float sHalfYaw = (float)Math.sin(halfYaw);
        final float cHalfYaw = (float)Math.cos(halfYaw);
        final float cHalfYaw_sHalfPitch = cHalfYaw * sHalfPitch;
        final float sHalfYaw_cHalfPitch = sHalfYaw * cHalfPitch;
        final float cHalfYaw_cHalfPitch = cHalfYaw * cHalfPitch;
        final float sHalfYaw_sHalfPitch = sHalfYaw * sHalfPitch;

        x = (cHalfYaw_sHalfPitch * cHalfRoll) + (sHalfYaw_cHalfPitch * sHalfRoll);
        y = (sHalfYaw_cHalfPitch * cHalfRoll) - (cHalfYaw_sHalfPitch * sHalfRoll);
        z = (cHalfYaw_cHalfPitch * sHalfRoll) - (sHalfYaw_sHalfPitch * cHalfRoll);
        w = (cHalfYaw_cHalfPitch * cHalfRoll) + (sHalfYaw_sHalfPitch * sHalfRoll);
        /* reference C++ CODE is from tf/include/linear_math/Quaternion.h line 75
        which is the same to the function setEuler(...)
        setValue(cosRoll * sinPitch * cosYaw + sinRoll * cosPitch * sinYaw, // x
			cosRoll * cosPitch * sinYaw - sinRoll * sinPitch * cosYaw,      // y
			sinRoll * cosPitch * cosYaw - cosRoll * sinPitch * sinYaw,      // z
			cosRoll * cosPitch * cosYaw + sinRoll * sinPitch * sinYaw);     // w
         */
        return this;
    }

    public Quaternion setEulerRadZYX(float yaw, float pitch, float roll){
        this.setEulerRad(roll,pitch,yaw);
        return this;
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeFloat(x);
        dest.writeFloat(y);
        dest.writeFloat(z);
        dest.writeFloat(w);
    }

    @Override
    public String toString() {
        return "Quaternion{" +
                "x=" + x +
                ", y=" + y +
                ", z=" + z +
                ", w=" + w +
                '}';
    }
}
