package com.segway.robot.sdk.baseconnectivity;

/**
 * This class provides methods to set the message listener and send messages to
 * the specific applications. It is a base class for {@code MobileMessageConnection} and
 * {@code RobotMessageConnection} implementations. It is recommended to use MobileMessageConnection
 * in mobile applications and RobotMessageConnection in robot applications.
 */
public abstract class MessageConnection {

    public abstract String getName();

    public abstract void setListeners(ConnectionStateListener connectionStateListener, MessageListener messageListener) throws Exception;

    public abstract MessageListener getMessageListener();

    public abstract ConnectionStateListener getConnectionStateListener();

    public abstract void removeMessageListener();

    public abstract void sendMessage(Message message) throws Exception;

    /**
     * This interface provides callbacks for the message state of sending messages and receiving messages.
     */
    public interface MessageListener {

        /**
         * This method is called when the MessageConnection fails to send messages.
         *
         * @param message the message that fails to be sent.
         * @param error   the detailed error information about why the message fails to be sent.
         */
        void onMessageSentError(Message message, String error);

        /**
         * This method is called when the MessageConnection sends messages successfully.
         *
         * @param message the message that is sent successfully.
         */
        void onMessageSent(Message message);

        /**
         * This method is called when the MessageConnection receives message successfully.
         *
         * @param message the message that is received successfully.
         */
        void onMessageReceived(Message message);
    }

    /**
     * This interface provides callbacks for the connection state of MessageConnection.
     */
    public interface ConnectionStateListener {

        /**
         * This method is called when the MessageConnection between the specific robot application and the mobile application are created.
         */
        void onOpened();

        /**
         * This method is called when the MessageConnection between the specific robot application and the mobile application are closed.
         *
         * @param error the detailed error information about why the MessageConnection is closed.
         */
        void onClosed(String error);
    }
}