package com.sdk.wisetracker.base.tracker.data.init;

import android.text.TextUtils;

import com.sdk.wisetracker.base.tracker.common.log.WiseLog;
import com.sdk.wisetracker.base.tracker.data.manager.BasicData;
import com.sdk.wisetracker.base.tracker.data.manager.BasicDataManager;
import com.sdk.wisetracker.base.tracker.data.manager.SessionDataManager;
import com.sdk.wisetracker.base.tracker.data.model.Const;
import com.sdk.wisetracker.base.tracker.data.model.Session;
import com.sdk.wisetracker.base.tracker.data.model.Token;
import com.sdk.wisetracker.base.tracker.network.SessionApi;
import com.sdk.wisetracker.base.tracker.network.SessionRetrofit;
import com.sdk.wisetracker.base.tracker.network.type.HeaderType;
import com.sdk.wisetracker.base.tracker.util.AES256Util;

import java.io.UnsupportedEncodingException;
import java.security.InvalidAlgorithmParameterException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;
import java.util.List;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;

import io.reactivex.Observable;

public class InitToken {

    private static InitToken instance = null;

    public static InitToken getInstance() {
        if (instance == null) {
            instance = new InitToken();
        }
        return instance;
    }

    /**
     * AccessToken 정보 호출
     * @return observable
     */
    Observable<Token> getAccessTokenObservable() {

        try {

            BasicData basicData = BasicDataManager.getInstance().getBasicData();
            Session session = SessionDataManager.getInstance().getSession();

            String tokenString = basicData.getToken();
            long tokenExpireTime = basicData.getTokenExpireTime();
            if (!TextUtils.isEmpty(tokenString) && tokenExpireTime > System.currentTimeMillis()) {

                Token token = new Token();
                token.setRawData(tokenString);
                token.setExpireTime(tokenExpireTime);
                return Observable.just(token);

            } else {

                String serviceNumber = String.valueOf(session.getServiceNumber());
                String packageName = session.getPackageName();

                SessionApi api = SessionRetrofit.getInstance().getBaseApi(SessionApi.class, HeaderType.TOKEN);
                Observable<Token> accessTokenObservable = api.getAccessToken(serviceNumber, packageName);
                return accessTokenObservable;

            }

        } catch (Exception e) {
            WiseLog.d("get access token observable error !!" + e.getMessage());
        }

        return null;

    }

    /**
     * 응답 받은 Token 유효성 체크 및 Token 정보 저장
     * @param token
     * @return Token 유효성에 따라 return
     */
    boolean isValidToken(Token token) {

        try {

            if (token == null || TextUtils.isEmpty(token.getRawData())) {
                WiseLog.d("token is null or empty");
                return false;
            }

            String decodingString = AES256Util.getInstance().aesDecode(Const.TOKEN_DECODE_KEY, token.getRawData());
            String[] splitString = decodingString.split("#");
            List<String> keyList = Arrays.asList(splitString);
            if (keyList == null || keyList.isEmpty()) {
                WiseLog.d("key list is null or empty");
                return false;
            }

            BasicData basicData = BasicDataManager.getInstance().getBasicData();
            long expireTime = Long.valueOf(keyList.get(1));
            long currentTime = System.currentTimeMillis() + basicData.getTimeDiff();
            WiseLog.d("ntpTime Result : " + expireTime +"/"+ currentTime +"=>"+ (expireTime - currentTime) );
            if (expireTime < currentTime) {
                WiseLog.d("token is expired");
                return false;
            }else{
                WiseLog.d("ntpTime Success");
            } 
            basicData.setToken(token.getRawData());
            basicData.setTokenExpireTime(expireTime);
            BasicDataManager.getInstance().saveBasicData(basicData);
            //SessionDB.getInstance().setPrimitiveData(SessionDB.PARAM.INIT.ACCESS_TOKEN, token.getRawData());
            //SessionDB.getInstance().setPrimitiveData(SessionDB.PARAM.INIT.ACCESS_TOKEN_EXPIRE_TIME, token.getExpireTime());
            //token.setExpireTime(expireTime);
            return true;

        } catch (UnsupportedEncodingException e) {
            WiseLog.d("UnsjupportedEncodingException, " + e.getMessage());
        } catch (NoSuchAlgorithmException e) {
            WiseLog.d("NoSuchAlgorithmException, " + e.getMessage());
        } catch (NoSuchPaddingException e) {
            WiseLog.d("NoSuchPaddingException, " + e.getMessage());
        } catch (InvalidKeyException e) {
            WiseLog.d("InvalidKeyException, " + e.getMessage());
        } catch (InvalidAlgorithmParameterException e) {
            WiseLog.d("InvalidAlgorithmParameterException, " + e.getMessage());
        } catch (IllegalBlockSizeException e) {
            WiseLog.d("IllegalBlockSizeException, " + e.getMessage());
        } catch (BadPaddingException e) {
            WiseLog.d("BadPaddingException, " + e.getMessage());
        } catch (Exception e) {
            WiseLog.e(e);
        }

        return false;

    }

}
