package com.sdk.wisetracker.base.tracker.data.background;

import android.annotation.SuppressLint;
import android.content.Context;

import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.google.gson.Gson;
import com.google.gson.JsonObject;
import com.sdk.wisetracker.base.tracker.common.WisetrackerDatabase;
import com.sdk.wisetracker.base.tracker.common.log.WiseLog;
import com.sdk.wisetracker.base.tracker.data.manager.BasicData;
import com.sdk.wisetracker.base.tracker.data.manager.BasicDataManager;
import com.sdk.wisetracker.base.tracker.data.manager.SessionDataManager;
import com.sdk.wisetracker.base.tracker.data.model.Session;
import com.sdk.wisetracker.base.tracker.network.SessionApi;
import com.sdk.wisetracker.base.tracker.network.SessionRetrofit;
import com.sdk.wisetracker.base.tracker.network.request.RequestRetention;
import com.sdk.wisetracker.base.tracker.network.type.HeaderType;

import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Action;
import io.reactivex.functions.Consumer;
import io.reactivex.schedulers.Schedulers;

/**
 * 리텐션 체크 위한 백그라운드 Worker
 */
public class RetentionWorker extends Worker {

    private final String RETENTION_COUNT = "RETENTION_COUNT";
    private final String LAST_RETENTION_TIME = "LAST_RETENTION_TIME";


    public RetentionWorker(Context context, WorkerParameters workerParams) {
        super(context, workerParams);
    }

     // WorkManager 등록시 즉시 백그라운드 Worker 동작으로 인해 최초 1회 이후 리텐션 전송
    @Override
    public Result doWork() {

        try {

            // TODO  : 동일 단말기에 여러개의 sdk가 적용된 앱이 존재할 수 있음을 고려 필요.
            WiseLog.d("retention worker execute");
            WisetrackerDatabase.init(getApplicationContext());

            int retentionCount = WisetrackerDatabase.getIntValue(RETENTION_COUNT);
            WiseLog.d("retention count = " + retentionCount);
            if (retentionCount > 0) {
                long lastRetentionTime = WisetrackerDatabase.getLongValue(LAST_RETENTION_TIME);
                if (lastRetentionTime == 0) {
                    WiseLog.d("retention diff time = " + lastRetentionTime);
                } else {
                    WiseLog.d("retention diff time = " + ((System.currentTimeMillis() - lastRetentionTime) / 1000 * 60) + "minute");
                }
                WisetrackerDatabase.setDatabase(LAST_RETENTION_TIME, System.currentTimeMillis());
                sendRetention();
            }
            WisetrackerDatabase.setDatabase(RETENTION_COUNT, ++retentionCount);

        } catch (Exception e) {
            WiseLog.e(e);
        }

        return Result.success();

    }

    // 리텐션 데이터 전송
    @SuppressLint("CheckResult")
    private void sendRetention() {
        WiseLog.d("send retention");
        SessionApi api = SessionRetrofit.getInstance().getBaseApi(SessionApi.class, HeaderType.DOT);
        api.sendRetention(getRequestRetention())
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<JsonObject>() {
                    @Override
                    public void accept(JsonObject jsonObject) {
                        BasicData basicData = BasicDataManager.getInstance().getBasicData();
                        long currentTime = System.currentTimeMillis() + basicData.getTimeDiff();
                        basicData.setLastRetentionTime(currentTime);
                        BasicDataManager.getInstance().saveBasicData(basicData);
                        WiseLog.d("send retention success");
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) {
                        WiseLog.e(throwable);
                    }
                }, new Action() {
                    @Override
                    public void run() {
                        WiseLog.d("send retention complete");
                    }
                });
    }

    private RequestRetention getRequestRetention() {

        try {

            BasicData basicData = BasicDataManager.getInstance().getBasicData();
            Session session = SessionDataManager.getInstance().getSession();
            RequestRetention requestRetention = new RequestRetention();
            requestRetention.setServiceNumber(String.valueOf(session.getServiceNumber()));
            requestRetention.setInstallChannel(session.getAdInstallSource());
            requestRetention.setInstallCampaign(session.getAdInstallCampaign());
            requestRetention.setInstallTye(session.getAdInstallType());
            requestRetention.setInstallKeyword(session.getAdInstallKeyword());
            requestRetention.setInstallTime(basicData.getFirstAppOpenTime());
            requestRetention.setLastRetentionTime(basicData.getLastRetentionTime());
            requestRetention.setAdId(session.getAdId());
            WiseLog.d("request retention : " + new Gson().toJson(requestRetention));
            return requestRetention;

        } catch (Exception e) {
            WiseLog.e(e);
        }

        return new RequestRetention();

    }

}
