package com.sdk.wisetracker.base.tracker.data.manager;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.ServiceConnection;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;

import com.onestore.android.external.installreferrer.IGetInstallReferrerService;
import com.sdk.wisetracker.base.tracker.common.log.WiseLog;
import com.sdk.wisetracker.base.tracker.data.init.InitInstallReferrer;
import com.sdk.wisetracker.base.tracker.data.init.InitializeManager;
import com.onestore.android.external.installreferrer.IGetInstallReferrerService;

import java.net.URLDecoder;
import java.sql.Ref;
import java.util.Iterator;
import java.util.List;

public class OneStoreInstallReferrerDataManager {

    private static OneStoreInstallReferrerDataManager instance = null;

    public static OneStoreInstallReferrerDataManager getInstance() {
        if (instance == null) {
            instance = new OneStoreInstallReferrerDataManager();
        }
        return instance;
    }

    /**
     * OneStore RPC Service binder.
     */
    private IGetInstallReferrerService mService;
    public final static String[] SERVICE_PACKAGE_NAMES = {"com.skt.skaf.A000Z00040", "com.kt.olleh.storefront", "com.kt.olleh.istore", "com.lguplus.appstore", "android.lgt.appstore"};
    public final static String SERVICE_NAME = "com.onestore.android.external.installreferrer.GetInstallReferrerService";
    public final static String SERVICE_ACTION_NAME = "com.onestore.android.external.BIND_GET_INSTALL_REFERRER_SERVICE";
    /**
     * Result codes
     */
    private static final int SUCCESS = 0;
    private static final int DB_ACCESS_FAIL = -1;
    private static final int NOT_FOUND_REFERRER = -2;
    private static final int NOT_ALLOWED_PACKAGE = -3;
    private static final int INVALID_PACKAGE_NAME = -4;
    private static final int ONESTORE_MIN_APP_VER = 60700;

    /**
     * Referrer Details
     */
    public ReferrerDetails mReferrerDetails;
    public ReferrerDetails getEmptyReferrer(){
        return new ReferrerDetails();
    }

    public boolean isOneStoreCompatible(String servicePackageName) {
        PackageManager pm = InitializeManager.applicationContext.getPackageManager();
        try {
            PackageInfo info = pm.getPackageInfo(servicePackageName, PackageManager.GET_META_DATA);
            int versionCode = info.versionCode;
            return versionCode >= ONESTORE_MIN_APP_VER;
        } catch (PackageManager.NameNotFoundException e) {
            return false;
        }
    }

    /**
     * Service connection
     */
    private ServiceConnection mServiceConnection = new ServiceConnection() {
        @Override
        public void onServiceConnected(ComponentName name, IBinder iBinder) {
            mService = IGetInstallReferrerService.Stub.asInterface(iBinder);
            Bundle bundle = null;
            try {
                bundle = mService.getInstallReferrer(InitializeManager.applicationContext.getPackageName());
                WiseLog.d(" oneStore bundle " + bundle);
             } catch (RemoteException e) {
                WiseLog.d(" OneStore Service onServiceConnected RemoteException : " + e.toString() );
            }
            if (bundle != null) {
                try {
                    mReferrerDetails = new ReferrerDetails(bundle);
                } catch (DbAccessFailException e) {
                    //todo handling DbAccessFailException
                } catch (NotFoundReferrerException e) {
                    //todo handling NotFoundReferrerException
                } catch (NotAllowedPackageException e) {
                    //todo handling NotAllowedException
                } catch (InvalidPackageNameException e) {
                    //todo handling InvalidPackageNameException
                } catch (UnknownException e) {
                    //todo handling UnknownException
                }finally {
                    if( mReferrerDetails == null ){
                        mReferrerDetails = new ReferrerDetails();
                    }
                }
            }else{
                mReferrerDetails = new ReferrerDetails();
            }
            System.out.println(" oneStoreObserveFlag false setting ");
            InitInstallReferrer.getInstance().oneStoreObserveFlag = false;
        }
        @Override
        public void onServiceDisconnected(ComponentName name) {
            mService = null;
            InitInstallReferrer.getInstance().oneStoreObserveFlag = false;
        }
    };

    public ServiceConnection getmServiceConnection(){
        return this.mServiceConnection;
    }
    public class ReferrerDetails {
        public String installReferrer;
        public String getInstallReferrer(){
            String result = this.installReferrer;
            try{
                /** OneStore에서 수신되는 Instal Referrer는 자동으로 decoding 처리를 해주지 않음으로, 직접 디코딩을 해서 사용해야함. **/
                if( result != null ){
                    if( result.indexOf("%26") >= 0 && result.indexOf("%3D") >= 0 ){
                        result = URLDecoder.decode(result,"utf-8");
                    }
                }
            }catch(Exception e){
                e.printStackTrace();
            }
            return result;
        }
        public final String onestorePid;
        public final long referrerClickTimeStampSeconds;
        public long getReferrerClickTimestampSeconds(){
            return this.referrerClickTimeStampSeconds;
        }
        public final long installBeginTimeStampSeconds;
        public long getInstallBeginTimestampSeconds(){
            return this.installBeginTimeStampSeconds;
        }

        public ReferrerDetails(){
            this.installReferrer = "";
            this.onestorePid = "";
            this.referrerClickTimeStampSeconds = 0;
            this.installBeginTimeStampSeconds = 0;
        }
        public ReferrerDetails(Bundle bundle) throws DbAccessFailException, NotFoundReferrerException, NotAllowedPackageException, InvalidPackageNameException, UnknownException {
            int resultCode = bundle.getInt("result_code", -10);
            if (resultCode == SUCCESS) {
                this.installReferrer = bundle.getString("install_referrer", "");
                /** 사용된 커스텀 스키마가 http://onestore.re 인 경우에는 인코딩이 2회되서 SDK에 수신되는 경우가 있음. 때문에 한번더 인코딩 여부 체크 **/
                if( this.installReferrer != null ){
                    if( this.installReferrer.indexOf("%2526") >= 0 || this.installReferrer.indexOf("%253D") >= 0 ){
                        try{
                            this.installReferrer = URLDecoder.decode(this.installReferrer,"utf-8");
                        }catch(Exception e){}
                    }
                }
                this.onestorePid = bundle.getString("onestore_pid", "");
                this.referrerClickTimeStampSeconds = bundle.getLong("referrer_click_timestamp_seconds", -1);
                this.installBeginTimeStampSeconds = bundle.getLong("install_begin_timestamp_seconds", -1);
            } else if (resultCode == DB_ACCESS_FAIL) {
                throw new DbAccessFailException(bundle.getString("description", "DB_ACCESS_FAIL"));
            } else if (resultCode == NOT_FOUND_REFERRER) {
                throw new NotFoundReferrerException(bundle.getString("description", "NOT_FOUND_REFERRER"));
            } else if (resultCode == NOT_ALLOWED_PACKAGE) {
                throw new NotAllowedPackageException(bundle.getString("description", "NOT_ALLOWED_PACKAGE"));
            } else if (resultCode == INVALID_PACKAGE_NAME) {
                throw new InvalidPackageNameException(bundle.getString("description", "INVALID_PACKAGE_NAME"));
            } else {
                throw new UnknownException("resultCode is " + resultCode);
            }
        }
    }
    private class DbAccessFailException extends Exception {
        public DbAccessFailException(String errorMsg) {
            super(errorMsg);
        }
    }
    private class NotFoundReferrerException extends Exception {
        public NotFoundReferrerException(String errorMsg) {
            super(errorMsg);
        }
    }
    private class NotAllowedPackageException extends Exception {
        public NotAllowedPackageException(String errorMsg) {
            super(errorMsg);
        }
    }
    private class InvalidPackageNameException extends Exception {
        public InvalidPackageNameException(String errorMsg) {
            super(errorMsg);
        }
    }
    private class UnknownException extends Exception {
        public UnknownException(String errorMsg) {
            super(errorMsg);
        }
    }
}
