package com.sdk.wisetracker.base.tracker.data.init;

import android.annotation.SuppressLint;
import android.content.Context;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.sdk.wisetracker.base.tracker.common.WisetrackerDatabase;
import com.sdk.wisetracker.base.tracker.common.log.WiseLog;
import com.sdk.wisetracker.base.tracker.data.manager.BasicDataManager;
import com.sdk.wisetracker.base.tracker.data.model.Token;

import io.reactivex.Observable;
import io.reactivex.android.schedulers.AndroidSchedulers;
import io.reactivex.functions.Consumer;
import io.reactivex.functions.Function3;
import io.reactivex.schedulers.Schedulers;
import io.reactivex.subjects.PublishSubject;

/**
 * SDK 초기화 수행 매니져, 초기화 완료시 SDK 정상 동작
 */
public class InitializeManager {

    // 초기화 완료 여부 flag
    public static boolean initComplete = false;
    // context 정보
    public static Context applicationContext = null;
    // 하위 모듈 사용 subject
    public static PublishSubject<Boolean> initSubject = PublishSubject.create();

    // Facebook SDK를 사용중이여서, facebook 링크를 기다려야 하는 경우.
    public static boolean waitingForFbDeferred = false; // default false
    public static String useFacebookAppLinkData = "N";

    public static void init(Context context) {
        try {
            baseInit(context);
            asyncInit(context);
        } catch (Exception e) {
            WiseLog.e(e);
        }
    }

    /**
     * SDK 동기 초기화 처리
     * @param context
     */
    private static void baseInit(Context context) {
        // context
        applicationContext = context;
        // log
        WiseLog.d("init call");
        WiseLog.d("init context info -> " + context);
        // db
        WisetrackerDatabase.init(context);
        // check app open status
        InitAppOpen.getInstance().checkAppOpen();
        // app key setting
        InitAppKey.getInstance().setAppKey(context);
        // environment setting
        InitEnvironment.getInstance().setEnvironment(context);
    }

    /**
     * SDK 비동기 초기화 처리
     * ADID, NTP-TIME, TOKEN 정보 획득 후 앱 최초 실행시 인스톨 레퍼러 정보 호출, 최초 실행 아닐 경우 초기화 완료
     * @param context
     */
    @SuppressLint("CheckResult")
    private static void asyncInit(final Context context) {

        // load advertising id
        Observable<AdvertisingIdClient.Info> advertisingObservable = InitAdvertisingId.getInstance().getAdvertisingObservable(context);
        // ntp time setting
        Observable<Long> ntpTimeObservable = InitNtpTime.getInstance().getNtpTimeObservable();
        // token setting
        Observable<Token> tokenObservable = InitToken.getInstance().getAccessTokenObservable();

        Observable.zip(advertisingObservable, ntpTimeObservable, tokenObservable, new Function3<AdvertisingIdClient.Info, Long, Token, Boolean>() {
            @Override
            public Boolean apply(AdvertisingIdClient.Info info, Long diff, Token token) {
                InitAdvertisingId.getInstance().setAdvertisingId(info);
                InitNtpTime.getInstance().setNtpTime(diff);
                return InitToken.getInstance().isValidToken(token);
            }
        })
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(new Consumer<Boolean>() {
                    @Override
                    public void accept(Boolean isValid) {
                        // 토큰 유효성 체크
                        if (!isValid) {
                            WiseLog.d("token is invalid");
                            return;
                        }
                        boolean isFirstAppOpen = BasicDataManager.getInstance().getBasicData().getIsFirstAppOpen().equals("Y");
                        WiseLog.d("!!!! isFirstAppOpen : " + isFirstAppOpen  );
                        WiseLog.d("!!!! BasicDataManager.getInstance().getBasicData().getIsFirstAppOpen() : " + BasicDataManager.getInstance().getBasicData().getIsFirstAppOpen());
                        if (!isFirstAppOpen) {
                            // 앱 최초 실행 X, 초기화 완료
                            initComplete = true;
                            initSubject.onNext(true);
                            return;
                        }
                        // 앱 최초 실행시 IR 정보 확인
                        InitInstallReferrer.getInstance().loadInstallReferrer(context, referrerInitCallBack);
                    }
                }, new Consumer<Throwable>() {
                    @Override
                    public void accept(Throwable throwable) {
                        WiseLog.e(throwable);
                    }
                });

    }

    private static ReferrerInitCallBack referrerInitCallBack = new ReferrerInitCallBack() {
        @Override
        public void onCallback(boolean isComplete) {
            // IR 정보 확인 후 콜백, 초기화 완료
            initComplete = true;
            initSubject.onNext(true);
        }
    };

}
