package com.scansolutions.mrzscannerlib;

import android.graphics.Bitmap;
import android.os.Parcel;
import android.os.Parcelable;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;

public class MRZResultModel implements Parcelable {

    static {
        System.loadLibrary("mrz-native-lib");
    }

    public MRZResultModel() {
    }

    public MRZResultModel(Parcel in) {
        rawResult = in.readString();
        documentTypeRaw = in.readString();
        issuingCountry = in.readString();
        surnames = in.createStringArray();
        givenNames = in.createStringArray();
        documentNumber = in.readString();
        documentNumberWithCheckDigit = in.readString();
        nationality = in.readString();
        dobRaw = in.readString();
        dobWithCheckDigit = in.readString();
        sex = in.readString();
        estIssuingDateRaw = in.readString();
        expirationDateRaw = in.readString();
        expirationDateWithCheckDigit = in.readString();
        masterCheckDigit = in.readString();
        areCheckDigitsValid = in.readInt() > 0;
        optionals = in.createStringArray();
        dateScanned = in.readLong();
        portrait = in.readParcelable(Bitmap.class.getClassLoader());
        signature = in.readParcelable(Bitmap.class.getClassLoader());
        fullImage = in.readParcelable(Bitmap.class.getClassLoader());
        idBack = in.readParcelable(Bitmap.class.getClassLoader());
        idFront = in.readParcelable(Bitmap.class.getClassLoader());

        parseRawToReadable();
    }

    public native String parseReadableDocumentType(String text);

    public native String calculateEstIssuingDate(String documentTypeRaw, String issuingCountry, String dobRaw, String expirationDateRaw);

    public native String parseDate(String text);

    public String rawResult = "N/A";
    public String documentTypeRaw = "N/A";
    public String documentTypeReadable = "N/A";
    public String issuingCountry = "N/A";
    public String[] surnames;
    public String[] givenNames;
    public String documentNumber = "N/A";
    public String documentNumberWithCheckDigit = "N/A";
    public String nationality = "N/A";
    public String dobRaw = "N/A";
    public String dobWithCheckDigit = "N/A";
    public String dobReadable = "N/A";
    public String sex = "N/A";
    public String expirationDateRaw = "N/A";
    public String expirationDateWithCheckDigit = "N/A";
    public String expirationDateReadable = "N/A";
    public String estIssuingDateRaw = "N/A";
    public String estIssuingDateReadable = "N/A";
    public String masterCheckDigit = "N/A";
    public boolean areCheckDigitsValid;
    public String[] optionals;
    public long dateScanned;
    public Bitmap portrait;
    public Bitmap signature;
    public Bitmap fullImage;
    public Bitmap idBack;
    public Bitmap idFront;

    public MRZResultModel(String documentTypeRaw,
                          String issuingCountry,
                          String[] surnames,
                          String[] givenNames,
                          String documentNumber,
                          String documentNumberWithCheckDigit,
                          String nationality,
                          String dobRaw,
                          String dobWithCheckDigit,
                          String sex,
                          String estIssuingDateRaw,
                          String expirationDateRaw,
                          String expirationDateWithCheckDigit,
                          String masterCheckDigit,
                          String[] optionals,
                          long dateScanned,
                          Bitmap portrait,
                          Bitmap signature,
                          Bitmap fullImage,
                          Bitmap idBack,
                          Bitmap idFront,
                          boolean areCheckDigitsValid) {
        this.documentTypeRaw = documentTypeRaw;
        this.issuingCountry = issuingCountry;
        this.surnames = surnames;
        this.givenNames = givenNames;
        this.documentNumber = documentNumber;
        this.documentNumberWithCheckDigit = documentNumberWithCheckDigit;
        this.nationality = nationality;
        this.dobRaw = dobRaw;
        this.dobWithCheckDigit = dobWithCheckDigit;
        this.sex = sex;
        this.expirationDateRaw = expirationDateRaw;
        this.estIssuingDateRaw = estIssuingDateRaw;
        this.expirationDateWithCheckDigit = expirationDateWithCheckDigit;
        this.masterCheckDigit = masterCheckDigit;
        this.optionals = optionals;
        this.dateScanned = dateScanned;
        this.portrait = portrait;
        this.signature = signature;
        this.fullImage = fullImage;
        this.idBack = idBack;
        this.idFront = idFront;
        this.areCheckDigitsValid = areCheckDigitsValid;

        parseRawToReadable();
    }

    MRZResultModel(JSONObject jsonResult, long dateScanned) throws JSONException {
        this.dateScanned = dateScanned;

        rawResult = jsonResult.getString("raw_result");
        documentTypeRaw = jsonResult.getString("document_type_raw");
        documentTypeReadable = jsonResult.getString("document_type_readable");
        issuingCountry = jsonResult.getString("issuing_country");
        documentNumber = jsonResult.getString("document_number");
        documentNumberWithCheckDigit = jsonResult.getString("document_number_with_check_digit");
        nationality = jsonResult.getString("nationality");
        dobRaw = jsonResult.getString("dob_raw");
        dobReadable = jsonResult.getString("dob_readable");
        dobWithCheckDigit = jsonResult.getString("dob_with_check_digit");
        sex = jsonResult.getString("sex");
        expirationDateRaw = jsonResult.getString("expiration_date_raw");
        expirationDateWithCheckDigit = jsonResult.getString("expiration_date_with_check_digit");
        expirationDateReadable = jsonResult.getString("expiration_date_readable");
        masterCheckDigit = jsonResult.getString("master_check_digit");
        areCheckDigitsValid = jsonResult.getBoolean("are_check_digits_valid");

        JSONArray givenNamesArr = jsonResult.getJSONArray("given_names");
        if (givenNamesArr.length() > 0) {
            givenNames = new String[givenNamesArr.length()];

            for (int i = 0; i < givenNamesArr.length(); i++) {
                givenNames[i] = givenNamesArr.getString(i);
            }
        }

        JSONArray surnamesArr = jsonResult.getJSONArray("surnames");
        if (surnamesArr.length() > 0) {
            surnames = new String[surnamesArr.length()];

            for (int i = 0; i < surnamesArr.length(); i++) {
                surnames[i] = surnamesArr.getString(i);
            }
        }

        JSONArray optionalsArr = jsonResult.getJSONArray("optionals");
        if (optionalsArr.length() > 0) {
            optionals = new String[optionalsArr.length()];

            for (int i = 0; i < optionalsArr.length(); i++) {
                optionals[i] = optionalsArr.getString(i);
            }
        }

        estIssuingDateRaw = calculateEstIssuingDate(documentTypeRaw, issuingCountry, dobRaw, expirationDateRaw);
        parseRawToReadable();
    }

    public String getFullName() {
        return givenNamesReadable() + " " + surnamesReadable();
    }

    public void parseRawToReadable() {
        documentTypeReadable = parseReadableDocumentType(documentTypeRaw);
        dobReadable = parseDOBToReadable(dobRaw);
        expirationDateReadable = parseExpirationDateToReadable(expirationDateRaw);
        estIssuingDateReadable = parseExpirationDateToReadable(estIssuingDateRaw);
    }

    private String parseDOBToReadable(String dateRaw) {
        if (dateRaw == null || dateRaw.length() != 6) {
            return "Unknown";
        }

        try {
            Calendar calendar = Calendar.getInstance();
            int currentYear = calendar.get(Calendar.YEAR);
            currentYear %= 100;
            int dobYear = Integer.parseInt(dateRaw.substring(0, 2));

            dobYear = dobYear > currentYear ? dobYear + 1900 : dobYear + 2000;

            return parseDateToReadable(dateRaw, dobYear);
        } catch (NumberFormatException ignored) {
        }

        return "Unknown";
    }

    private String parseExpirationDateToReadable(String dateRaw) {
        if (dateRaw == null || dateRaw.length() != 6) {
            return "Unknown";
        }

        try {
            int expYear = Integer.parseInt(dateRaw.substring(0, 2));
            expYear = expYear < 70 ? expYear + 2000 : expYear + 1900;

            return parseDateToReadable(dateRaw, expYear);
        } catch (NumberFormatException ignored) {
        }

        return "Unknown";
    }

    private String parseDateToReadable(String dateRaw, int year) throws NumberFormatException {
        Calendar calendar = Calendar.getInstance();

        calendar.set(year, Integer.parseInt(dateRaw.substring(2, 4)) - 1, Integer.parseInt(dateRaw.substring(4, 6)));
        SimpleDateFormat sdf = new SimpleDateFormat(MRZScanner.getDateFormat(), Locale.getDefault());

        return sdf.format(calendar.getTime());
    }

    public String givenNamesReadable() {
        if (givenNames == null) return "";

        StringBuilder fullName = new StringBuilder();

        for (String s : givenNames) {
            fullName.append(s).append(" ");
        }

        return fullName.toString().trim();
    }

    public String surnamesReadable() {
        if (surnames == null) return "";

        StringBuilder fullName = new StringBuilder();

        for (String s : surnames) {
            fullName.append(s).append(" ");
        }

        return fullName.toString().trim();
    }

    public String optionalsReadable() {
        StringBuilder optionalsReadable = new StringBuilder();

        if (optionals != null) {
            for (String s : optionals) {
                optionalsReadable.append(s).append(" ");
            }
        }

        return optionalsReadable.toString().trim();
    }

    public String dob_check_digit() {
        if (isStringValid(dobWithCheckDigit))
            return dobWithCheckDigit.substring(dobWithCheckDigit.length() - 1);

        return null;
    }

    public String expiration_date_check_digit() {
        if (isStringValid(expirationDateWithCheckDigit))
            return expirationDateWithCheckDigit.substring(expirationDateWithCheckDigit.length() - 1);

        return null;
    }

    public String document_number_check_digit() {
        if (isStringValid(documentNumberWithCheckDigit))
            return documentNumberWithCheckDigit.substring(documentNumberWithCheckDigit.length() - 1);

        return null;
    }

    public JSONObject toJSON() throws JSONException {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("document_type_raw", documentTypeRaw);
        jsonObject.put("document_type_readable", documentTypeReadable);
        jsonObject.put("issuing_country", issuingCountry);
        jsonObject.put("surname", surnamesReadable());
        jsonObject.put("document_number", documentNumber);
        jsonObject.put("document_number_with_check_digit", documentNumberWithCheckDigit);
        jsonObject.put("document_number_check_digit", document_number_check_digit());
        jsonObject.put("nationality", nationality);
        jsonObject.put("dob_raw", dobRaw);
        jsonObject.put("dob_with_check_digit", dobWithCheckDigit);
        jsonObject.put("dob_check_digit", dob_check_digit());
        jsonObject.put("dob_readable", dobReadable);
        jsonObject.put("sex", sex);
        jsonObject.put("est_issuing_date_raw", estIssuingDateRaw);
        jsonObject.put("est_issuing_date_readable", estIssuingDateReadable);
        jsonObject.put("expiration_date_raw", expirationDateRaw);
        jsonObject.put("expiration_date_with_check_digit", expirationDateWithCheckDigit);
        jsonObject.put("expiration_date_check_digit", expiration_date_check_digit());
        jsonObject.put("expiration_date_readable", expirationDateReadable);
        jsonObject.put("master_check_digit", masterCheckDigit);
        jsonObject.put("given_names_readable", givenNamesReadable());
        jsonObject.put("optionals", optionalsReadable());

        return jsonObject;
    }

    public boolean isExpired() {
        if (expirationDateRaw.length() == 6 && !expirationDateRaw.contains("*")) {
            int expYear = Integer.parseInt(expirationDateRaw.substring(0, 2));
            expYear = expYear < 70 ? expYear + 2000 : expYear + 1900;
            Calendar calendar = Calendar.getInstance();

            calendar.set(expYear, Integer.parseInt(expirationDateRaw.substring(2, 4)) - 1, Integer.parseInt(expirationDateRaw.substring(4, 6)));
            return calendar.before(Calendar.getInstance());
        }

        return false;
    }

    public String toCSV() {
        return documentTypeRaw + ","
                + documentTypeReadable + ","
                + issuingCountry + ","
                + surnamesReadable() + ","
                + givenNamesReadable() + ","
                + documentNumber + ","
                + documentNumberWithCheckDigit + ","
                + document_number_check_digit() + ","
                + nationality + ","
                + dobRaw + ","
                + dobWithCheckDigit + ","
                + dob_check_digit() + ","
                + dobReadable + ","
                + sex + ","
                + estIssuingDateRaw + ","
                + estIssuingDateReadable + ","
                + expirationDateRaw + ","
                + expirationDateWithCheckDigit + ","
                + expiration_date_check_digit() + ","
                + expirationDateReadable + ","
                + masterCheckDigit + ","
                + optionalsReadable() + ","
                + dateScanned;
    }

    public String toReadableString() {
        return "Document number:  " + documentNumber +
                "\nFull name:  " + getFullName() +
                "\nNationality:  " + nationality +
                "\nIssuing country:  " + issuingCountry +
                "\nSex:  " + sex +
                "\nDate of birth:  " + dobReadable +
                "\nDate of birth (raw):  " + dobRaw +
                "\nExpiration date:  " + expirationDateReadable +
                "\nExpiration date (raw):  " + expirationDateRaw +
                "\nEstimated issuing date:  " + estIssuingDateReadable +
                "\nEstimated issuing date (raw):  " + estIssuingDateRaw +
                "\nOptional value:  " + optionalsReadable();
    }

    public boolean isId() {
        return documentTypeRaw.startsWith("I") || documentTypeRaw.startsWith("A") || documentTypeRaw.startsWith("C");
    }

    public boolean isPassport() {
        return documentTypeRaw.startsWith("P");
    }

    public boolean isVisa() {
        return documentTypeRaw.startsWith("V");
    }

    public boolean isResidencePermit() {
        return documentTypeRaw.startsWith("A");
    }

    private boolean isStringValid(String value) {
        return value != null && !value.isEmpty() && !value.equals("N/A");
    }

    @Override
    public int describeContents() {
        return 0;
    }

    @Override
    public void writeToParcel(Parcel dest, int flags) {
        dest.writeString(rawResult);
        dest.writeString(documentTypeRaw);
        dest.writeString(issuingCountry);
        dest.writeStringArray(surnames);
        dest.writeStringArray(givenNames);
        dest.writeString(documentNumber);
        dest.writeString(documentNumberWithCheckDigit);
        dest.writeString(nationality);
        dest.writeString(dobRaw);
        dest.writeString(dobWithCheckDigit);
        dest.writeString(sex);
        dest.writeString(estIssuingDateRaw);
        dest.writeString(expirationDateRaw);
        dest.writeString(expirationDateWithCheckDigit);
        dest.writeString(masterCheckDigit);
        dest.writeInt(areCheckDigitsValid ? 1 : 0);
        dest.writeStringArray(optionals);
        dest.writeLong(dateScanned);
        dest.writeValue(portrait);
        dest.writeValue(signature);
        dest.writeValue(fullImage);
        dest.writeValue(idBack);
        dest.writeValue(idFront);
    }

    public static final Parcelable.Creator CREATOR = new Parcelable.Creator() {
        public MRZResultModel createFromParcel(Parcel in) {
            return new MRZResultModel(in);
        }

        public MRZResultModel[] newArray(int size) {
            return new MRZResultModel[size];
        }
    };

}
