package com.scansolutions.mrzscannerlib;

import android.content.Context;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Point;
import android.graphics.PorterDuff;
import android.graphics.PorterDuffColorFilter;
import android.graphics.Rect;
import android.os.Handler;
import android.os.Looper;
import android.util.AttributeSet;
import android.view.View;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.AlphaAnimation;
import android.view.animation.Animation;
import android.widget.RelativeLayout;

class MRZOverlay extends RelativeLayout {

    private static final String TAG = "MRZOverlay";

    enum OverlayColor {
        OVERLAY_COLOR_GREEN,
        OVERLAY_COLOR_YELLOW
    }

    private static MRZPercentageRect finderRect = new MRZPercentageRect(0, 0, 100, 100);

    public static final MRZPercentageRect TOP_RECT = new MRZPercentageRect(5, 10, 90, 35);
    public static final MRZPercentageRect BOTTOM_RECT = new MRZPercentageRect(5, 55, 90, 35);
    public static final MRZPercentageRect LEFT_RECT = new MRZPercentageRect(3, 18, 46, 64);
    public static final MRZPercentageRect RIGHT_RECT = new MRZPercentageRect(51, 18, 46, 64);
    private static final String BACK_SIDE_TEXT = "First, scan ID's back";
    private static final String FRONT_SIDE_TEXT = "... then, scan ID's front";

    private static final float TEXT_MARGIN = 0.5f;
    private static final float TEXT_SIZE = 2.3f;

    private final Context context;
    private final Bitmap icon;
    private final Bitmap icon90;
    private final Bitmap icon180;
    private final Bitmap icon270;

    private boolean layoutDrawn = false;
    private MRZLocationOverlay mrzLocationOverlay;
    Animation fadeOut;

    public MRZOverlay(Context context, AttributeSet attrs) {
        this(context, attrs, 10, 20, 80, 60);
    }

    public MRZOverlay(Context context, AttributeSet attrs, int x, int y, int width, int height) {
        super(context, attrs);
        this.context = context;
        finderRect = new MRZPercentageRect(x, y, width, height);

        icon = ImageUtils.getBitmap(context, R.drawable.amrz_ic_scan_border);

        Matrix matrix = new Matrix();
        matrix.postRotate(90);
        icon90 = Bitmap.createBitmap(icon, 0, 0, icon.getWidth(), icon.getHeight(), matrix, true);
        matrix.postRotate(180);
        icon180 = Bitmap.createBitmap(icon, 0, 0, icon.getWidth(), icon.getHeight(), matrix, true);
        matrix.postRotate(270);
        icon270 = Bitmap.createBitmap(icon, 0, 0, icon.getWidth(), icon.getHeight(), matrix, true);

        fadeOut = new AlphaAnimation(1, 0);
        fadeOut.setInterpolator(new AccelerateInterpolator());
        fadeOut.setStartOffset(0);
        fadeOut.setDuration(1500);
        fadeOut.setFillAfter(true);
    }

    public void setRect(float x, float y, float width, float height) {
        setRect(x, y, width, height, 1, 1);
    }

    public void setRect(float x, float y, float width, float height, float widthRatio, float heightRatio) {
        finderRect = MRZUtils.calculateScaledRect(x, y, width, height, widthRatio, heightRatio, false);

        if (layoutDrawn) {
            invalidate();
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        if (MRZCore.overlayState == MRZOverlayState.NORMAL) {
            drawRect(canvas, finderRect, Color.WHITE);
        } else {
            boolean isPortrait = getResources().getConfiguration().orientation == Configuration.ORIENTATION_PORTRAIT;
            int topColor = MRZCore.overlayState == MRZOverlayState.SCANNING_BACK ? Color.WHITE : Color.parseColor("#8000FF00");
            int bottomColor = MRZCore.overlayState == MRZOverlayState.SCANNING_BACK ? Color.parseColor("#80FFFFFF") : Color.WHITE;

            if (isPortrait) {
                drawRect(canvas, TOP_RECT, topColor);
                drawRect(canvas, BOTTOM_RECT, bottomColor);
                drawText(canvas, BACK_SIDE_TEXT, TOP_RECT.getTopCenter(), true);
                drawText(canvas, FRONT_SIDE_TEXT, BOTTOM_RECT.getTopCenter(), true);
            } else {
                drawRect(canvas, LEFT_RECT, topColor);
                drawRect(canvas, RIGHT_RECT, bottomColor);
                drawText(canvas, BACK_SIDE_TEXT, LEFT_RECT.getTopCenter(), false);
                drawText(canvas, FRONT_SIDE_TEXT, RIGHT_RECT.getTopCenter(), false);
            }
        }

        layoutDrawn = true;
    }

    private void drawText(Canvas canvas, String text, Point topCenter, boolean isPortrait) {
        Paint paint = new Paint();
        paint.setColor(Color.WHITE);

        int margin = (int) (TEXT_MARGIN / 100.0f * (isPortrait ? getHeight() : getWidth()));
        int textSize = (int) (TEXT_SIZE / 100.0f * (isPortrait ? getHeight() : getWidth()));
        paint.setTextSize(textSize);

        Rect r = new Rect();
        paint.getTextBounds(text, 0, text.length(), r);
        int yPos = (int) (topCenter.y / 100.0f * getHeight() - margin - textSize);
        int xPos = (int) (topCenter.x / 100.0f * getWidth() - r.width() / 2);

        canvas.drawText(text, xPos, yPos, paint);
    }

    private void drawRect(Canvas canvas, MRZPercentageRect percentageRect, int color) {
        if (percentageRect.x + percentageRect.y + percentageRect.width + percentageRect.height > 1 && getHeight() > 0) {
            float xP = percentageRect.x / 100.0f * getWidth();
            float yP = percentageRect.y / 100.0f * getHeight();
            float widthP = percentageRect.width / 100.0f * getWidth();
            float heightP = percentageRect.height / 100.0f * getHeight();

            Paint paint = new Paint();
            paint.setColorFilter(new PorterDuffColorFilter(color, PorterDuff.Mode.SRC_IN));

            canvas.drawBitmap(icon, xP, yP + heightP - icon.getHeight(), paint);
            canvas.drawBitmap(icon90, xP, yP, paint);
            canvas.drawBitmap(icon180, xP + widthP - icon.getWidth(), yP + heightP - icon.getHeight(), paint);
            canvas.drawBitmap(icon270, xP + widthP - icon.getWidth(), yP, paint);
        }
    }

    public void drawLocationOverlay(final int[] points, final int imageWidth, final int imageHeight, final OverlayColor overlayColor) {
        if (points.length != 8 ||
                points[0] == 0 ||
                points[1] == 0 ||
                points[2] == 0 ||
                points[3] == 0 ||
                points[4] == 0 ||
                points[5] == 0 ||
                points[6] == 0 ||
                points[7] == 0)
            return;

        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                if (mrzLocationOverlay == null) {
                    mrzLocationOverlay = new MRZLocationOverlay(context);
                    addView(mrzLocationOverlay);
                }

                mrzLocationOverlay.setLocationPoints(points, imageWidth, imageHeight, overlayColor);
                mrzLocationOverlay.clearAnimation();
                mrzLocationOverlay.startAnimation(fadeOut);
            }
        });
    }

    private static class MRZLocationOverlay extends View {
        Point[] locationPoints;
        private float imageWidth;
        private float imageHeight;
        private final Paint paint37;
        private final Paint paint89;

        public MRZLocationOverlay(Context context) {
            super(context);

            paint37 = new Paint();
            paint37.setStyle(Paint.Style.FILL);

            paint89 = new Paint();
            paint89.setStrokeWidth(3);
        }

        @Override
        protected void onDraw(Canvas canvas) {
            super.onDraw(canvas);

            float scaleX = getWidth() / imageWidth;
            float scaleY = getHeight() / imageHeight;

            if (locationPoints != null && locationPoints.length == 4) {
                Path path = new Path();
                path.moveTo(locationPoints[0].x * scaleX, locationPoints[0].y * scaleY);

                for (int i = 1; i < locationPoints.length; i++) {
                    path.lineTo(locationPoints[i].x * scaleX, locationPoints[i].y * scaleY);
                }

                path.lineTo(locationPoints[0].x * scaleX, locationPoints[0].y * scaleY);
                path.close();

                canvas.drawPath(path, paint37);
                paint89.setStyle(Paint.Style.STROKE);
                canvas.drawPath(path, paint89);

                paint89.setStyle(Paint.Style.FILL);
                for (Point p : locationPoints) {
                    canvas.drawCircle(p.x * scaleX, p.y * scaleY, 5, paint89);
                }
            }
        }

        public void setLocationPoints(int[] points, int imageWidth, int imageHeight, OverlayColor overlayColor) {
            if (overlayColor == OverlayColor.OVERLAY_COLOR_GREEN) {
                paint37.setColor(0x614CD863);
                paint89.setColor(0xE54CD863);
            } else {
                paint37.setColor(0x61d8be4c);
                paint89.setColor(0xE5d8b34c);
            }

            if (applyLocationPoints(points, imageWidth, imageHeight))
                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        invalidate();
                    }
                });
        }

        private boolean applyLocationPoints(int[] points, int imageWidth, int imageHeight) {
            boolean isLengthAcceptable = points.length == 8;
            if (isLengthAcceptable) {
                locationPoints = new Point[4];
                locationPoints[0] = new Point(points[0], points[1]);
                locationPoints[1] = new Point(points[2], points[3]);
                locationPoints[2] = new Point(points[4], points[5]);
                locationPoints[3] = new Point(points[6], points[7]);
                this.imageWidth = imageWidth;
                this.imageHeight = imageHeight;
            }

            return isLengthAcceptable;
        }
    }
}
