package com.scansolutions.mrzscannerlib;

import android.content.Context;
import android.hardware.Camera;
import android.os.Handler;
import android.util.Log;
import android.view.Surface;
import android.view.SurfaceHolder;

import com.scansolutions.mrzscannerlib.open.OpenCamera;
import com.scansolutions.mrzscannerlib.open.OpenCameraInterface;

import java.io.IOException;
import java.util.List;

final class CameraManager {

    private static final String TAG = CameraManager.class.getSimpleName();

    private final CameraConfigurationManager configManager;
    private OpenCamera camera;
    private boolean initialized;
    private boolean previewing;
//    private final boolean mManualFocusEngaged = false;

    /**
     * Preview frames are delivered here, which we pass on to the registered handler. Make sure to
     * clear the handler so it will only receive one message.
     */
    private final PreviewCallback previewCallback;

    CameraManager(Context context) {
        this.configManager = new CameraConfigurationManager(context);
        previewCallback = new PreviewCallback(configManager);
    }

    /**
     * Opens the camera driver and initializes the hardware parameters.
     *
     * @param holder The surface object which the camera will draw preview frames into.
     * @throws IOException Indicates the camera driver failed to open.
     */
    void openDriver(SurfaceHolder holder) throws IOException {
        OpenCamera theCamera = camera;
        if (theCamera == null) {
            int requestedCameraId = OpenCameraInterface.NO_REQUESTED_CAMERA;
            theCamera = OpenCameraInterface.open(requestedCameraId);
            if (theCamera == null) {
                CameraLogger.addLog("Camera.open() failed to return object from driver");
                throw new IOException("Camera.open() failed to return object from driver");
            }
            camera = theCamera;
        }

        if (!initialized) {
            initialized = true;
            configManager.initFromCameraParameters(theCamera);
        }

        Camera cameraObject = theCamera.getCamera();
        Camera.Parameters parameters = cameraObject.getParameters();

        String parametersFlattened = parameters == null ? null : parameters.flatten(); // Save these, temporarily
        try {
            configManager.setDesiredCameraParameters(theCamera, false);
        } catch (RuntimeException re) {
            // Driver failed
            CameraLogger.addLog("Camera rejected parameters. Setting only minimal safe-mode parameters");
            CameraLogger.addLog("Resetting to saved camera params: " + parametersFlattened);
            CameraLogger.addLog("RuntimeException: " + re.getMessage());

            Log.w(TAG, "Camera rejected parameters. Setting only minimal safe-mode parameters");
            Log.i(TAG, "Resetting to saved camera params: " + parametersFlattened);
            // Reset:
            if (parametersFlattened != null) {
                parameters = cameraObject.getParameters();
                parameters.unflatten(parametersFlattened);
                try {
                    cameraObject.setParameters(parameters);
                    configManager.setDesiredCameraParameters(theCamera, true);
                } catch (RuntimeException re2) {
                    // Well, darn. Give up
                    Log.w(TAG, "Camera rejected even safe-mode parameters! No configuration");
                    CameraLogger.addLog("Camera rejected even safe-mode parameters! No configuration");
                    CameraLogger.addLog("RuntimeException: " + re.getMessage());
                }
            }
        }
        cameraObject.setPreviewDisplay(holder);
    }

    void updateOrientation(int rotation) {
        if (camera != null) {
            Camera c = camera.getCamera();

            if (c != null) {
                int cwRotationFromNaturalToDisplay;
                switch (rotation) {
                    case Surface.ROTATION_0:
                        cwRotationFromNaturalToDisplay = 0;
                        break;
                    case Surface.ROTATION_90:
                        cwRotationFromNaturalToDisplay = 90;
                        break;
                    case Surface.ROTATION_180:
                        cwRotationFromNaturalToDisplay = 180;
                        break;
                    case Surface.ROTATION_270:
                        cwRotationFromNaturalToDisplay = 270;
                        break;
                    default:
                        // Have seen this return incorrect values like -90
                        if (rotation % 90 == 0) {
                            cwRotationFromNaturalToDisplay = (360 + rotation) % 360;
                        } else {
                            CameraLogger.addLog("Bad rotation: " + rotation);
                            throw new IllegalArgumentException("Bad rotation: " + rotation);
                        }
                }

                int cwRotationFromNaturalToCamera = camera.getOrientation();
                int cwRotationFromDisplayToCamera = (360 + cwRotationFromNaturalToCamera - cwRotationFromNaturalToDisplay) % 360;

                c.setDisplayOrientation(cwRotationFromDisplayToCamera);
            }
        }
    }

    public boolean isOpen() {
        return camera != null;
    }

    /**
     * Closes the camera driver if still in use.
     */
    void closeDriver() {
        if (camera != null) {
            camera.getCamera().release();
            camera = null;
        }
    }

    /**
     * Asks the camera hardware to begin drawing preview frames to the screen.
     */
    void startPreview() {
        if (camera != null && !previewing) {
            camera.getCamera().startPreview();
            previewing = true;
            Camera.Parameters params = camera.getCamera().getParameters();

            if (params.getSupportedFocusModes().contains(Camera.Parameters.FOCUS_MODE_CONTINUOUS_PICTURE)) {
                params.setFocusMode(Camera.Parameters.FOCUS_MODE_CONTINUOUS_PICTURE);
                camera.getCamera().setParameters(params);
            }
        }
    }

    /**
     * Tells the camera to stop drawing preview frames.
     */
    void stopPreview() {
        if (camera != null && previewing) {
            camera.getCamera().setPreviewCallbackWithBuffer(null);
            camera.getCamera().setPreviewCallback(null);
            camera.getCamera().stopPreview();
            previewCallback.setHandler(null, 0);
            previewing = false;
        }
    }

    /**
     * @param newSetting if {@code true}, light should be turned on if currently off. And vice versa.
     */
    void setTorch(boolean newSetting) {
        if (camera != null && newSetting != configManager.getTorchState(camera.getCamera())) {
            configManager.setTorch(camera.getCamera(), newSetting);
        }
    }

    /**
     * A single preview frame will be returned to the handler supplied. The data will arrive as byte[]
     * in the message.obj field, with width and height encoded as message.arg1 and message.arg2,
     * respectively.
     *
     * @param handler The handler to send the message to.
     * @param message The what field of the message to be sent.
     */
    void requestPreviewFrame(Handler handler, int message) {
        OpenCamera theCamera = camera;
        if (theCamera != null && previewing) {
            previewCallback.setHandler(handler, message);
            theCamera.getCamera().setPreviewCallback(previewCallback);
        }
    }

    boolean doesCameraMeetMinimum() {
        boolean isResSupported = false;
        boolean hasAutoFocus = false;

        if (Camera.getNumberOfCameras() > 0) {
            android.hardware.Camera.Parameters parameters = camera.getCamera().getParameters();

            List<String> supportedFocusModes = parameters.getSupportedFocusModes();
            hasAutoFocus = supportedFocusModes != null && supportedFocusModes.contains(Camera.Parameters.FOCUS_MODE_AUTO);
            List<Camera.Size> sizes = parameters.getSupportedPictureSizes();

            for (Camera.Size size : sizes)
                if (size.width * size.height > 4500000) {
                    isResSupported = true;
                    break;
                }
        }

        return hasAutoFocus && isResSupported;
    }

    // Left for future development
//    void triggerAutoFocus() {
//        new Handler(Looper.getMainLooper()).post(new Runnable() {
//            @Override
//            public void run() {
//                if (camera != null && camera.getCamera() != null && previewing && !mManualFocusEngaged) {
//                    Camera.Parameters params = camera.getCamera().getParameters();
//                    List<Camera.Area> focusArea = new ArrayList<Camera.Area>();
//                    focusArea.add(getFocusRect());
//
//                    if (params.getMaxNumFocusAreas() > 0) {
//                        List<Camera.Area> meteringAreas = new ArrayList<Camera.Area>();
//                        meteringAreas.add(getFocusRect());
//                        params.setFocusAreas(meteringAreas);
//                        Log.i("MeteringAreas",meteringAreas.get(0).rect.toString());
//                    }
//
//                    if (params.getSupportedFocusModes().contains(Camera.Parameters.FOCUS_MODE_AUTO)) {
//                        mManualFocusEngaged = true;
//                        params.setFocusMode(Camera.Parameters.FOCUS_MODE_AUTO);
//                        camera.getCamera().setParameters(params);
//
//                        camera.getCamera().autoFocus(new Camera.AutoFocusCallback() {
//                            @Override
//                            public void onAutoFocus(boolean success, Camera camera) {
//                                mManualFocusEngaged = false;
//                            }
//                        });
//                    }
//                }
//            }
//        });
//    }
//
//    private Camera.Area getFocusRect() {
//        Camera.Area focusArea;
//        Camera.CameraInfo cameraInfo = new Camera.CameraInfo();
//        Camera.getCameraInfo(0, cameraInfo);
//
//        int orientation = (ImageUtils.ORIENTATIONS.get(camera.getOrientation()) + cameraInfo.orientation) % 360;
//
//        switch (orientation) {
//            case 90:
//                focusArea = generateFocusArea(scanningRectY, scanningRectX, scanningRectHeight, scanningRectWidth);
//                break;
//            case 180:
//                focusArea = generateFocusArea((100 - scanningRectX - scanningRectWidth), (100 - scanningRectY - scanningRectHeight),
//                        scanningRectWidth, scanningRectHeight);
//                break;
//            case 270:
//                focusArea = generateFocusArea((100 - scanningRectY - scanningRectHeight), (100 - scanningRectX - scanningRectWidth),
//                        scanningRectHeight, scanningRectWidth);
//                break;
//            default:
//                focusArea = generateFocusArea(scanningRectX, scanningRectY, scanningRectWidth, scanningRectHeight);
//        }
//        return focusArea;
//    }
//
//    private Camera.Area generateFocusArea(float x, float y, float width, float height) {
//        return new Camera.Area(new Rect(
//                (int) (x / 100.0f * 2000 - 1000),
//                (int) (y / 100.0f * 2000 - 1000),
//                (int) ((x + width) / 100.0f * 2000 - 1000),
//                (int) ((y + height) / 100.0f * 2000 - 1000)
//        ), 1000);
//    }

}
