package com.scansolutions.mrzscannerlib;

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.res.Configuration;
import android.graphics.Bitmap;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraMetadata;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.os.VibrationEffect;
import android.os.Vibrator;
import android.util.Log;
import android.widget.ImageView;

import org.json.JSONException;
import org.json.JSONObject;
import org.opencv.android.Utils;
import org.opencv.core.CvException;
import org.opencv.core.CvType;
import org.opencv.core.Mat;

import static com.scansolutions.mrzscannerlib.MRZUtils.calculateScaledRect;
import static com.scansolutions.mrzscannerlib.MRZUtils.isScannerTypeDoc;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_ID_FRONT;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_PASSPORT;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_ID_SESSION;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_MRZ;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_SWISS_DRIVING_LICENCE;


class MRZCore {

    private static final String TAG = MRZCore.class.getSimpleName();
    private static final int VIBRATION_LENGTH = 220;

    static {
        System.loadLibrary("mrz-native-lib");
    }

    boolean isTorchOn = false;
    private ScannerType scannerType = SCANNER_TYPE_MRZ;
    private final Vibrator vibrator;
    private static boolean validateCountryCodesEnabled = false;
    static MRZEffortLevel effortLevel = MRZEffortLevel.EFFORT_LEVEL_TRY_HARDER;
    static MRZResultModel tempIdSessionResult;
    static MRZOverlayState overlayState = MRZOverlayState.NORMAL;

    MRZCore(MRZOverlay mrzOverlay, ImageView debugPreview, MRZScannerListener scannerListener, Context context, ScannerType scannerType) {
        this.mrzOverlay = mrzOverlay;
        this.debugPreview = debugPreview;
        this.scannerListener = scannerListener;
        this.scannerType = scannerType;
        vibrator = (Vibrator) context.getSystemService(Context.VIBRATOR_SERVICE);
        TesseractHelper.prepareTesseract(context);
    }

    static native String mrzString(long addrInputImage,
                                   int[] squareArr,
                                   boolean forceFixImage,
                                   String path,
                                   boolean isIDActive,
                                   boolean isPassportActive,
                                   boolean isVisaActive,
                                   float scanningRectX,
                                   float scanningRectY,
                                   float scanningRectWidth,
                                   float scanningRectHeight,
//                                   long addrDebugImage,
                                   int scannerMode,
                                   boolean enableUpsideDown,
                                   boolean validateCountryCodes,
                                   int effortLevel,
                                   boolean ignoreInvalidCheckDigits);

    static native String sdkVersionNative();

    static native int registerWithLicenseKey(Context context, String key, String bundleID, String modelName, String manufacturer);

    static native long shouldCallBack(String key);

    static native String getLid(String key);

    static native String encryptCommunication(String rawString);

    static native String decryptCommunication(String encryptedString);

    static native String getAndroidID(Context context);

    static native void setValidCountryCodes(String[] strings);

    static native int faceDetection(long addrInputImage,
                                    long addrFaceOutputImage,
                                    long addrSignatureOutputImage,
                                    long addrPassportOutputImage,
                                    String directory,
                                    String countryCode,
                                    int[] squarePoints,
                                    boolean extractPortraitEnabled,
                                    boolean extractSignatureEnabled,
                                    boolean extractFullImageEnabled);

    static native int scanForIDBack(long addrInputImage,
                                    long addrIDBackImage,
                                    int[] squarePoints);

    static native String finalCacheKey(String cachedKey, String outputString, int offlineKeyRes);

    static native int[] findDocumentImageNative(long addrInputImage,
                                                String directory,
                                                boolean isPassport,
                                                boolean forceFront,
                                                float scanningRectX,
                                                float scanningRectY,
                                                float scanningRectWidth,
                                                float scanningRectHeight);

    static native boolean checkForMotion(long addrCurrentImage,
                                         long addrLastImage,
                                         float scanningRectX,
                                         float scanningRectY,
                                         float scanningRectWidth,
                                         float scanningRectHeight);

    private final MRZScannerListener scannerListener;

    static boolean isIDActive = true;
    static boolean isPassportActive = true;
    static boolean isVisaActive = true;

    static boolean enableUpsideDown = false;
    static boolean enableVibrator = true;
    static boolean continuousScanning = false;
    static boolean ignoreDuplicates = true;
    static boolean ignoreInvalidCheckDigits = true;
    static String lastResult = "";

    static boolean shouldWarnCamera = true;

    static final int STATE_SCANNING = 0;
    static final int STATE_STOPPED = 1;
    static final int STATE_FINDING_SQUARE = 2;

    static float scanningRectX = 1;
    static float scanningRectY = 30;
    static float scanningRectWidth = 98;
    static float scanningRectHeight = 40;
    static float widthRatio = 1;
    static float heightRatio = 1;

    static boolean extractPortrait = true;
    static boolean extractSignature = true;
    static boolean extractFullImage = true;
    static boolean extractIdBack = true;

    static int scanningState = STATE_STOPPED;

    static int activeThreads = 0;

    static final int AVAILABLE_THREADS = Runtime.getRuntime().availableProcessors();
    static int maxThreads = Math.min(2, AVAILABLE_THREADS);
    private final MRZOverlay mrzOverlay;
    private final ImageView debugPreview;

    void scan(Mat imageGrab) {
        if (scannerType == SCANNER_TYPE_MRZ) {
            scanForMRZ(imageGrab, true, 0);
        } else if (scannerType == SCANNER_TYPE_SWISS_DRIVING_LICENCE) {
            scanForMRZ(imageGrab, true, 2);
        } else if (isScannerTypeDoc(scannerType)) {
            scanForDocumentImage(imageGrab);
        } else if (scannerType == SCANNER_TYPE_ID_SESSION) {
            if (overlayState == MRZOverlayState.SCANNING_BACK)
                scanForMRZ(imageGrab, true, 0);
            else if (overlayState == MRZOverlayState.SCANNING_FRONT)
                scanForDocumentImage(imageGrab);
        }
    }

    private void scanForDocumentImage(Mat imageGrab) {
        int imageWidth = imageGrab.cols();
        int imageHeight = imageGrab.rows();

        if (scanningState == STATE_STOPPED)
            return;

        MRZPercentageRect scaledScanningRect = calculateScaledRect(scanningRectX, scanningRectY, scanningRectWidth, scanningRectHeight, widthRatio, heightRatio, false);
        boolean forceFront = scannerType == SCANNER_TYPE_DOC_IMAGE_ID_FRONT ||
                (scannerType == SCANNER_TYPE_ID_SESSION && overlayState == MRZOverlayState.SCANNING_FRONT);

        int[] locationSquare = findDocumentImageNative(imageGrab.getNativeObjAddr(),
                TesseractHelper.getDataPath(),
                scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT,
                forceFront,
                scaledScanningRect.x,
                scaledScanningRect.y,
                scaledScanningRect.width,
                scaledScanningRect.height
        );

        if (mrzOverlay != null)
            mrzOverlay.drawLocationOverlay(locationSquare, imageWidth, imageHeight, MRZOverlay.OverlayColor.OVERLAY_COLOR_GREEN);

        if (locationSquare.length == 8 && (scanningState == STATE_FINDING_SQUARE || scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT)) {
            if (scanningState != STATE_STOPPED) {
                scanningState = STATE_STOPPED;

                final Bitmap bitmapImg = ImageUtils.getBitmap(imageGrab);

                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        vibrateOnSuccess(vibrator);

                        if (scannerListener != null) {
                            if (scannerType == SCANNER_TYPE_ID_SESSION) {
                                ImageView imageView = null;
                                if (mrzOverlay != null) {
                                    imageView = new ImageView(mrzOverlay.getContext());
                                    imageView.setImageBitmap(bitmapImg);

                                    tempIdSessionResult.idFront = bitmapImg;
                                    scannerListener.successfulScanWithResult(tempIdSessionResult);
                                    overlayState = MRZOverlayState.NORMAL;
                                    tempIdSessionResult = null;
                                }
                            } else {
                                scannerListener.successfulScanWithDocumentImage(bitmapImg);
                            }
                        }
                    }
                });
            }
        }
    }

    static void vibrateOnSuccess(Vibrator v) {
        if (enableVibrator) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O) {
                v.vibrate(VibrationEffect.createOneShot(VIBRATION_LENGTH, VibrationEffect.DEFAULT_AMPLITUDE));
            } else {
                v.vibrate(VIBRATION_LENGTH);
            }
        }
    }

    void scanForMRZ(Bitmap bmp) {
        Mat m = new Mat(bmp.getWidth(), bmp.getHeight(), CvType.CV_8U);
        Utils.bitmapToMat(bmp, m);

        boolean success = false;
        scanningState = STATE_SCANNING;

        for (int i = 0; i < 16; i++) {
            success = scanForMRZ(m, i % 2 != 0, 1, true);
            if (success)
                break;
        }

        if (!success) {
            MRZCore.triggerScanImageFailedOnMainThread(scannerListener);
        }
    }

    private boolean scanForMRZ(Mat imageGrab, boolean forceFixImage, int scannerMode) {
        return scanForMRZ(imageGrab, forceFixImage, scannerMode, false);
    }

    private boolean scanForMRZ(Mat imageGrab, boolean forceFixImage, int scannerMode, boolean forceFullImageScan) {
        MRZPercentageRect scaledScanningRect = calculateScaledRect(scanningRectX, scanningRectY, scanningRectWidth, scanningRectHeight, widthRatio, heightRatio, forceFullImageScan);

        int[] intArr = new int[8];
        final String result = mrzString(
                imageGrab.getNativeObjAddr(),
                intArr,
                forceFixImage,
                TesseractHelper.getDataPath(),
                isIDActive,
                isPassportActive,
                isVisaActive,
                scaledScanningRect.x,
                scaledScanningRect.y,
                scaledScanningRect.width,
                scaledScanningRect.height,
//                debugMat.getNativeObjAddr(),
                scannerMode,
                enableUpsideDown,
                validateCountryCodesEnabled,
                effortLevel.getValue(),
                ignoreInvalidCheckDigits);

        if (result != null && !result.isEmpty()) {
            try {
                JSONObject jsonObject = new JSONObject(result);

                MRZOverlay.OverlayColor overlayColor = jsonObject.getBoolean("are_check_digits_valid")
                        ? MRZOverlay.OverlayColor.OVERLAY_COLOR_GREEN
                        : MRZOverlay.OverlayColor.OVERLAY_COLOR_YELLOW;

                if (mrzOverlay != null)
                    mrzOverlay.drawLocationOverlay(intArr, imageGrab.cols(), imageGrab.rows(), overlayColor);

                if (jsonObject.getBoolean("success")) {
                    if (scanningState != STATE_STOPPED) {
                        if (!continuousScanning)
                            scanningState = STATE_STOPPED;

                        long dateScanned = System.currentTimeMillis() / 1000;
                        final MRZResultModel resultModel = new MRZResultModel(jsonObject, dateScanned);

                        boolean ignoreDuplicate = ignoreDuplicates && lastResult.equals(resultModel.documentNumber);
                        boolean ignoreIfIdSessionNotId = scannerType == SCANNER_TYPE_ID_SESSION && !resultModel.isId();

                        if (ignoreDuplicate || ignoreIfIdSessionNotId) {
                            scanningState = STATE_SCANNING;
                            return false;
                        }

                        lastResult = resultModel.documentNumber;

                        vibrateOnSuccess(vibrator);

                        boolean extractImageEnabled = (extractPortrait || extractSignature || extractFullImage);
                        if (resultModel.isPassport() && extractImageEnabled) {
                            PassportImages images = scanForFaceAndSignature(imageGrab, resultModel.issuingCountry, intArr);
                            resultModel.portrait = images.face;
                            resultModel.signature = images.signature;
                            resultModel.fullImage = images.passport;
                        } else if (resultModel.isId() && extractIdBack) {
                            Mat idBackMat = new Mat();

                            scanForIDBack(imageGrab.getNativeObjAddr(),
                                    idBackMat.getNativeObjAddr(),
                                    intArr);

                            resultModel.idBack = ImageUtils.getBitmap(idBackMat);
                        }

                        new Handler(Looper.getMainLooper()).post(new Runnable() {
                            @Override
                            public void run() {
                                if (scannerType == SCANNER_TYPE_ID_SESSION) {
                                    tempIdSessionResult = resultModel;
                                    setIdSessionSecondStage();
                                } else if (scannerListener != null) {
                                    scannerListener.successfulScanWithResult(resultModel);
                                }
                            }
                        });

                        return true;
                    }
                }

//        if (BuildConfig.DEBUG) {
//            if (debugMat.cols() > 0 && debugMat.rows() > 0) {
//                final Bitmap bitmapImg = Bitmap.createBitmap(debugMat.cols(), debugMat.rows(), Bitmap.Config.ARGB_8888);
//                Utils.matToBitmap(debugMat, bitmapImg);
//                new Handler(Looper.getMainLooper()).post(new Runnable() {
//                    @Override
//                    public void run() {
//                        debugPreview.setImageBitmap(bitmapImg);
//                    }
//                });
//            }
//        }

            } catch (CvException | JSONException e) {
                Log.d("Exception", "Error message: " + e.getMessage());
            }
        }

        return false;
    }

    private void setIdSessionSecondStage() {
        overlayState = MRZOverlayState.SCANNING_FRONT;

        int orientation = mrzOverlay.getContext().getResources().getConfiguration().orientation;
        setIdSessionRect(orientation);

        new Handler().postDelayed(new Runnable() {
            @Override
            public void run() {
                scanningState = STATE_FINDING_SQUARE;
            }
        }, 500);
    }

    static void setIdSessionRect(int orientation) {
        if (MRZCore.overlayState == MRZOverlayState.SCANNING_FRONT) {
            if (orientation == Configuration.ORIENTATION_PORTRAIT) {
                MRZCore.scanningRectX = MRZOverlay.BOTTOM_RECT.x;
                MRZCore.scanningRectY = MRZOverlay.BOTTOM_RECT.y;
                MRZCore.scanningRectWidth = MRZOverlay.BOTTOM_RECT.width;
                MRZCore.scanningRectHeight = MRZOverlay.BOTTOM_RECT.height;
            } else {
                MRZCore.scanningRectX = MRZOverlay.RIGHT_RECT.x;
                MRZCore.scanningRectY = MRZOverlay.RIGHT_RECT.y;
                MRZCore.scanningRectWidth = MRZOverlay.RIGHT_RECT.width;
                MRZCore.scanningRectHeight = MRZOverlay.RIGHT_RECT.height;
            }
        } else if (MRZCore.overlayState == MRZOverlayState.SCANNING_BACK)
            if (orientation == Configuration.ORIENTATION_PORTRAIT) {
                MRZCore.scanningRectX = MRZOverlay.TOP_RECT.x;
                MRZCore.scanningRectY = MRZOverlay.TOP_RECT.y;
                MRZCore.scanningRectWidth = MRZOverlay.TOP_RECT.width;
                MRZCore.scanningRectHeight = MRZOverlay.TOP_RECT.height;
            } else {
                MRZCore.scanningRectX = MRZOverlay.LEFT_RECT.x;
                MRZCore.scanningRectY = MRZOverlay.LEFT_RECT.y;
                MRZCore.scanningRectWidth = MRZOverlay.LEFT_RECT.width;
                MRZCore.scanningRectHeight = MRZOverlay.LEFT_RECT.height;
            }
    }

    private PassportImages scanForFaceAndSignature(Mat inputImage, String issuingCountry, int[] squreArr) {
        PassportImages result = new PassportImages();
        Mat faceMat = new Mat();
        Mat signatureMat = new Mat();
        Mat passportMat = new Mat();

        faceDetection(inputImage.getNativeObjAddr(),
                faceMat.getNativeObjAddr(),
                signatureMat.getNativeObjAddr(),
                passportMat.getNativeObjAddr(),
                TesseractHelper.getDataPath(),
                issuingCountry,
                squreArr,
                extractPortrait,
                extractSignature,
                extractFullImage);

        result.face = ImageUtils.getBitmap(faceMat);
        result.signature = ImageUtils.getBitmap(signatureMat);
        result.passport = ImageUtils.getBitmap(passportMat);

        return result;
    }

    static void triggerScanImageFailedOnMainThread(final MRZScannerListener scannerListener) {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                scannerListener.scanImageFailed();
            }
        });
    }

    static boolean allowCamera2Support(Context context) {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            android.hardware.camera2.CameraManager manager = (android.hardware.camera2.CameraManager) context.getSystemService(Context.CAMERA_SERVICE);
            if (manager != null) {
                try {
                    String cameraIdS = manager.getCameraIdList()[0];
                    CameraCharacteristics characteristics = manager.getCameraCharacteristics(cameraIdS);
                    int support = characteristics.get(CameraCharacteristics.INFO_SUPPORTED_HARDWARE_LEVEL);
                    CameraLogger.addLog("allowCamera2Support: " + support);

                    return support != CameraMetadata.INFO_SUPPORTED_HARDWARE_LEVEL_LEGACY;
                } catch (Exception e) {
                    CameraLogger.addLog("allowCamera2Support Errror: " + e.getMessage());
                    e.printStackTrace();
                }
            }
        }

        return false;
    }

    void setScannerType(ScannerType scannerType) {
        this.scannerType = scannerType;
    }

    ScannerType getScannerType() {
        return this.scannerType;
    }

    static void setValidateCountryCodesEnabled(boolean enabled) {
        validateCountryCodesEnabled = enabled;
    }

}
