package com.scansolutions.mrzscannerlib;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.os.Vibrator;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.view.LayoutInflater;
import android.view.SurfaceView;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.view.WindowManager;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.RelativeLayout;
import android.widget.ScrollView;
import android.widget.Toast;

import static com.scansolutions.mrzscannerlib.MRZCore.continuousScanning;
import static com.scansolutions.mrzscannerlib.MRZCore.enableVibrator;
import static com.scansolutions.mrzscannerlib.MRZCore.ignoreDuplicates;
import static com.scansolutions.mrzscannerlib.PermissionsHelper.PERMISSION_CAMERA;
import static com.scansolutions.mrzscannerlib.PermissionsHelper.PERMISSION_GALLERY;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_MRZ;

public class MRZScanner extends Fragment implements Camera2Impl.CameraInitListener, MRZScannerListener {

    private static final int CAMERA_VERSION_1 = 1;
    private static final int CAMERA_VERSION_2 = 2;
    private static int cameraVersion = CAMERA_VERSION_2;
    private static String dateFormat = "dd.MM.yyyy";
    private TextureView mTextureView = null;
    private SurfaceView mSurfaceView = null;
    private ImageView debugPreview;
    private ImageView customImageView;
    private int customImageResId = -1;
    private MRZOverlay mrzOverlay;
    private MRZScannerListener scannerListener;
    private Camera2Impl camera2Impl;
    private Camera1Impl camera1Impl;
    private boolean permissionsWereDenied = false;
    private Bitmap customImage = null;
    private static boolean enableScanFromGallery = false;
    private ImageButton btnGallery;
    private ScannerType scannerType = SCANNER_TYPE_MRZ;

    public MRZScanner() {
    }

    /**
     * @return the current SDK Version.
     */
    public static String sdkVersion() {
        return MRZCore.sdkVersionNative();
    }

    /**
     * @return the currently set date format in which the parsed dates are formatted.
     */
    public static String getDateFormat() {
        return dateFormat;
    }

    /**
     * Set the date format in which the parsed dates are formatted.
     *
     * @param dateFormat the pattern describing the date format. Example: "dd.MM.yyyy"
     */
    public static void setDateFormat(String dateFormat) {
        MRZScanner.dateFormat = dateFormat;
    }

    @Override
    public void onResume() {
        super.onResume();
        checkPermissionsAndInit();
    }

    private void checkPermissionsAndInit() {
        if (permissionsWereDenied) return;

        if (!PermissionsHelper.hasPermissions(getActivity(), Manifest.permission.CAMERA)) {
            PermissionsHelper.start(getActivity(), PERMISSION_CAMERA, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted) {
                        initScanner();
                    } else {
                        permissionsWereDenied = true;
                        scannerListener.permissionsWereDenied();
                    }
                }
            });
        } else {
            initScanner();
        }
    }

    private void initScanner() {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.resume();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                if (mTextureView.isAvailable()) {
                    camera2Impl.openCamera2(mTextureView.getWidth(), mTextureView.getHeight());
                } else {
                    mTextureView.setSurfaceTextureListener(camera2Impl.mSurfaceTextureListener);
                }
            }
        }
    }

    @Override
    public void onPause() {
        super.onPause();
        stopScanning();
    }

    private void stopScanning() {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.stopScanner();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                camera2Impl.closeCamera2();
            }
        }
    }

    private void pauseScanning() {
        if (!continuousScanning) {
            if (cameraVersion == CAMERA_VERSION_1) {
                camera1Impl.pauseScanner();
            } else if (cameraVersion == CAMERA_VERSION_2) {
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    camera2Impl.pauseScanning();
                }
            }
        }
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState) {
        cameraVersion = (Build.VERSION.SDK_INT >= 21 && MRZCore.allowCamera2Support(getActivity()))
                ? CAMERA_VERSION_2
                : CAMERA_VERSION_1;
//        cameraVersion = CAMERA_VERSION_1;
        if (getActivity() != null)
            getActivity().getWindow().setFlags(
                    WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
                    WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED);

        final View v = inflater.inflate(R.layout.amrz_activity_scanner, container, false);
        mrzOverlay = v.findViewById(R.id.mrz_overlay);

        initScrollContent(v);
        initCameraPreviewImpl(v);
//        if (BuildConfig.DEBUG) {
//            debugPreview = v.findViewById(R.id.debug_preview);
//        }

        if (savedInstanceState == null) {
            MRZCore.lastResult = "";
        }

        return v;
    }

    private void initScrollContent(final View v) {
        final ScrollView scrollView = ViewCreator.initScrollView(v);
        final RelativeLayout rlScrollContent = v.findViewById(R.id.mrz_scroll_content);

        rlScrollContent.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
            @Override
            public void onGlobalLayout() {
                rlScrollContent.getViewTreeObserver().removeOnGlobalLayoutListener(this);

                MRZSize viewSize = new MRZSize(v.getWidth(), v.getHeight());
                MRZSize scaledSize = MRZUtils.scaleToDisplayRatio(getActivity(), viewSize);

                MRZCore.widthRatio = v.getWidth() / scaledSize.width;
                MRZCore.heightRatio = v.getHeight() / scaledSize.height;

                if (customImage == null && customImageResId == -1) {
                    if (MRZUtils.isScannerTypeDoc(scannerType)) {
                        mrzOverlay.setRect(1, 30, 98, 40);
                    } else {
                        mrzOverlay.setRect(MRZCore.scanningRectX,
                                MRZCore.scanningRectY,
                                MRZCore.width,
                                MRZCore.height,
                                MRZCore.widthRatio,
                                MRZCore.heightRatio);
                    }
                } else {
                    customImageView = ViewCreator.initCustomImageView(v, rlScrollContent, customImage, customImageResId);
                }

                ViewGroup.LayoutParams lpOverlay = mrzOverlay.getLayoutParams();
                lpOverlay.width = (int) scaledSize.width;
                lpOverlay.height = (int) scaledSize.height;
                mrzOverlay.setLayoutParams(lpOverlay);

                ViewGroup.LayoutParams lpScrollContent = rlScrollContent.getLayoutParams();
                lpScrollContent.width = (int) scaledSize.width;
                lpScrollContent.height = (int) scaledSize.height;
                rlScrollContent.setLayoutParams(lpScrollContent);

                if (cameraVersion == CAMERA_VERSION_1) {
                    mSurfaceView.setLayoutParams(new RelativeLayout.LayoutParams((int) scaledSize.width, (int) scaledSize.height));
                    rlScrollContent.addView(mSurfaceView, 0);
                } else {
                    mTextureView.setLayoutParams(new RelativeLayout.LayoutParams((int) scaledSize.width, (int) scaledSize.height));
                    rlScrollContent.addView(mTextureView, 0);
                }

                if (scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_ID) {
                    ViewCreator.initCaptureButton(v);
                }

                if (scannerType == ScannerType.SCANNER_TYPE_MRZ && enableScanFromGallery) {
                    ViewCreator.initGalleryButton(v, new View.OnClickListener() {
                        @Override
                        public void onClick(View v) {
                            pauseScanning();
                            scanFromGallery(getActivity(), scannerListener);
                        }
                    });
                }

                scrollView.getViewTreeObserver().addOnGlobalLayoutListener(new ViewTreeObserver.OnGlobalLayoutListener() {
                    public void onGlobalLayout() {
                        scrollView.getViewTreeObserver().removeOnGlobalLayoutListener(this);
                        int xPosition = scrollView.getChildAt(0).getWidth() / 2 - scrollView.getWidth() / 2;
                        int yPosition = scrollView.getChildAt(0).getHeight() / 2 - scrollView.getHeight() / 2;
                        scrollView.scrollTo(xPosition, yPosition);
                    }
                });
            }
        });
    }

    private void initCameraPreviewImpl(View v) {
        ImageButton btnFlash = v.findViewById(R.id.mrz_flash);
        if (cameraVersion == CAMERA_VERSION_1) {
            mSurfaceView = new SurfaceView(v.getContext());
            camera1Impl = new Camera1Impl(getActivity(),
                    mSurfaceView,
                    btnFlash,
                    mrzOverlay,
                    debugPreview,
                    this,
                    this,
                    scannerType);
        } else if (cameraVersion == CAMERA_VERSION_2) {
            mTextureView = new TextureView(v.getContext());
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                camera2Impl = new Camera2Impl(getActivity(),
                        this,
                        mTextureView,
                        btnFlash,
                        this,
                        mrzOverlay,
                        debugPreview,
                        scannerType);
            }
        }
    }

    @Override
    public void warnIncompatibleCamera() {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                Context c = getActivity();

                if (c != null) {
                    Toast.makeText(c, R.string.warning_message, Toast.LENGTH_LONG)
                            .show();
                }
            }
        });
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);

        try {
            scannerListener = (MRZScannerListener) context;
        } catch (ClassCastException e) {
            throw new ClassCastException(context.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);

        try {
            scannerListener = (MRZScannerListener) activity;
        } catch (ClassCastException e) {
            throw new ClassCastException(activity.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    /**
     * Set the scanning rectangle to limit the scanning area. The parameters' values are representing percentages of the scanning preview.
     *
     * @param x      the top left point of the scanning rectangle. [0,...,100] Default value: 1.
     * @param y      the top left point of the scanning rectangle. [0,...,100] Default value: 30.
     * @param width  width the width of the scanning rectangle. [0,...,100] Default value: 98.
     * @param height height the height of the scanning rectangle. [0,...,100] Default value: 40.
     */
    public void setScanningRectangle(int x, int y, int width, int height) {
        MRZCore.scanningRectX = x;
        MRZCore.scanningRectY = y;
        MRZCore.width = width;
        MRZCore.height = height;

        if (customImage != null && customImageResId == -1)
            mrzOverlay.setRect(x, y, width, height, MRZCore.widthRatio, MRZCore.heightRatio);
    }

    /**
     * Specify whether the scanner should detect and return result for IDs.
     *
     * @param isIDActive [true, false]. The default value is true.
     */
    public static void setIDActive(boolean isIDActive) {
        MRZCore.isIDActive = isIDActive;
    }

    /**
     * Specify whether the scanner should detect and return result for passports.
     *
     * @param isPassportActive [true, false]. The default value is true.
     */
    public static void setPassportActive(boolean isPassportActive) {
        MRZCore.isPassportActive = isPassportActive;
    }

    /**
     * Specify whether the scanner should detect and return result for visas.
     *
     * @param isVisaActive [true, false]. Default value is true.
     */
    public static void setVisaActive(boolean isVisaActive) {
        MRZCore.isVisaActive = isVisaActive;
    }

    /**
     * Specify the maximum number of CPU threads that the scanner can use during the scanning process.
     *
     * @param maxThreads number of CPU threads. Default value is 2.
     */
    public static void setMaxThreads(int maxThreads) {
        MRZCore.maxThreads = Math.min(MRZCore.AVAILABLE_THREADS, Math.max(maxThreads, 0));
    }

    /**
     * Specify which scanner type you want to use. There are two options: "MRZ Scanner" and "Document Image scanner".
     * The "MRZ Scanner" option is used to scan for MRZ.
     * The "Document image scanner" is used for capturing front and back image of the ID documents.
     *
     * @param scannerType [SCANNER_TYPE_MRZ, SCANNER_TYPE_DOC_IMAGE_ID, SCANNER_TYPE_DOC_IMAGE_PASSPORT]. Default value is SCANNER_TYPE_MRZ
     */
    public void setScannerType(ScannerType scannerType) {
        this.scannerType = scannerType;

        if (camera1Impl != null) {
            camera1Impl.setScannerType(scannerType);
        } else if (camera2Impl != null) {
            camera2Impl.setScannerType(scannerType);
        }
    }

    /**
     * Resume scanning after the scanner has been paused/stopped. Usually after a successful scan when continuous scanning is not enabled.
     */
    public void resumeScanning() {
        MRZCore.lastResult = "";

        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.resumeScanner();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                camera2Impl.resumeScanning();
            }
        }

        if (customImage == null && customImageResId == -1) {
            if (mrzOverlay != null) {
                if (MRZUtils.isScannerTypeDoc(scannerType)) {
                    mrzOverlay.setRect(1, 30, 98, 40);
                } else {
                    mrzOverlay.setRect(MRZCore.scanningRectX,
                            MRZCore.scanningRectY,
                            MRZCore.width,
                            MRZCore.height,
                            MRZCore.widthRatio,
                            MRZCore.heightRatio);
                }
            }
        }

        MRZCore.scanningState = MRZCore.STATE_SCANNING;
    }

    /**
     * Register with the licence key provided to remove the asterisks (*) from the result.
     *
     * @param context the activity's context.
     * @param key     the provided licence key.
     * @return 0 for success, -1 if registration failed.
     */
    public static int registerWithLicenseKey(final Context context, final String key) {
        final String bundleID = context.getApplicationInfo().packageName;

        new Thread(new Runnable() {

            @Override
            public void run() {
                HttpsConnectionHelper.reportRegistration(context, bundleID, key);
            }
        }).start();

        return MRZCore.registerWithLicenseKey(context, key, bundleID, Build.MODEL, Build.MANUFACTURER);
    }

    /**
     * Scan for a single bitmap.
     *
     * @param bitmap   the image to be scanned.
     * @param activity the initialisation activity context.
     */
    public static void scanBitmap(final Bitmap bitmap, final Activity activity, final MRZScannerListener listener) {
        if (!PermissionsHelper.hasPermissions(activity, Manifest.permission.READ_EXTERNAL_STORAGE)) {
            PermissionsHelper.start(activity, PERMISSION_GALLERY, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted)
                        scanBitmap(bitmap, activity, listener);
                    else if (listener != null)
                        listener.permissionsWereDenied();
                }
            });
            return;
        }

        new MRZCore(null, null, listener, activity, SCANNER_TYPE_MRZ)
                .scanForMRZ(bitmap);
    }

    /**
     * Trigger an image picker.
     *
     * @param activity the initialisation activity context.
     */
    public static void scanFromGallery(final Activity activity, final MRZScannerListener listener) {
        if (!PermissionsHelper.hasPermissions(activity, Manifest.permission.READ_EXTERNAL_STORAGE)) {
            PermissionsHelper.start(activity, PERMISSION_GALLERY, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted)
                        scanFromGallery(activity, listener);
                    else if (listener != null)
                        listener.permissionsWereDenied();
                }
            });
            return;
        }

        ImagePickerActivity.start(activity, listener);
    }

    /**
     * Show a warning alert dialog if the camera's resolution is lower than 5 MP and if the camera lacks the AutoFocus feature.
     *
     * @param shouldWarn [true, false]. Default value is true.
     */
    public static void warnIncompatibleCamera(boolean shouldWarn) {
        MRZCore.shouldWarnCamera = shouldWarn;
    }

    /**
     * Set valid nationality and issuing country codes. Successful scans with country codes not included in the array will be ignored.
     *
     * @param validCountryCodes array with active country codes. Example value: new String[]{"GBR", "USA"}. Default: {} (ALL).
     */
    public static void setValidCountryCodes(String[] validCountryCodes) {
        MRZCore.setValidCountryCodes(validCountryCodes);
    }

    /**
     * Specify whether validation of country codes is enabled.
     *
     * @param enabled indicates whether validation is enabled. Default: true.
     */
    public static void setValidateCountryCodesEnabled(Boolean enabled) {
        MRZCore.setValidateCountryCodesEnabled(enabled);
    }

    /**
     * Set custom overlay image for the scanner.
     *
     * @param customImage bitmap representation of the overlay image
     */
    public void setCustomImageOverlay(Bitmap customImage) {
        this.customImage = customImage;
        if (customImageView != null) {
            customImageView.setImageBitmap(customImage);
        }
    }

    /**
     * Set custom overlay image for the scanner.
     *
     * @param customImageResId resource id of the overlay image
     */
    public void setCustomImageOverlay(int customImageResId) {
        this.customImageResId = customImageResId;
        if (customImageView != null) {
            customImageView.setImageResource(customImageResId);
        }
    }

    /**
     * Enable scan from gallery. If enabled, a "scan from gallery" button will appear in the scanner preview.
     *
     * @param enabled indicates whether the "scan from gallery"  is enabled. Default value: false.
     */
    public void enableScanFromGallery(boolean enabled) {
        enableScanFromGallery = enabled;
    }

    /**
     * Enable upside-down scanning. If enabled, the scanner will also try the upside-down format of the preview.
     *
     * @param enabled indicates whether the "upside-down" feature  is enabled. Default value: false.
     */
    public static void setEnableUpsideDownScanning(boolean enabled) {
        MRZCore.enableUpsideDown = enabled;
    }

    /**
     * Turn flash on or off.
     *
     * @param on true = on, false = off. Default value: false.
     */
    public void toggleFlash(Boolean on) {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.toggleFlash(on);
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            camera2Impl.switchFlash(on);
        }
    }

    /**
     * Enable vibration on successful scan.
     *
     * @param enabled indicates whether the vibration feature  is enabled. Default value: true.
     */
    public static void setEnableVibrationOnSuccess(boolean enabled) {
        enableVibrator = enabled;
    }

    /**
     * If enabled, after successful scan, the scanner view will not be paused.
     *
     * @param enabled indicates whether the scanner should stay active after a successful scan. Default value: false.
     */
    public void setContinuousScanningEnabled(boolean enabled) {
        continuousScanning = enabled;
    }

    /**
     * Ignore duplicates when scanning continuously.
     *
     * @param ignore indicates whether the scanner should repeat the last successful scan. Default value: true.
     */
    public void setIgnoreDuplicates(boolean ignore) {
        ignoreDuplicates = ignore;
    }

    @Override
    public void successfulScanWithResult(MRZResultModel resultModel) {
        vibrate();
        scannerListener.successfulScanWithResult(resultModel);
        if (!continuousScanning)
            pauseScanning();
    }

    @Override
    public void successfulScanWithDocumentImage(Bitmap image) {
        vibrate();
        scannerListener.successfulScanWithDocumentImage(image);
    }

    @Override
    public void scanImageFailed() {
        scannerListener.scanImageFailed();
    }

    @Override
    public void permissionsWereDenied() {
        scannerListener.permissionsWereDenied();
    }

    private void vibrate() {
        Vibrator v = (Vibrator) getActivity().getSystemService(Context.VIBRATOR_SERVICE);
        MRZCore.vibrateOnSuccess(v);
    }

}
