package com.scansolutions.mrzscannerlib;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Matrix;
import android.graphics.Paint;
import android.graphics.Path;
import android.graphics.Point;
import android.os.Handler;
import android.os.Looper;
import android.util.AttributeSet;
import android.view.View;
import android.view.animation.AccelerateInterpolator;
import android.view.animation.AlphaAnimation;
import android.view.animation.Animation;
import android.widget.RelativeLayout;

class MRZOverlay extends RelativeLayout {
    private static MRZPercentageRect finderRect = new MRZPercentageRect(0, 0, 100, 100);
    private Context context;
    private Bitmap icon;
    private Bitmap icon90;
    private Bitmap icon180;
    private Bitmap icon270;
    private boolean layoutDrawn = false;
    private MRZLocationOverlay mrzLocationOverlay;
    Animation fadeOut;

    public MRZOverlay(Context context, AttributeSet attrs) {
        this(context, attrs, 10, 20, 80, 60);
    }

    public MRZOverlay(Context context, AttributeSet attrs, int x, int y, int width, int height) {
        super(context, attrs);
        this.context = context;
        finderRect = new MRZPercentageRect(x, y, width, height);

        icon = ImageUtils.getBitmap(context, R.drawable.amrz_ic_scan_border);

        Matrix matrix = new Matrix();
        matrix.postRotate(90);
        icon90 = Bitmap.createBitmap(icon, 0, 0, icon.getWidth(), icon.getHeight(), matrix, true);
        matrix.postRotate(180);
        icon180 = Bitmap.createBitmap(icon, 0, 0, icon.getWidth(), icon.getHeight(), matrix, true);
        matrix.postRotate(270);
        icon270 = Bitmap.createBitmap(icon, 0, 0, icon.getWidth(), icon.getHeight(), matrix, true);

        fadeOut = new AlphaAnimation(1, 0);
        fadeOut.setInterpolator(new AccelerateInterpolator()); //and this
        fadeOut.setStartOffset(0);
        fadeOut.setDuration(1500);
        fadeOut.setFillAfter(true);
    }

    public void setRect(float x, float y, float width, float height) {
        setRect(x, y, width, height, 1, 1);
    }

    public void setRect(float x, float y, float width, float height, float widthRatio, float heightRatio) {
        finderRect = MRZUtils.calculateScaledRect(x, y, width, height, widthRatio, heightRatio, false);

        if (layoutDrawn) {
            invalidate();
        }
    }

    @Override
    protected void onDraw(Canvas canvas) {
        super.onDraw(canvas);

        if (finderRect.x + finderRect.y + finderRect.width + finderRect.height > 1 && getHeight() > 0) {
            float xP = finderRect.x / 100.0f * getWidth();
            float yP = finderRect.y / 100.0f * getHeight();
            float widthP = finderRect.width / 100.0f * getWidth();
            float heightP = finderRect.height / 100.0f * getHeight();

            canvas.drawBitmap(icon, xP, yP + heightP - icon.getHeight(), null);
            canvas.drawBitmap(icon90, xP, yP, null);
            canvas.drawBitmap(icon180, xP + widthP - icon.getWidth(), yP + heightP - icon.getHeight(), null);
            canvas.drawBitmap(icon270, xP + widthP - icon.getWidth(), yP, null);
        }

        layoutDrawn = true;
    }


    public void drawLocationOverlay(final int[] points, final int imageWidth, final int imageHeight) {
        if (points.length != 8 ||
                points[0] == 0 ||
                points[1] == 0 ||
                points[2] == 0 ||
                points[3] == 0 ||
                points[4] == 0 ||
                points[5] == 0 ||
                points[6] == 0 ||
                points[7] == 0)
            return;

        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                if (mrzLocationOverlay == null) {
                    mrzLocationOverlay = new MRZLocationOverlay(context);
                    addView(mrzLocationOverlay);
                }

                mrzLocationOverlay.setLocationPoints(points, imageWidth, imageHeight);
                mrzLocationOverlay.clearAnimation();
                mrzLocationOverlay.startAnimation(fadeOut);
            }
        });
    }

    private static class MRZLocationOverlay extends View {
        Point[] locationPoints;
        private float imageWidth;
        private float imageHeight;
        private Paint paint37;
        private Paint paint89;

        public MRZLocationOverlay(Context context) {
            super(context);

            paint37 = new Paint();
            paint37.setColor(0x614CD863);
            paint37.setStyle(Paint.Style.FILL);

            paint89 = new Paint();
            paint89.setStrokeWidth(3);
            paint89.setColor(0xE54CD863);
        }

        @Override
        protected void onDraw(Canvas canvas) {
            super.onDraw(canvas);
            float scaleX = getWidth() / imageWidth;
            float scaleY = getHeight() / imageHeight;

            if (locationPoints != null && locationPoints.length == 4) {
                Path path = new Path();
                path.moveTo(locationPoints[0].x * scaleX, locationPoints[0].y * scaleY);

                for (int i = 1; i < locationPoints.length; i++) {
                    path.lineTo(locationPoints[i].x * scaleX, locationPoints[i].y * scaleY);
                }

                path.lineTo(locationPoints[0].x * scaleX, locationPoints[0].y * scaleY);
                path.close();

                canvas.drawPath(path, paint37);
                paint89.setStyle(Paint.Style.STROKE);
                canvas.drawPath(path, paint89);

                paint89.setStyle(Paint.Style.FILL);
                for (Point p : locationPoints) {
                    canvas.drawCircle(p.x * scaleX, p.y * scaleY, 5, paint89);
                }
            }
        }

        public void setLocationPoints(int[] points, int imageWidth, int imageHeight) {
            if (applyLocationPoints(points, imageWidth, imageHeight))
                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        invalidate();
                    }
                });
        }

        private boolean applyLocationPoints(int[] points, int imageWidth, int imageHeight) {
            boolean isLengthAcceptable = points.length == 8;
            if (isLengthAcceptable) {
                locationPoints = new Point[4];
                locationPoints[0] = new Point(points[0], points[1]);
                locationPoints[1] = new Point(points[2], points[3]);
                locationPoints[2] = new Point(points[4], points[5]);
                locationPoints[3] = new Point(points[6], points[7]);
                this.imageWidth = imageWidth;
                this.imageHeight = imageHeight;
            }

            return isLengthAcceptable;
        }
    }
}
