package com.scansolutions.mrzscannerlib;

import android.Manifest;
import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.os.Build;
import android.os.Bundle;
import android.os.Handler;
import android.os.Looper;
import android.support.annotation.Nullable;
import android.support.v4.app.Fragment;
import android.view.LayoutInflater;
import android.view.SurfaceView;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.ImageButton;
import android.widget.ImageView;
import android.widget.Toast;

import static com.scansolutions.mrzscannerlib.PermissionsHelper.PERMISSION_CAMERA;
import static com.scansolutions.mrzscannerlib.PermissionsHelper.PERMISSION_GALLERY;

public class MRZScanner extends Fragment implements Camera2Impl.CameraInitListener, MRZScannerListener {

    private static final int CAMERA_VERSION_1 = 1;
    private static final int CAMERA_VERSION_2 = 2;
    private static int cameraVersion = CAMERA_VERSION_2;
    private static String dateFormat = "dd.MM.yyyy";
    private TextureView mTextureView;
    private ImageView debugPreview;
    private ImageView customImageView;
    private int customImageResId = -1;
    private MRZOverlay mrzOverlay;
    private MRZScannerListener scannerListener;
    private Camera2Impl camera2Impl;
    private Camera1Impl camera1Impl;
    private boolean permissionsWereDenied = false;
    private Bitmap customImage = null;
    private boolean enableScanFromGallery = false;
    private ImageButton btnGallery;

    public MRZScanner() {
    }

    /**
     * @return the current SDK Version.
     */
    public static String sdkVersion() {
        return MRZCore.sdkVersionNative();
    }

    /**
     * @return the currently set date format in which the parsed dates are formatted.
     */
    public static String getDateFormat() {
        return dateFormat;
    }

    /**
     * Set the date format in which the parsed dates are formatted.
     *
     * @param dateFormat the pattern describing the date format. Example: "dd.MM.yyyy"
     */
    public static void setDateFormat(String dateFormat) {
        MRZScanner.dateFormat = dateFormat;
    }

    @Override
    public void onResume() {
        super.onResume();
        if (permissionsWereDenied) return;

        if (!PermissionsHelper.hasPermissions(getActivity(), /*Manifest.permission.WRITE_EXTERNAL_STORAGE,*/ Manifest.permission.CAMERA)) {
            PermissionsHelper.start(getActivity(), PERMISSION_CAMERA, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted) {
                        if (cameraVersion == CAMERA_VERSION_1) {
                            camera1Impl.initCamera();
                        } else if (cameraVersion == CAMERA_VERSION_2) {
                            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                                if (mTextureView.isAvailable()) {
                                    camera2Impl.openCamera2(mTextureView.getWidth(), mTextureView.getHeight());
                                } else {
                                    mTextureView.setSurfaceTextureListener(camera2Impl.mSurfaceTextureListener);
                                }
                            }
                        }
                    } else {
                        permissionsWereDenied = true;
                        scannerListener.permissionsWereDenied();
                    }
                }
            });
        } else {
            initScanner();
        }

        View v = getView();
        if (v != null) {
            Button btnCapture = v.findViewById(R.id.captureButton);

            if (MRZCore.scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_ID) {
                btnCapture.setVisibility(View.VISIBLE);
                btnCapture.setOnClickListener(new View.OnClickListener() {
                    @Override
                    public void onClick(View v) {
                        MRZCore.scanningState = MRZCore.STATE_FINDING_SQUARE;
                    }
                });
            } else {
                btnCapture.setVisibility(View.GONE);
            }
        }
    }

    private void initScanner() {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.resume();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                if (mTextureView.isAvailable()) {
                    camera2Impl.openCamera2(mTextureView.getWidth(), mTextureView.getHeight());
                } else {
                    mTextureView.setSurfaceTextureListener(camera2Impl.mSurfaceTextureListener);
                }
            }
        }
    }

    @Override
    public void onPause() {
        super.onPause();

        stopScanning();
    }

    private void stopScanning() {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.stopScanner();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                camera2Impl.closeCamera2();
            }
        }
    }

    private void pauseScanning() {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.pauseScanner();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                camera2Impl.pauseScanning();
            }
        }
    }

    @Nullable
    @Override
    public View onCreateView(LayoutInflater inflater, @Nullable ViewGroup container, Bundle savedInstanceState) {
        cameraVersion = (Build.VERSION.SDK_INT >= 21 && MRZCore.allowCamera2Support(getActivity()))
                ? CAMERA_VERSION_2
                : CAMERA_VERSION_1;
        cameraVersion = CAMERA_VERSION_1;
        if (getActivity() != null)
            getActivity().getWindow().setFlags(
                    WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED,
                    WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED);

        View v = inflater.inflate(R.layout.amrz_activity_scanner, container, false);
        ImageButton btnFlash = v.findViewById(R.id.mrz_flash);

        if (MRZCore.scannerType == ScannerType.SCANNER_TYPE_MRZ) {
            btnGallery = v.findViewById(R.id.mrz_gallery);
            if (enableScanFromGallery) {
                btnGallery.setVisibility(View.VISIBLE);
            } else {
                btnGallery.setVisibility(View.GONE);
            }

            btnGallery.setOnClickListener(new View.OnClickListener() {
                @Override
                public void onClick(View v) {
                    pauseScanning();
                    scanFromGallery(getActivity(), scannerListener);
                }
            });
        }

        mrzOverlay = v.findViewById(R.id.mrz_overlay);
        customImageView = v.findViewById(R.id.custom_image);

        if (customImage != null) {
            customImageView.setImageBitmap(customImage);
        } else if (customImageResId != -1) {
            customImageView.setImageResource(customImageResId);
        } else {
            mrzOverlay.setRect(MRZCore.scanningRectX, MRZCore.scanningRectY, MRZCore.width, MRZCore.height);
        }

//        if (BuildConfig.DEBUG) {
//            debugPreview = v.findViewById(R.id.debug_preview);
//        }

        if (cameraVersion == CAMERA_VERSION_1) {
            SurfaceView mSurfaceView = v.findViewById(R.id.preview_view);
            mSurfaceView.setVisibility(View.VISIBLE);
            camera1Impl = new Camera1Impl(getActivity(), mSurfaceView, btnFlash, mrzOverlay, debugPreview, this, this);
        } else if (cameraVersion == CAMERA_VERSION_2) {
            mTextureView = v.findViewById(R.id.texture);
            mTextureView.setVisibility(View.VISIBLE);
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                camera2Impl = new Camera2Impl(getActivity(), this, mTextureView, btnFlash, this, mrzOverlay, debugPreview);
            }
        }

        return v;
    }

    @Override
    public void warnIncompatibleCamera() {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                Context c = getActivity();

                if (c != null) {
                    Toast.makeText(c, R.string.warning_message, Toast.LENGTH_LONG)
                            .show();
                }
            }
        });
    }

    @Override
    public void onAttach(Context context) {
        super.onAttach(context);

        try {
            scannerListener = (MRZScannerListener) context;
        } catch (ClassCastException e) {
            throw new ClassCastException(context.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    @Override
    public void onAttach(Activity activity) {
        super.onAttach(activity);

        try {
            scannerListener = (MRZScannerListener) activity;
        } catch (ClassCastException e) {
            throw new ClassCastException(activity.toString()
                    + "MRZScannerListener must be implemented");
        }
    }

    /**
     * Set the scanning rectangle to limit the scanning area. The parameters' values are representing percentages of the scanning preview.
     *
     * @param x      the top left point of the scanning rectangle. [0,...,100] Default value: 1.
     * @param y      the top left point of the scanning rectangle. [0,...,100] Default value: 30.
     * @param width  width the width of the scanning rectangle. [0,...,100] Default value: 98.
     * @param height height the height of the scanning rectangle. [0,...,100] Default value: 40.
     */
    public void setScanningRectangle(int x, int y, int width, int height) {
        MRZCore.scanningRectX = x;
        MRZCore.scanningRectY = y;
        MRZCore.width = width;
        MRZCore.height = height;
        mrzOverlay.setRect(x, y, width, height);
    }

    /**
     * Specify whether the scanner should detect and return result for IDs.
     *
     * @param isIDActive [true, false]. The default value is true.
     */
    public static void setIDActive(boolean isIDActive) {
        MRZCore.isIDActive = isIDActive;
    }

    /**
     * Specify whether the scanner should detect and return result for passports.
     *
     * @param isPassportActive [true, false]. The default value is true.
     */
    public static void setPassportActive(boolean isPassportActive) {
        MRZCore.isPassportActive = isPassportActive;
    }

    /**
     * Specify whether the scanner should detect and return result for visas.
     *
     * @param isVisaActive [true, false]. Default value is true.
     */
    public static void setVisaActive(boolean isVisaActive) {
        MRZCore.isVisaActive = isVisaActive;
    }

    /**
     * Specify the maximum number of CPU threads that the scanner can use during the scanning process.
     *
     * @param maxThreads number of CPU threads. Default value is 2.
     */
    public static void setMaxThreads(int maxThreads) {
        MRZCore.maxThreads = Math.min(MRZCore.AVAILABLE_THREADS, Math.max(maxThreads, 0));
    }

    /**
     * Specify which scanner type you want to use. There are two options: "MRZ Scanner" and "Document Image scanner".
     * The "MRZ Scanner" option is used to scan for MRZ.
     * The "Document image scanner" is used for capturing front and back image of the ID documents.
     *
     * @param scannerType [SCANNER_TYPE_MRZ, SCANNER_TYPE_DOC_IMAGE_ID, SCANNER_TYPE_DOC_IMAGE_PASSPORT]. Default value is SCANNER_TYPE_MRZ
     */
    public void setScannerType(ScannerType scannerType) {
        MRZCore.scannerType = scannerType;

        if (scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_PASSPORT || scannerType == ScannerType.SCANNER_TYPE_DOC_IMAGE_ID) {
            setScanningRectangle(1, 30, 98, 40);
        }
    }

    /**
     * Resume scanning after the scanner has been paused/stopped. Usually after a successful scan.
     */
    public void resumeScanning() {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.resumeScanner();
        } else if (cameraVersion == CAMERA_VERSION_2) {
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                camera2Impl.resumeScanning();
            }
        }

        MRZCore.scanningState = MRZCore.STATE_SCANNING;
    }

    /**
     * Register with the licence key provided to remove the asterisks (*) from the result.
     *
     * @param context the activity's context.
     * @param key     the provided licence key.
     * @return 0 for success, -1 if registration failed.
     */
    public static int registerWithLicenseKey(final Context context, final String key) {
        final String bundleID = context.getApplicationInfo().packageName;

        new Thread(new Runnable() {

            @Override
            public void run() {
                HttpsConnectionHelper.reportRegistration(context, bundleID, key);
            }
        }).start();

        return MRZCore.registerWithLicenseKey(context, key, bundleID, Build.MODEL, Build.MANUFACTURER);
    }

    /**
     * Scan for a single bitmap.
     *
     * @param bitmap   the image to be scanned.
     * @param activity the initialisation activity context.
     */
    public static void scanBitmap(final Bitmap bitmap, final Activity activity, final MRZScannerListener listener) {
        if (!PermissionsHelper.hasPermissions(activity, Manifest.permission.READ_EXTERNAL_STORAGE)) {
            PermissionsHelper.start(activity, PERMISSION_GALLERY, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted)
                        scanBitmap(bitmap, activity, listener);
                    else if (listener != null)
                        listener.permissionsWereDenied();
                }
            });
            return;
        }

        new MRZCore(null, null, listener, activity)
                .scanForMRZ(bitmap);
    }

    /**
     * Trigger an image picker.
     *
     * @param activity the initialisation activity context.
     */
    public static void scanFromGallery(final Activity activity, final MRZScannerListener listener) {
        if (!PermissionsHelper.hasPermissions(activity, Manifest.permission.READ_EXTERNAL_STORAGE)) {
            PermissionsHelper.start(activity, PERMISSION_GALLERY, new PermissionsHelper.PermissionsHelperCallback() {
                @Override
                public void onRequestedPermissionsGranted(boolean granted) {
                    if (granted)
                        scanFromGallery(activity, listener);
                    else if (listener != null)
                        listener.permissionsWereDenied();
                }
            });
            return;
        }

        ImagePickerActivity.start(activity, listener);
    }

    /**
     * Show a warning alert dialog if the camera's resolution is lower than 5 MP and if the camera lacks the AutoFocus feature.
     *
     * @param shouldWarn [true, false]. Default value is true.
     */
    public static void warnIncompatibleCamera(boolean shouldWarn) {
        MRZCore.shouldWarnCamera = shouldWarn;
    }

    /**
     * Set valid nationality and issuing country codes. Successful scans with country codes not included in the array will be ignored.
     *
     * @param validCountryCodes array with active country codes. Example value: new String[]{"GBR", "USA"}. Default: {} (ALL).
     */
    public static void setValidCountryCodes(String[] validCountryCodes) {
        MRZCore.setValidCountryCodes(validCountryCodes);
    }

    /**
     * Specify whether validation of country codes is enabled.
     *
     * @param enabled indicates whether validation is enabled. Default: true.
     */
    public static void setValidateCountryCodesEnabled(Boolean enabled) {
        MRZCore.setValidateCountryCodesEnabled(enabled);
    }

    /**
     * Set custom overlay image for the scanner.
     *
     * @param customImage bitmap representation of the overlay image
     */
    public void setCustomImageOverlay(Bitmap customImage) {
        this.customImage = customImage;
        if (customImageView != null) {
            customImageView.setImageBitmap(customImage);
        }
    }

    /**
     * Set custom overlay image for the scanner.
     *
     * @param customImageResId resource id of the overlay image
     */
    public void setCustomImageOverlay(int customImageResId) {
        this.customImageResId = customImageResId;
        if (customImageView != null) {
            customImageView.setImageResource(customImageResId);
        }
    }

    /**
     * Enable scan from gallery. If enabled, a "scan from gallery" button will appear in the scanner preview.
     *
     * @param enabled indicates whether the "scan from gallery"  is enabled. Default value: false.
     */
    public void enableScanFromGallery(boolean enabled) {
        enableScanFromGallery = enabled;

        if (btnGallery != null) {
            if (enableScanFromGallery) {
                btnGallery.setVisibility(View.VISIBLE);
            } else {
                btnGallery.setVisibility(View.GONE);
            }
        }
    }

    /**
     * Enable upside-down scanning. If enabled, the scanner will also try the upside-down format of the preview.
     *
     * @param enabled indicates whether the "upside-down" feature  is enabled. Default value: false.
     */
    public static void setEnableUpsideDownScanning(boolean enabled) {
        MRZCore.enableUpsideDown = enabled;
    }

    @Override
    public void successfulScanWithResult(MRZResultModel resultModel) {
        scannerListener.successfulScanWithResult(resultModel);
        pauseScanning();
    }

    @Override
    public void successfulScanWithDocumentImage(Bitmap image) {
        scannerListener.successfulScanWithDocumentImage(image);
    }

    @Override
    public void scanImageFailed() {
        scannerListener.scanImageFailed();
    }

    @Override
    public void permissionsWereDenied() {
        scannerListener.permissionsWereDenied();
    }

    /**
     * Turn flash on or off.
     *
     * @param on true = on, false = off. Default value: false.
     */
    public void toggleFlash(Boolean on) {
        if (cameraVersion == CAMERA_VERSION_1) {
            camera1Impl.toggleFlash(on);
        } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            camera2Impl.switchFlash(on);
        }
    }

}
