package com.scansolutions.mrzscannerlib;

import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.hardware.camera2.CameraCharacteristics;
import android.hardware.camera2.CameraMetadata;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.util.SparseIntArray;
import android.view.Surface;
import android.widget.ImageView;

import org.opencv.android.Utils;
import org.opencv.core.Core;
import org.opencv.core.CvException;
import org.opencv.core.CvType;
import org.opencv.core.Mat;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.List;

import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_ID;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_PASSPORT;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_MRZ;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_SWISS_DRIVING_LICENCE;


class MRZCore {

    private static final String TAG = MRZCore.class.getSimpleName();

    static {
        System.loadLibrary("mrz-native-lib");
    }

    boolean isTorchOn = false;

    MRZCore(MRZOverlay mrzOverlay, ImageView debugPreview, MRZScannerListener scannerListener, Context context) {
        this.mrzOverlay = mrzOverlay;
        this.debugPreview = debugPreview;
        this.scannerListener = scannerListener;
        prepareTesseract(context);
    }

    static native String mrzString(long addrInputImage,
                                   int[] squareArr,
                                   boolean forceFixImage,
                                   String path,
                                   boolean isIDActive,
                                   boolean isPassportActive,
                                   boolean isVisaActive,
                                   int scanningRectX,
                                   int scanningRectY,
                                   int scanningRectWidth,
                                   int scanningRectHeight,
//                                   long addrDebugImage,
                                   int scannerMode,
                                   boolean enableUpsideDown);

    static native String sdkVersionNative();

    static native int registerWithLicenseKey(Context context, String key, String bundleID, String modelName, String manufacturer);

    static native long shouldCallBack(String key);

    static native String getLid(String key);

    static native String encryptCommunication(String rawString);

    static native String decryptCommunication(String encryptedString);

    static native String getAndroidID(Context context);

    static native void setValidCountryCodes(String[] strings);

    static native void setValidateCountryCodesEnabled(boolean enabled);

    static native int faceDetection(long addrInputImage,
                                    long addrFaceOutputImage,
                                    long addrSignatureOutputImage,
                                    long addrPassportOutputImage,
                                    String directory,
                                    String countryCode,
                                    int[] squarePoints);

    static native int[] findDocumentImageNative(long addrInputImage, boolean isPassport);

    private MRZScannerListener scannerListener;

    private static final SparseIntArray ORIENTATIONS = new SparseIntArray();

    static {
        ORIENTATIONS.append(Surface.ROTATION_0, 90);
        ORIENTATIONS.append(Surface.ROTATION_90, 0);
        ORIENTATIONS.append(Surface.ROTATION_180, 270);
        ORIENTATIONS.append(Surface.ROTATION_270, 180);
    }

    static boolean isIDActive = true;
    static boolean isPassportActive = true;
    static boolean isVisaActive = true;

    static boolean enableUpsideDown = false;

    static boolean shouldWarnCamera = true;

    static final int STATE_SCANNING = 0;
    static final int STATE_STOPPED = 1;
    static final int STATE_FINDING_SQUARE = 2;

    static int scanningRectX = 1;
    static int scanningRectY = 30;
    static int width = 98;
    static int height = 40;

    static int scanningState = STATE_STOPPED;
    static ScannerType scannerType = SCANNER_TYPE_MRZ;

    private static final String TESSDATA = "tessdata";
    private static String DATA_PATH;

    static int activeThreads = 0;

    static final int AVAILABLE_THREADS = Runtime.getRuntime().availableProcessors();
    static int maxThreads = Math.min(2, AVAILABLE_THREADS);
    private MRZOverlay mrzOverlay;
    private ImageView debugPreview;

    private static long lastRes;

    void scan(Mat imageGrab) {
        if (scannerType == SCANNER_TYPE_MRZ) {
            scanForMRZ(imageGrab, true, 0);
        } else if (scannerType == SCANNER_TYPE_SWISS_DRIVING_LICENCE) {
            scanForMRZ(imageGrab, true, 2);
        } else if (isScannerTypeDoc()) {
            scanForDocumentImage(imageGrab);
        }
    }

    private void scanForDocumentImage(Mat imageGrab) {
        int imageWidth = imageGrab.cols();
        int imageHeight = imageGrab.rows();

        if (scanningState == STATE_STOPPED)
            return;

        int[] locationSquare = findDocumentImageNative(imageGrab.getNativeObjAddr(), scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT);

        if (mrzOverlay != null)
            mrzOverlay.drawLocationOverlay(locationSquare, imageWidth, imageHeight);

        if (locationSquare.length == 8 && (scanningState == STATE_FINDING_SQUARE || scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT)) {
            if (scanningState != STATE_STOPPED) {
                scanningState = STATE_STOPPED;

                final Bitmap bitmapImg = Bitmap.createBitmap(imageGrab.cols(), imageGrab.rows(), Bitmap.Config.ARGB_8888);
                Utils.matToBitmap(imageGrab, bitmapImg);

                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        if (scannerListener != null) {
                            scannerListener.successfulScanWithDocumentImage(bitmapImg);
                        }
                    }
                });
            }
        }
    }

    void scanForMRZ(Bitmap bmp) {
        Mat m = new Mat(bmp.getWidth(), bmp.getHeight(), CvType.CV_8U);
        Utils.bitmapToMat(bmp, m);

        boolean success = false;
        scanningState = STATE_SCANNING;

        for (int i = 0; i < 10; i++) {
            success = scanForMRZ(m, i % 2 != 0, 1);
            if (success)
                break;
        }

        if (!success) {
            MRZCore.triggerScanImageFailedOnMainThread(scannerListener);
        }
    }

    private boolean scanForMRZ(Mat imageGrab, boolean forceFixImage, int scannerMode) {
        int[] intArr = new int[8];
        String result = mrzString(
                imageGrab.getNativeObjAddr(),
                intArr,
                forceFixImage,
                DATA_PATH + TESSDATA,
                isIDActive,
                isPassportActive,
                isVisaActive,
                scanningRectX,
                scanningRectY,
                width,
                height,
//                debugMat.getNativeObjAddr(),
                scannerMode,
                enableUpsideDown);

        if (mrzOverlay != null)
            mrzOverlay.drawLocationOverlay(intArr, imageGrab.cols(), imageGrab.rows());

//        if (BuildConfig.DEBUG) {
//            if (debugMat.cols() > 0 && debugMat.rows() > 0) {
//                final Bitmap bitmapImg = Bitmap.createBitmap(debugMat.cols(), debugMat.rows(), Bitmap.Config.ARGB_8888);
//                Utils.matToBitmap(debugMat, bitmapImg);
//                new Handler(Looper.getMainLooper()).post(new Runnable() {
//                    @Override
//                    public void run() {
//                        debugPreview.setImageBitmap(bitmapImg);
//                    }
//                });
//            }
//        }

        if (!result.isEmpty()) {
            try {
                if (scanningState != STATE_STOPPED) {
                    long currentRes = System.currentTimeMillis();

                    if (currentRes - lastRes < 150)
                        return false;

                    lastRes = currentRes;
                    scanningState = STATE_STOPPED;

                    long dateScanned = System.currentTimeMillis() / 1000;
                    final MRZResultModel resultModel = new MRZResultModel(result, dateScanned);

                    if (resultModel.documentTypeRaw.startsWith("P")) {
                        ArrayList<Bitmap> images = scanForFaceAndSignature(imageGrab, resultModel.issuingCountry, intArr);
                        resultModel.portrait = images.get(0);
                        resultModel.signature = images.get(1);
                        resultModel.fullImage = images.get(2);
                    }

                    new Handler(Looper.getMainLooper()).post(new Runnable() {
                        @Override
                        public void run() {
                            if (scannerListener != null) {
                                scannerListener.successfulScanWithResult(resultModel);
                            }
                        }
                    });
                    return true;
                }
            } catch (CvException e) {
                Log.d("Exception", "Error message: " + e.getMessage());
            }
        }

        return false;
    }

    private ArrayList<Bitmap> scanForFaceAndSignature(Mat inputImage, String issuingCountry, int[] squreArr) {
        ArrayList<Bitmap> imagesList = new ArrayList<>(3);
        Mat faceMat = new Mat();
        Mat signatureMat = new Mat();
        Mat passportMat = new Mat();

        faceDetection(inputImage.getNativeObjAddr(), faceMat.getNativeObjAddr(), signatureMat.getNativeObjAddr(), passportMat.getNativeObjAddr(),
                DATA_PATH + TESSDATA, issuingCountry, squreArr);

        if (!faceMat.empty()) {
            Bitmap faceImg = Bitmap.createBitmap(faceMat.cols(), faceMat.rows(), Bitmap.Config.ARGB_8888);
            Utils.matToBitmap(faceMat, faceImg);
            imagesList.add(0, faceImg);
        } else {
            imagesList.add(0, null);
        }

        if (!signatureMat.empty()) {
            Bitmap signatureImg = Bitmap.createBitmap(signatureMat.cols(), signatureMat.rows(), Bitmap.Config.ARGB_8888);
            Utils.matToBitmap(signatureMat, signatureImg);
            imagesList.add(1, signatureImg);
        } else {
            imagesList.add(1, null);
        }

        if (!passportMat.empty()) {
            Bitmap passportImg = Bitmap.createBitmap(passportMat.cols(), passportMat.rows(), Bitmap.Config.ARGB_8888);
            Utils.matToBitmap(passportMat, passportImg);
            imagesList.add(2, passportImg);
        } else {
            imagesList.add(2, null);
        }

        return imagesList;
    }

    private static void prepareTesseract(Context context) {
        DATA_PATH = context.getFilesDir().getAbsolutePath() + "/TesseractFiles/";
        try {
            prepareDirectory(DATA_PATH + TESSDATA);
        } catch (Exception e) {
            e.printStackTrace();
        }

        copyTessDataFiles(context);
    }

    private static void prepareDirectory(String path) {
        File dir = new File(path);
        dir.setReadable(true);
        dir.setWritable(true);

        if (!dir.mkdirs()) {
//            Log.e(TAG, "ERROR: Creation of directory " + path + " failed, check does Android Manifest have permission to write to external storage.");
        }
    }

    private static void copyTessDataFiles(Context context) {
        try {
            String[] fileList = context.getAssets().list(TESSDATA);

            if (fileList != null) {
                for (String fileName : fileList) {
                    String pathToDataFile = DATA_PATH + TESSDATA + "/" + fileName;
                    InputStream in = context.getAssets().open(TESSDATA + "/" + fileName);

                    OutputStream out = new FileOutputStream(pathToDataFile, false);

                    byte[] buf = new byte[1024];
                    int len;

                    while ((len = in.read(buf)) > 0) {
                        out.write(buf, 0, len);
                    }

                    in.close();
                    out.close();
                }
            }
        } catch (IOException e) {
            Log.e(TAG, "Unable to copy files to tessdata " + e.toString());
        }
    }

    static Point getBestFitSize(List<org.opencv.core.Size> sizes, float screenAR) {
        org.opencv.core.Size bestFit = null;
        int difference = Integer.MAX_VALUE;
        final float ASPECT_TOLERANCE = 0.2f;

        for (org.opencv.core.Size s : sizes) {
            float currentAR = (float) (Math.max(s.width, s.height) / (float) Math.min(s.width, s.height));

            if (Math.abs(currentAR - screenAR) <= ASPECT_TOLERANCE) {
                int biggerSide = (int) Math.max(s.width, s.height);
                int currentDiff = Math.abs(biggerSide - 1280);

                if (currentDiff < difference) {
                    difference = currentDiff;
                    bestFit = s;
                } else if (currentDiff == difference && bestFit != null) {
                    float bfAR = (float) (Math.max(bestFit.width, bestFit.height) / (float) Math.min(bestFit.width, bestFit.height));
                    if (Math.abs(currentAR - screenAR) < Math.abs(bfAR - screenAR)) {
                        bestFit = s;
                    }
                }
            }
        }

        if (bestFit == null) {
            for (org.opencv.core.Size s : sizes) {
                int biggerSide = (int) Math.max(s.width, s.height);
                int currentDiff = Math.abs(biggerSide - 1280);

                if (currentDiff < difference) {
                    difference = currentDiff;
                    bestFit = s;
                } else if (currentDiff == difference && bestFit != null) {
                    float currentSmallAR = (float) (Math.max(s.width, s.height) / (float) Math.min(s.width, s.height));
                    float bfSmallAR = (float) (Math.max(bestFit.width, bestFit.height) / (float) Math.min(bestFit.width, bestFit.height));
                    if (Math.abs(currentSmallAR - screenAR) < Math.abs(bfSmallAR - screenAR)) {
                        bestFit = s;
                    }
                }
            }
        }

        return new Point((int) bestFit.width, (int) bestFit.height);
    }

    Mat rotateMat(Mat mat, Context context, int sensorOrientation) {
        int rotation = ((Activity) context).getWindowManager().getDefaultDisplay().getRotation();
        int orientation = getOrientation(rotation, sensorOrientation);
        if (orientation == 180 || orientation == -180) {
            Core.flip(mat, mat, -1);
        } else if (orientation == 90 || orientation == -270) {
            Core.flip(mat.t(), mat, 1);
        } else if (orientation == 270 || orientation == -90) {
            Core.flip(mat.t(), mat, 0);
        }
        return mat;
    }

    /**
     * Retrieves the JPEG orientation from the specified screen rotation.
     *
     * @param rotation The screen rotation.
     * @return The JPEG orientation (one of 0, 90, 270, and 360)
     */
    private int getOrientation(int rotation, int sensorOrientation) {
        // Sensor orientation is 90 for most devices, or 270 for some devices (eg. Nexus 5X)
        // We have to take that into account and rotate JPEG properly.
        // For devices with orientation of 90, we simply return our mapping from ORIENTATIONS.
        // For devices with orientation of 270, we need to rotate the JPEG 180 degrees.

        return (ORIENTATIONS.get(rotation) + sensorOrientation + 270) % 360;
    }

    private static boolean isScannerTypeDoc() {
        return scannerType == SCANNER_TYPE_DOC_IMAGE_ID || scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT;
    }

    static void triggerScanImageFailedOnMainThread(final MRZScannerListener scannerListener) {
        new Handler(Looper.getMainLooper()).post(new Runnable() {
            @Override
            public void run() {
                scannerListener.scanImageFailed();
            }
        });
    }

    static boolean allowCamera2Support(Context context) {
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP) {
            android.hardware.camera2.CameraManager manager = (android.hardware.camera2.CameraManager) context.getSystemService(Context.CAMERA_SERVICE);
            if (manager != null) {
                try {

                    String cameraIdS = manager.getCameraIdList()[0];
                    CameraCharacteristics characteristics = manager.getCameraCharacteristics(cameraIdS);
                    int support = characteristics.get(CameraCharacteristics.INFO_SUPPORTED_HARDWARE_LEVEL);
                    return support == CameraMetadata.INFO_SUPPORTED_HARDWARE_LEVEL_LIMITED || support == CameraMetadata.INFO_SUPPORTED_HARDWARE_LEVEL_FULL;
                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }

        return false;
    }

}
