package com.scansolutions.mrzscannerlib;

import android.graphics.Bitmap;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Locale;

public class MRZResultModel {

    static {
        System.loadLibrary("native-lib");
    }

    public MRZResultModel() {
    }

    public native String parseReadableDocumentType(String text);

    public native String calculateEstIssuingDate(String document_type_raw, String issuing_country, String dob_raw, String expiration_date_raw);

    public native String parseDate(String text);

    public String document_type_raw = "N/A";
    public String document_type_readable = "N/A";
    public String issuing_country = "N/A";
    public String[] surnames;
    public String[] given_names;
    public String document_number = "N/A";
    public String nationality = "N/A";
    public String dob_raw = "N/A";
    public String dob_readable = "N/A";
    public String sex = "N/A";
    public String expiration_date_raw = "N/A";
    public String expiration_date_readable = "N/A";
    public String est_issuing_date_raw = "N/A";
    public String est_issuing_date_readable = "N/A";
    public String[] optionals;
    public long dateScanned;
    public Bitmap portrait;
    public Bitmap signature;

    public MRZResultModel(String document_type_raw,
                          String issuing_country,
                          String[] surnames,
                          String[] given_names,
                          String document_number,
                          String nationality,
                          String dob_raw,
                          String sex,
                          String est_issuing_date_raw,
                          String expiration_date_raw,
                          String[] optionals,
                          long dateScanned,
                          Bitmap portrait,
                          Bitmap signature) {
        this.document_type_raw = document_type_raw;
        this.issuing_country = issuing_country;
        this.surnames = surnames;
        this.given_names = given_names;
        this.document_number = document_number;
        this.nationality = nationality;
        this.dob_raw = dob_raw;
        this.sex = sex;
        this.expiration_date_raw = expiration_date_raw;
        this.est_issuing_date_raw = est_issuing_date_raw;
        this.optionals = optionals;
        this.dateScanned = dateScanned;
        this.portrait = portrait;
        this.signature = signature;

        parseRawToReadable();
    }

    MRZResultModel(String jsonText, long dateScanned) {
        this.dateScanned = dateScanned;

        try {
            JSONObject jsonResult = new JSONObject(jsonText);
            document_type_raw = jsonResult.getString("document_type_raw");
            document_type_readable = jsonResult.getString("document_type_readable");
            issuing_country = jsonResult.getString("issuing_country");
            document_number = jsonResult.getString("document_number");
            nationality = jsonResult.getString("nationality");
            dob_raw = jsonResult.getString("dob_raw");
            dob_readable = jsonResult.getString("dob_readable");
            sex = jsonResult.getString("sex");
            expiration_date_raw = jsonResult.getString("expiration_date_raw");
            expiration_date_readable = jsonResult.getString("expiration_date_readable");

            JSONArray givenNamesArr = jsonResult.getJSONArray("given_names");
            if (givenNamesArr.length() > 0) {
                given_names = new String[givenNamesArr.length()];

                for (int i = 0; i < givenNamesArr.length(); i++) {
                    given_names[i] = givenNamesArr.getString(i);
                }
            }

            JSONArray surnamesArr = jsonResult.getJSONArray("surnames");
            if (surnamesArr.length() > 0) {
                surnames = new String[surnamesArr.length()];

                for (int i = 0; i < surnamesArr.length(); i++) {
                    surnames[i] = surnamesArr.getString(i);
                }
            }

            JSONArray optionalsArr = jsonResult.getJSONArray("optionals");
            if (optionalsArr.length() > 0) {
                optionals = new String[optionalsArr.length()];

                for (int i = 0; i < optionalsArr.length(); i++) {
                    optionals[i] = optionalsArr.getString(i);
                }
            }

            est_issuing_date_raw = calculateEstIssuingDate(document_type_raw, issuing_country, dob_raw, expiration_date_raw);
            parseRawToReadable();
        } catch (JSONException e) {
            e.printStackTrace();
        }
    }

    public String getFullName() {
        return givenNamesReadable() + " " + surnamesReadable();
    }

    protected void parseRawToReadable() {
        document_type_readable = parseReadableDocumentType(document_type_raw);
        dob_readable = parseDOBToReadable(dob_raw);
        expiration_date_readable = parseExpirationDateToReadable(expiration_date_raw);
        est_issuing_date_readable = parseExpirationDateToReadable(est_issuing_date_raw);
    }

    private String parseDOBToReadable(String dateRaw) {
        if (dateRaw == null || dateRaw.length() != 6) {
            return "Unknown";
        }

        try {
            Calendar calendar = Calendar.getInstance();
            int currentYear = calendar.get(Calendar.YEAR);
            currentYear %= 100;
            int dobYear = Integer.valueOf(dateRaw.substring(0, 2));

            dobYear = dobYear > currentYear ? dobYear + 1900 : dobYear + 2000;

            return parseDateToReadable(dateRaw, dobYear);
        } catch (NumberFormatException ignored) {
        }

        return "Unknown";
    }

    private String parseExpirationDateToReadable(String dateRaw) {
        if (dateRaw == null || dateRaw.length() != 6) {
            return "Unknown";
        }

        try {
            int expYear = Integer.valueOf(dateRaw.substring(0, 2));
            expYear = expYear < 70 ? expYear + 2000 : expYear + 1900;

            return parseDateToReadable(dateRaw, expYear);
        } catch (NumberFormatException ignored) {
        }

        return "Unknown";
    }

    private String parseDateToReadable(String dateRaw, int year) throws NumberFormatException {
        Calendar calendar = Calendar.getInstance();

        calendar.set(year, Integer.valueOf(dateRaw.substring(2, 4)) - 1, Integer.valueOf(dateRaw.substring(4, 6)));
        SimpleDateFormat sdf = new SimpleDateFormat(MRZScanner.getDateFormat(), Locale.getDefault());

        return sdf.format(calendar.getTime());
    }

    public String givenNamesReadable() {
        if (given_names == null) return "";

        StringBuilder fullName = new StringBuilder();

        for (String s : given_names) {
            fullName.append(s).append(" ");
        }

        return fullName.toString().trim();
    }

    public String surnamesReadable() {
        if (surnames == null) return "";

        StringBuilder fullName = new StringBuilder();

        for (String s : surnames) {
            fullName.append(s).append(" ");
        }

        return fullName.toString().trim();
    }

    public String optionalsReadable() {
        StringBuilder optionalsReadable = new StringBuilder();

        if (optionals != null) {
            for (String s : optionals) {
                optionalsReadable.append(s).append(" ");
            }
        }

        return optionalsReadable.toString().trim();
    }

    public JSONObject toJSON() throws JSONException {
        JSONObject jsonObject = new JSONObject();

        jsonObject.put("document_type_raw", document_type_raw);
        jsonObject.put("document_type_readable", document_type_readable);
        jsonObject.put("issuing_country", issuing_country);
        jsonObject.put("surname", surnamesReadable());
        jsonObject.put("document_number", document_number);
        jsonObject.put("nationality", nationality);
        jsonObject.put("dob_raw", dob_raw);
        jsonObject.put("dob_readable", dob_readable);
        jsonObject.put("sex", sex);
        jsonObject.put("est_issuing_date_raw", est_issuing_date_raw);
        jsonObject.put("est_issuing_date_readable", est_issuing_date_readable);
        jsonObject.put("expiration_date_raw", expiration_date_raw);
        jsonObject.put("expiration_date_readable", expiration_date_readable);
        jsonObject.put("given_names_readable", givenNamesReadable());
        jsonObject.put("optionals", optionalsReadable());

        return jsonObject;
    }

    public String toCSV() {
        return document_type_raw + ","
                + document_type_readable + ","
                + issuing_country + ","
                + surnamesReadable() + ","
                + givenNamesReadable() + ","
                + document_number + ","
                + nationality + ","
                + dob_raw + ","
                + dob_readable + ","
                + sex + ","
                + est_issuing_date_raw + ","
                + est_issuing_date_readable + ","
                + expiration_date_raw + ","
                + expiration_date_readable + ","
                + optionalsReadable() + ","
                + dateScanned;
    }

}
