package com.scansolutions.mrzscannerlib;

import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Point;
import android.os.Environment;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;
import android.util.Size;
import android.util.SparseIntArray;
import android.view.Surface;
import android.widget.ImageView;

import com.googlecode.tesseract.android.TessBaseAPI;

import org.opencv.android.Utils;
import org.opencv.core.Core;
import org.opencv.core.CvException;
import org.opencv.core.Mat;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;

import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_ID;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_DOC_IMAGE_PASSPORT;
import static com.scansolutions.mrzscannerlib.ScannerType.SCANNER_TYPE_MRZ;

class MRZCore {

    private static final String TAG = MRZCore.class.getSimpleName();

    static {
        System.loadLibrary("native-lib");
    }

    //TODO: Clean resources on close
    boolean isTorchOn = false;

    MRZCore(MRZOverlay mrzOverlay, ImageView debugPreview, MRZScannerListener scannerListener) {
        this.mrzOverlay = mrzOverlay;
        this.debugPreview = debugPreview;
        this.scannerListener = scannerListener;
    }

    static native int[] mrzLocationMat(long addrInputImage, long addrOutputImage);

    static native long isTextID(String text);

    static native long isTextPassport(String text);

    static native long isTextVisa(String text, boolean validCountryCode);

    static native String parseResultJSON(String text, int visaSubtype);

    static native String sdkVersionNative();

    static native double frameQuality(long addrInputImage);

    static native int registerWithLicenseKey(String key, String bundleID, String modelName);

    static native long validateCountryCode(String cCode);

    static native int faceDetection(long addrInputImage,
                                    long addrFaceOutputImage,
                                    long addrSignatureOutputImage,
                                    String directory,
                                    String countryCode,
                                    int[] squarePoints);

    static native int[] findDocumentImageNative(long addrInputImage, boolean isPassport);

    private MRZScannerListener scannerListener;

    private static final SparseIntArray ORIENTATIONS = new SparseIntArray();

    static {
        ORIENTATIONS.append(Surface.ROTATION_0, 90);
        ORIENTATIONS.append(Surface.ROTATION_90, 0);
        ORIENTATIONS.append(Surface.ROTATION_180, 270);
        ORIENTATIONS.append(Surface.ROTATION_270, 180);
    }

    static boolean isIDActive = false;
    static boolean isPassportActive = false;
    static boolean isVisaActive = false;

    static final int STATE_SCANNING = 0;
    static final int STATE_STOPPED = 1;
    static final int STATE_FINDING_SQUARE = 2;

    private static final int FRAME_Q_THRESHOLD = 50;

    static int scanningState = STATE_STOPPED;
    static ScannerType scannerType = SCANNER_TYPE_MRZ;

    private static final String TESSDATA = "tessdata";
    private static final String DATA_PATH = Environment.getExternalStorageDirectory().toString() + "/TesseractFiles/";

    static int activeThreads = 0;

    static final int AVAILABLE_THREADS = Runtime.getRuntime().availableProcessors();
    static int maxThreads = Math.min(2, AVAILABLE_THREADS);
    private MRZOverlay mrzOverlay;
    private ImageView debugPreview;

    void scan(Mat imageGrab) {
        if (scannerType == SCANNER_TYPE_MRZ) {
            scanForMRZ(imageGrab);
        } else if (isScannerTypeDoc()) {
            scanForDocumentImage(imageGrab);
        }
    }

    private void scanForDocumentImage(Mat imageGrab) {
        int imageWidth = imageGrab.cols();
        int imageHeight = imageGrab.rows();

        int[] locationSquare = findDocumentImageNative(imageGrab.getNativeObjAddr(), scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT);

        mrzOverlay.drawLocationOverlay(locationSquare, imageWidth, imageHeight);

        if (locationSquare.length == 8 && (scanningState == STATE_FINDING_SQUARE || scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT) ) {
            double frameQuality = frameQuality(imageGrab.getNativeObjAddr());

            if (frameQuality > FRAME_Q_THRESHOLD) {
                Log.i(TAG, String.valueOf(frameQuality));
                scanningState = STATE_STOPPED;

                final Bitmap bitmapImg = Bitmap.createBitmap(imageGrab.cols(), imageGrab.rows(), Bitmap.Config.ARGB_8888);
                Utils.matToBitmap(imageGrab, bitmapImg);

                new Handler(Looper.getMainLooper()).post(new Runnable() {
                    @Override
                    public void run() {
                        if (scannerListener != null) {
                            scannerListener.successfulScanWithDocumentImage(bitmapImg);
                        }
                    }
                });
            }
        }
    }

    private void scanForMRZ(Mat imageGrab) {
        Mat outputMat = new Mat();
        int[] mrzLocationSquare = mrzLocationMat(imageGrab.getNativeObjAddr(), outputMat.getNativeObjAddr());

        mrzOverlay.drawLocationOverlay(mrzLocationSquare, imageGrab.cols(), imageGrab.rows());

        if (!outputMat.empty()) {
            double frameQuality = frameQuality(imageGrab.getNativeObjAddr());

            if (frameQuality > FRAME_Q_THRESHOLD) {
                try {
//                    long startTime = System.currentTimeMillis();

                    Bitmap bitmapImg = Bitmap.createBitmap(outputMat.cols(), outputMat.rows(), Bitmap.Config.ARGB_8888);
                    Utils.matToBitmap(outputMat, bitmapImg);
//                    long difference = System.currentTimeMillis() - startTime;
//                    Log.i(TAG, "difference " + difference/1000.0f);


//                    if (BuildConfig.BUILD_TYPE == "debug") {
//                        new Handler(Looper.getMainLooper()).post(new Runnable() {
//                            @Override
//                            public void run() {
////                                    Bitmap bitmapImgA = Bitmap.createBitmap(imageGrab.cols(), imageGrab.rows(), Bitmap.Config.ARGB_8888);
////                                    Utils.matToBitmap(imageGrab, bitmapImgA);
//                                debugPreview.setImageBitmap(finalBitmapImg);
//                            }
//                        });
//                    }

                    String result = extractText(bitmapImg);

                    if (result != null && !result.isEmpty()) {
                        result = result
                                .replaceAll("\n", "")
                                .replaceAll(" ", "");

                        boolean foundSTH = (isIDActive && isTextID(result) > 0) ||
                                (isPassportActive && isTextPassport(result) > 0);

                        boolean isCountryCodeValid = validateCountryCode(result) > 0;
                        int visaSubtype = 0;

                        if (isVisaActive && !foundSTH) {
                            visaSubtype = (int) isTextVisa(result, isCountryCodeValid);
                            foundSTH = visaSubtype != 0;
                        }

                        if (foundSTH && scanningState != STATE_STOPPED) {
                            scanningState = STATE_STOPPED;

                            final String finalResult = parseResultJSON(result, visaSubtype);

                            long dateScanned = System.currentTimeMillis() / 1000;
                            final MRZResultModel resultModel = new MRZResultModel(finalResult, dateScanned);

                            if (resultModel.document_type_raw.startsWith("P")) {
                                ArrayList<Bitmap> images = scanForFaceAndSignature(imageGrab, resultModel.issuing_country, mrzLocationSquare);
                                resultModel.portrait = images.get(0);
                                resultModel.signature = images.get(1);
                            }

                            new Handler(Looper.getMainLooper()).post(new Runnable() {
                                @Override
                                public void run() {
                                    if (scannerListener != null) {
                                        scannerListener.successfulScanWithResult(resultModel);
                                    }
                                }
                            });
                        }
                    }
                } catch (CvException e) {
                    Log.d("Exception", e.getMessage());
                }
            }
        }
    }

    private String extractText(Bitmap bmp) {
        String extractedText = "";

        try {
            TessBaseAPI tessBaseApi = new TessBaseAPI();

            tessBaseApi.init(DATA_PATH, "mrz");
            tessBaseApi.setPageSegMode(TessBaseAPI.PageSegMode.PSM_SINGLE_BLOCK);

            //EXTRA SETTINGS
            tessBaseApi.setVariable(TessBaseAPI.VAR_CHAR_WHITELIST, "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ<");
            tessBaseApi.setVariable("load_system_dawg", "false");
            tessBaseApi.setVariable("load_freq_dawg", "false");

            tessBaseApi.setImage(bmp);

            extractedText = tessBaseApi.getUTF8Text();
            tessBaseApi.end();
        } catch (Exception e) {
            Log.e(TAG, e.getMessage());
        }

        return extractedText;
    }

    private ArrayList<Bitmap> scanForFaceAndSignature(Mat inputImage, String issuingCountry, int[] squre) {
        ArrayList<Bitmap> imagesList = new ArrayList<>(2);
        Mat faceMat = new Mat();
        Mat signatureMat = new Mat();

        faceDetection(inputImage.getNativeObjAddr(), faceMat.getNativeObjAddr(), signatureMat.getNativeObjAddr(), DATA_PATH + TESSDATA, issuingCountry, squre);

        if (!faceMat.empty()) {
            Bitmap faceImg = Bitmap.createBitmap(faceMat.cols(), faceMat.rows(), Bitmap.Config.ARGB_8888);
            Utils.matToBitmap(faceMat, faceImg);
            imagesList.add(0, faceImg);
        } else {
            imagesList.add(0, null);
        }

        if (!signatureMat.empty()) {
            Bitmap signatureImg = Bitmap.createBitmap(signatureMat.cols(), signatureMat.rows(), Bitmap.Config.ARGB_8888);
            Utils.matToBitmap(signatureMat, signatureImg);
            imagesList.add(1, signatureImg);
        } else {
            imagesList.add(1, null);
        }

        return imagesList;
    }

    static void prepareTesseract(Context context) {
        try {
            prepareDirectory(DATA_PATH + TESSDATA);
        } catch (Exception e) {
            e.printStackTrace();
        }

        copyTessDataFiles(context);
    }

    private static void prepareDirectory(String path) {
        File dir = new File(path);
        if (!dir.mkdirs()) {
            Log.e(TAG, "ERROR: Creation of directory " + path + " failed, check does Android Manifest have permission to write to external storage.");
        }
    }

    private static void copyTessDataFiles(Context context) {
        try {
            String fileList[] = context.getAssets().list(TESSDATA);

            for (String fileName : fileList) {
                String pathToDataFile = DATA_PATH + TESSDATA + "/" + fileName;

//                if (!(new File(pathToDataFile)).exists()) {
                InputStream in = context.getAssets().open(TESSDATA + "/" + fileName);

                OutputStream out = new FileOutputStream(pathToDataFile, false);

                byte[] buf = new byte[1024];
                int len;

                while ((len = in.read(buf)) > 0) {
                    out.write(buf, 0, len);
                }

                in.close();
                out.close();
            }
//            }
        } catch (IOException e) {
            Log.e(TAG, "Unable to copy files to tessdata " + e.toString());
        }
    }

    static Point getBestFitSize(Size[] sizes, float screenAR) {
        Size bestFit = null;
        int difference = Integer.MAX_VALUE;

        for (Size s : sizes) {
            int biggerSide = Math.max(s.getWidth(), s.getHeight());
            int currentDiff = Math.abs(biggerSide - 1920);

            if (currentDiff < difference/* && biggerSide != biggestSizeMax*/) {
                difference = currentDiff;
                bestFit = s;
            } else if (currentDiff == difference) {
                float currentSmallAR = Math.max(s.getWidth(), s.getHeight()) / (float) Math.min(s.getWidth(), s.getHeight());
                float bfSmallAR = Math.max(bestFit.getWidth(), bestFit.getHeight()) / (float) Math.min(bestFit.getWidth(), bestFit.getHeight());
                if (Math.abs(currentSmallAR - screenAR) < Math.abs(bfSmallAR - screenAR) /*&& biggerSide != biggestSizeMax*/) {
                    bestFit = s;
                }
            }
        }

        return new Point(bestFit.getWidth(), bestFit.getHeight());
    }

    Mat rotateMat(Mat mat, Context context, int sensorOrientation) {
        int rotation = ((Activity) context).getWindowManager().getDefaultDisplay().getRotation();
        int orientation = getOrientation(rotation, sensorOrientation);
        if (orientation == 180 || orientation == -180) {
            Core.flip(mat, mat, -1);
        } else if (orientation == 90 || orientation == -270) {
            Core.flip(mat.t(), mat, 1);
        } else if (orientation == 270 || orientation == -90) {
            Core.flip(mat.t(), mat, 0);
        }
        return mat;
    }

    /**
     * Retrieves the JPEG orientation from the specified screen rotation.
     *
     * @param rotation The screen rotation.
     * @return The JPEG orientation (one of 0, 90, 270, and 360)
     */
    private int getOrientation(int rotation, int sensorOrientation) {
        // Sensor orientation is 90 for most devices, or 270 for some devices (eg. Nexus 5X)
        // We have to take that into account and rotate JPEG properly.
        // For devices with orientation of 90, we simply return our mapping from ORIENTATIONS.
        // For devices with orientation of 270, we need to rotate the JPEG 180 degrees.

        return (ORIENTATIONS.get(rotation) + sensorOrientation + 270) % 360;
    }

    private static boolean isScannerTypeDoc() {
        return scannerType == SCANNER_TYPE_DOC_IMAGE_ID || scannerType == SCANNER_TYPE_DOC_IMAGE_PASSPORT;
    }

}
