/*
 * Decompiled with CFR 0.152.
 */
package com.sap.cloud.sdk.datamodel.odata.generator;

import com.google.common.collect.Lists;
import com.google.common.collect.Multimap;
import com.sap.cloud.sdk.cloudplatform.util.StringUtils;
import com.sap.cloud.sdk.datamodel.odata.generator.AllowedFunctionsResolver;
import com.sap.cloud.sdk.datamodel.odata.generator.ApiFunction;
import com.sap.cloud.sdk.datamodel.odata.generator.CodeModelClassGenerator;
import com.sap.cloud.sdk.datamodel.odata.generator.DataModelGeneratorConfig;
import com.sap.cloud.sdk.datamodel.odata.generator.DatamodelMetadataGeneratorAdapter;
import com.sap.cloud.sdk.datamodel.odata.generator.EdmService;
import com.sap.cloud.sdk.datamodel.odata.generator.EdmxFile;
import com.sap.cloud.sdk.datamodel.odata.generator.MessageCollector;
import com.sap.cloud.sdk.datamodel.odata.generator.ODataGeneratorReadException;
import com.sap.cloud.sdk.datamodel.odata.generator.ODataGeneratorWriteException;
import com.sap.cloud.sdk.datamodel.odata.generator.Service;
import com.sap.cloud.sdk.datamodel.odata.generator.ServiceDetails;
import com.sap.cloud.sdk.datamodel.odata.generator.ServiceDetailsResolver;
import com.sap.cloud.sdk.datamodel.odata.utility.EdmxValidator;
import com.sap.cloud.sdk.datamodel.odata.utility.ServiceNameMappings;
import io.vavr.CheckedFunction0;
import io.vavr.control.Try;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.olingo.odata2.api.edm.Edm;
import org.apache.olingo.odata2.api.edm.provider.DataServices;
import org.apache.olingo.odata2.api.edm.provider.EdmProvider;
import org.apache.olingo.odata2.core.edm.provider.EdmImplProv;
import org.apache.olingo.odata2.core.edm.provider.EdmxProvider;
import org.slf4j.Logger;
import org.springframework.util.AntPathMatcher;

class ODataToVdmGenerator {
    private static final Logger logger = MessageCollector.getLogger(ODataToVdmGenerator.class);
    private static final String SWAGGER_FILE_EXTENSION_PATTERN = "\\.json$";
    private static final String METADATA_FILE_EXTENSION_PATTERN = "\\.(edmx|xml)$";
    private static final String NAMESPACE_PARENT_PACKAGE_SPACE = "namespaces";
    private static final String SERVICE_PACKAGE_SPACE = "services";
    private static final List<String> EDMX_VERSIONS = Lists.newArrayList((Object[])new String[]{"1.0", "2.0"});
    private static final Charset FILE_ENCODING = StandardCharsets.UTF_8;

    ODataToVdmGenerator() {
    }

    void generate(@Nonnull DataModelGeneratorConfig config) {
        if (!config.getInputDirectory().exists()) {
            throw new ODataGeneratorReadException("The given input directory does not exist: " + this.getCanonicalPath(config.getInputDirectory()));
        }
        Collection<File> inputFiles = this.getInputFiles(config.getInputDirectory());
        Collection<EdmxFile> allEdmxFiles = this.loadServicesFromInput(config, inputFiles);
        if (allEdmxFiles.isEmpty()) {
            logger.warn(String.format("No OData service definitions found in the input directory '%s' - exiting.", this.getCanonicalPath(config.getInputDirectory())));
            return;
        }
        Collection allODataServices = allEdmxFiles.stream().filter(EdmxFile::isSuccessfullyParsed).map(edmxFile -> (Service)edmxFile.getService().get()).collect(Collectors.toCollection(ArrayList::new));
        this.storeConfiguration(config.getServiceNameMappings(), allODataServices);
        int numServices = allODataServices.size();
        logger.info("Processing " + numServices + " OData service" + (numServices == 1 ? "" : "s") + "...");
        CodeModelClassGenerator classGenerator = this.createCodeModelForServices(config, allODataServices);
        for (EdmxFile edmxFile2 : allEdmxFiles) {
            if (!edmxFile2.isSuccessfullyParsed() || !classGenerator.wasServiceGenerated((String)edmxFile2.getServiceName().get())) continue;
            edmxFile2.setSuccessfullyGenerated();
        }
        if (config.isDeleteTargetDirectory() && config.getOutputDirectory().exists()) {
            this.cleanDirectory(config.getOutputDirectory());
        }
        logger.info("Generating Java classes to " + this.getCanonicalPath(config.getOutputDirectory()));
        classGenerator.writeClasses(config, FILE_ENCODING);
        new DatamodelMetadataGeneratorAdapter(logger).generateMetadataIfApplicable(Paths.get(config.getInputDirectory().getAbsolutePath(), new String[0]), Paths.get(config.getOutputDirectory().getAbsolutePath(), new String[0]), allEdmxFiles, classGenerator.getServiceClassGenerator());
    }

    private CodeModelClassGenerator createCodeModelForServices(@Nonnull DataModelGeneratorConfig config, @Nonnull Iterable<Service> allODataServices) {
        String packageName = config.getPackageName();
        String namespaceParentPackageName = this.buildPackageName(packageName, NAMESPACE_PARENT_PACKAGE_SPACE);
        String servicePackageName = this.buildPackageName(packageName, SERVICE_PACKAGE_SPACE);
        CodeModelClassGenerator classGenerator = new CodeModelClassGenerator(config, namespaceParentPackageName, servicePackageName);
        for (Service service : allODataServices) {
            try {
                classGenerator.processService(service, config.getIncludedEntitySets(), config.getIncludedFunctionImports());
            }
            catch (ODataGeneratorReadException e) {
                logger.warn(String.format("Error in file %s; unable to generate all VDM classes.", service.getName()), (Throwable)e);
            }
        }
        return classGenerator;
    }

    private Collection<EdmxFile> loadServicesFromInput(@Nonnull DataModelGeneratorConfig config, @Nonnull Collection<File> inputFiles) {
        LinkedList<EdmxFile> allEdmxFiles = new LinkedList<EdmxFile>();
        ServiceNameMappings serviceNameMappings = this.loadPropertiesConfiguration(config.getServiceNameMappings());
        for (File edmxFile : inputFiles) {
            if (this.excludePatternMatch(config.getExcludeFilePattern(), edmxFile.getName())) {
                logger.info("Skipping EDMX file due to exclusion rule: {}", (Object)edmxFile);
                continue;
            }
            if (!EdmxValidator.isQualified((File)edmxFile, (EdmxValidator.Version)EdmxValidator.Version.V2)) {
                logger.info("Skipping EDMX file due to incompatible OData version: {}", (Object)edmxFile);
                continue;
            }
            String canonicalPath = this.getCanonicalPath(edmxFile);
            String servicePath = FilenameUtils.getFullPath((String)canonicalPath);
            String serviceName = FilenameUtils.getBaseName((String)canonicalPath);
            logger.info(String.format("Loading OData service %s", serviceName));
            File serviceSwaggerFile = this.getSwaggerFile(servicePath, serviceName);
            logger.info(String.format("Reading metadata file: %s", canonicalPath));
            logger.info(String.format("Reading swagger file:  %s", this.getCanonicalPath(serviceSwaggerFile)));
            try {
                Service service = this.buildService(serviceName, serviceNameMappings, config.getDefaultBasePath(), edmxFile, serviceSwaggerFile, config.isGenerateLinksToApiBusinessHub());
                allEdmxFiles.add(new EdmxFile(edmxFile.toPath(), service));
            }
            catch (ODataGeneratorReadException e) {
                logger.warn(String.format("Error in file %s; unable to generate VDM classes. The file will be skipped and generation continues for the next file.", edmxFile.getName()), (Throwable)e);
                allEdmxFiles.add(new EdmxFile(edmxFile.toPath(), null));
            }
        }
        return allEdmxFiles;
    }

    @Nullable
    private File getSwaggerFile(@Nonnull String servicePath, @Nonnull String serviceName) {
        String patternText = "^" + serviceName + SWAGGER_FILE_EXTENSION_PATTERN;
        Pattern pattern = Pattern.compile(patternText, 2);
        File[] files = new File(servicePath).listFiles((dir, fileName) -> pattern.matcher(fileName).matches());
        return files == null || files.length == 0 ? null : files[0];
    }

    @Nonnull
    private Collection<File> getInputFiles(@Nonnull File inputDir) {
        Pattern pattern = Pattern.compile(".*\\.(edmx|xml)$", 2);
        Predicate<Path> edmxPredicate = p -> pattern.matcher(p.getFileName().toString()).matches();
        return ((Stream)Try.of((CheckedFunction0 & Serializable)() -> Files.walk(inputDir.toPath(), new FileVisitOption[0])).getOrElse(Stream::empty)).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).filter(edmxPredicate).map(Path::toFile).collect(Collectors.toList());
    }

    private Service buildService(@Nonnull String serviceName, @Nonnull ServiceNameMappings serviceNameMappings, @Nullable String defaultBasePath, @Nonnull File serviceMetadataFile, @Nullable File serviceSwaggerFile, boolean linkToApiBusinessHub) {
        Edm metadata = this.getMetadata(serviceMetadataFile);
        ServiceDetails serviceDetails = new ServiceDetailsResolver(defaultBasePath, FILE_ENCODING).createServiceDetails(serviceMetadataFile, serviceSwaggerFile);
        Multimap<String, ApiFunction> allowedFunctions = new AllowedFunctionsResolver(FILE_ENCODING).findAllowedFunctions(metadata, serviceSwaggerFile, serviceMetadataFile);
        EdmService newService = new EdmService(serviceName, serviceNameMappings, metadata, serviceDetails, allowedFunctions, linkToApiBusinessHub);
        logger.info(String.format("  Title: %s", newService.getTitle()));
        logger.info(String.format("  Raw URL: %s", newService.getServiceUrl()));
        logger.info(String.format("  Java Package Name: %s", newService.getJavaPackageName()));
        logger.info(String.format("  Java Class Name: %s", newService.getJavaClassName()));
        return newService;
    }

    private Edm getMetadata(File serviceMetadataFile) {
        EdmImplProv edmImplProv;
        block10: {
            InputStream stream = Files.newInputStream(serviceMetadataFile.toPath(), new OpenOption[0]);
            try {
                EdmxProvider provider = new EdmxProvider().parse(stream, false);
                String oDataVersion = this.getODataVersion(provider);
                if (!EDMX_VERSIONS.contains(oDataVersion)) {
                    throw new ODataGeneratorReadException("Metadata file being read is not an OData V2 service " + serviceMetadataFile.getPath());
                }
                edmImplProv = new EdmImplProv((EdmProvider)provider);
                if (stream == null) break block10;
            }
            catch (Throwable throwable) {
                try {
                    if (stream != null) {
                        try {
                            stream.close();
                        }
                        catch (Throwable throwable2) {
                            throwable.addSuppressed(throwable2);
                        }
                    }
                    throw throwable;
                }
                catch (IOException e) {
                    throw new ODataGeneratorReadException(e);
                }
                catch (Exception e) {
                    throw new ODataGeneratorReadException("Parsing of the metadata file failed. Please check that the metadata file is valid and belongs to an OData Version 2.0 service.", e);
                }
            }
            stream.close();
        }
        return edmImplProv;
    }

    private String buildPackageName(String basePackageName, String subPackageName) {
        if (StringUtils.isBlank((String)basePackageName)) {
            return subPackageName;
        }
        return String.format("%s.%s", basePackageName, subPackageName);
    }

    private void cleanDirectory(File outputDir) {
        try {
            FileUtils.cleanDirectory((File)outputDir);
        }
        catch (IOException e) {
            throw new ODataGeneratorWriteException(e);
        }
    }

    private ServiceNameMappings loadPropertiesConfiguration(File serviceMappingsFile) {
        return ServiceNameMappings.load((Path)serviceMappingsFile.toPath());
    }

    private void storeConfiguration(File serviceMappingsFile, Iterable<Service> allODataServices) {
        this.ensureFileExists(serviceMappingsFile);
        ServiceNameMappings mappings = ServiceNameMappings.load((Path)serviceMappingsFile.toPath());
        for (Service oDataService : allODataServices) {
            String javaClassNameKey = oDataService.getName() + ".className";
            mappings.putString(javaClassNameKey, oDataService.getJavaClassName(), new String[]{oDataService.getTitle()});
            String javaPackageNameKey = oDataService.getName() + ".packageName";
            mappings.putString(javaPackageNameKey, oDataService.getJavaPackageName(), new String[0]);
        }
        try {
            mappings.save();
        }
        catch (IOException e) {
            throw new ODataGeneratorWriteException(e);
        }
    }

    private void ensureFileExists(File serviceMappingsFile) {
        if (!serviceMappingsFile.exists()) {
            boolean success;
            if (logger.isInfoEnabled()) {
                logger.info("The service mappings file at '" + this.getCanonicalPath(serviceMappingsFile) + "' does not exist. Creating an empty one.");
            }
            try {
                success = serviceMappingsFile.createNewFile();
            }
            catch (IOException e) {
                throw new ODataGeneratorWriteException(e);
            }
            if (!success) {
                throw new ODataGeneratorWriteException("Could not create service mappings file at '" + this.getCanonicalPath(serviceMappingsFile) + "'");
            }
        }
    }

    private String getCanonicalPath(@Nullable File outputDir) {
        if (outputDir == null) {
            return null;
        }
        try {
            return outputDir.getCanonicalPath();
        }
        catch (IOException e) {
            throw new ODataGeneratorReadException(e);
        }
    }

    private boolean excludePatternMatch(String excludeFilePattern, String serviceMetadataFilename) {
        ArrayList<String> excludeFilePatternEach = new ArrayList<String>(Arrays.asList(excludeFilePattern.split(",")));
        AntPathMatcher antPathMatcher = new AntPathMatcher();
        for (String filePattern : excludeFilePatternEach) {
            if (!antPathMatcher.match(filePattern, serviceMetadataFilename)) continue;
            logger.info(String.format("Excluding metadata file %s, as it matches with the excludes pattern.", serviceMetadataFilename));
            return true;
        }
        return false;
    }

    private String getODataVersion(EdmxProvider edmxProvider) throws IllegalAccessException, NoSuchFieldException {
        Field field = EdmxProvider.class.getDeclaredField("dataServices");
        field.setAccessible(true);
        DataServices dataServices = (DataServices)field.get(edmxProvider);
        return dataServices.getDataServiceVersion();
    }
}

