package com.samebug.notifier;

import java.io.PrintStream;
import java.util.UUID;

import com.samebug.notifier.core.IConfiguration;
import com.samebug.notifier.core.IResponse;
import com.samebug.notifier.core.Solution;
import com.samebug.notifier.core.exceptions.NotifierException;
import com.samebug.notifier.core.exceptions.RecorderError;

/**
 * Presenting the notifier response to the developer.
 * 
 * @author poroszd
 * 
 */
public class StreamReporter implements IReporter {
    private final IConfiguration config;
    private final PrintStream info;
    private final PrintStream error;

    public StreamReporter(final IConfiguration config) {
        this(config, System.err, System.err);
    }

    public StreamReporter(final IConfiguration config, final PrintStream info, final PrintStream error) {
        this.config = config;
        this.info = info;
        this.error = error;
    }

    public IResponse handleResponse(final IResponse response) {
        switch (config.getHelpMode()) {
        case 1:
            info.println("You can inspect your crash at " + response.getBugURL());
            break;
        case 2:
            UUID solId = response.getSolutionId();
            if (solId == null) {
                info.println("Samebug logged your crash. Check it at " + response.getBugURL());
            } else {
                Solution sol = response.getSolution();
                if (sol == null) {
                    info.println("Ask for help related to your crash at " + response.getSolutionURL());
                } else {
                    info.println("@" + sol.getAuthor() + " says:\n" + sol.getComment() + "\nSee more at " + response.getSolutionURL());
                }
            }
            break;
        }
        return response;
    }

    public RecorderError handleRecorderError(final RecorderError re) {
        error.format("Failed to record an error to Samebug. Server responsed %d:\n%s\n", re.getCode(), re.getMessage());
        if (config.getDebug()) {
            re.printStackTrace(error);
        }
        return re;
    }

    public NotifierException handleNotifierException(final NotifierException e) {
        error.println("Error in samebug notifier (code " + e.getErrorCode() + "): " + e.getMessage());
        if (config.getDebug()) {
            e.printStackTrace(error);
        }
        return e;
    }
}
