package com.samebug.notifier;

import java.lang.Thread.UncaughtExceptionHandler;
import java.util.Date;

import com.samebug.notifier.core.IConfiguration;
import com.samebug.notifier.core.INotifier;
import com.samebug.notifier.core.IResponse;
import com.samebug.notifier.core.SamebugNotifier;
import com.samebug.notifier.core.exceptions.BadConfigFile;
import com.samebug.notifier.core.exceptions.BadHelpMode;
import com.samebug.notifier.core.exceptions.BadServerAddress;
import com.samebug.notifier.core.exceptions.MultipleConfigFileException;
import com.samebug.notifier.core.exceptions.NoConfigFileException;
import com.samebug.notifier.core.exceptions.NotifierException;
import com.samebug.notifier.core.proxy.ThrowableProxy;

/**
 * Static facilities of Samebug.
 * 
 * @author poroszd
 * 
 */
public class Samebug {
    private static boolean initialized = false;
    private static INotifier notifier;

    /**
     * Install a default exception handler on the current thread.
     * <p>
     * In case of any error (properties file not found, multiple file found,
     * wrong format of properties, missing obligatory entries) the constructed
     * notifier will not function as intended.
     */
    public static void handleUncaughtExceptions() throws NoConfigFileException, MultipleConfigFileException, BadConfigFile, BadHelpMode, BadServerAddress {
        handleUncaughtExceptions(ConfigurationFactory.fromDefault());
    }

    /**
     * Install a default exception handler on the current thread, with the given
     * application key.
     */
    public static void handleUncaughtExceptions(final String applicationKey) throws BadHelpMode {
        handleUncaughtExceptions(ConfigurationFactory.fromAppKey(applicationKey));
    }

    /**
     * Install a default exception handler on the current thread, with the given
     * config
     */
    public static void handleUncaughtExceptions(final IConfiguration config) {
        UncaughtExceptionHandler handler = new SamebugUncaughtExceptionHandler(config, getCurrentHandler());
        Thread.setDefaultUncaughtExceptionHandler(handler);
    }

    private static UncaughtExceptionHandler getCurrentHandler() {
        UncaughtExceptionHandler handler = Thread.getDefaultUncaughtExceptionHandler();
        if (handler == null) return new DefaultUncaughtExceptionHandler();
        else return handler;
    }

    /**
     * Initialize a singleton notifier instance.
     * <p>
     * Constructs a singleton Samebug Notifier instance using the default
     * configuration.
     * <p>
     * You only need to use this initialization method if you want to use
     * {Samebug{@link #notify(String, Throwable)} explicitly. Only the first
     * call has effect, every consecutive call is practically ignored.
     * <p>
     * Using explicit notifications is discouraged, one should rather use
     * uncaught exception handler or logger appenders.
     */
    public static synchronized void init() throws NoConfigFileException, MultipleConfigFileException, BadConfigFile, BadHelpMode, BadServerAddress {
        if (!initialized) {
            init(ConfigurationFactory.fromDefault());
        }
    }

    /**
     * Initialize a singleton notifier instance.
     * <p>
     * Constructs a singleton Samebug Notifier instance with the given
     * application key.
     * <p>
     * You only need to use this initialization method if you want to use
     * {Samebug{@link #notify(String, Throwable)} explicitly. Only the first
     * call has effect, every consecutive call is practically ignored.
     * <p>
     * Using explicit notifications is discouraged, one should rather use
     * uncaught exception handler or logger appenders.
     */
    public static synchronized void init(final String applicationKey) throws BadHelpMode {
        if (!initialized) {
            init(ConfigurationFactory.fromAppKey(applicationKey));
        }
    }

    /**
     * Initialize a singleton notifier instance.
     * <p>
     * Constructs a singleton Samebug Notifier instance with the given config.
     * <p>
     * You only need to use this initialization method if you want to use
     * {Samebug{@link #notify(String, Throwable)} explicitly. Only the first
     * call has effect, every consecutive call is practically ignored.
     * <p>
     * Using explicit notifications is discouraged, one should rather use
     * uncaught exception handler or logger appenders.
     */
    public static synchronized void init(final IConfiguration config) {
        if (!initialized) {
            notifier = new SamebugNotifier(config);
            initialized = true;
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwable
     *            the caught exception
     * @return the response of Samebug
     * @throws NotifierException
     */
    public static IResponse notify(final String message, final Throwable throwable) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwable);
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwable
     *            the caught exception
     * @param timestamp
     *            the time of occurrence
     * @return the response of Samebug
     * @throws NotifierException
     */
    public static IResponse notify(final String message, final Throwable throwable, final Date timestamp) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwable, timestamp);
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwableProxy
     *            proxy for the caught exception
     * @return the response of Samebug
     * @throws NotifierException
     */
    public static IResponse notify(final String message, final ThrowableProxy throwableProxy) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwableProxy);
        }
    }

    /**
     * Notify Samebug about an exception
     * 
     * @param message
     *            custom message describing the event
     * @param throwableProxy
     *            proxy for the caught exception
     * @param timestamp
     *            the time of occurrence
     * @return the response of Samebug
     * @throws NotifierException
     */
    public static IResponse notify(final String message, final ThrowableProxy throwableProxy, final Date timestamp) throws NotifierException {
        if (!initialized) {
            throw new IllegalStateException("Samebug not initialized");
        } else {
            return notifier.notify(message, throwableProxy, timestamp);
        }
    }
}
