/*
 * Created on 2013.09.02.
 * 
 * Copyright 2013 progos.hu All rights reserved. PROGOS
 * PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * Author: arpadtamasi
 * $URL$
 * $Rev$
 * $Author$
 * $Date$
 * $Id$
 *
 */

package com.samebug.notifier.encoder;

import java.io.IOException;
import java.io.Writer;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;

import com.samebug.notifier.exceptions.JsonEncodingException;
import com.samebug.notifier.proxy.StackTraceProxy;
import com.samebug.notifier.proxy.ThrowableProxy;

public class ContentEncoder {
    private final Writer writer;

    private static final DateFormat F_XSD_DATE = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ssZ");

    private boolean first;

    public ContentEncoder(final Writer writer) {
        this.writer = writer;
        this.first = true;
    }

    public void startObject() throws JsonEncodingException {
        write("{");
    }

    public void endObject() throws JsonEncodingException {
        write("}");
    }

    public void writeField(final String name, final String value) throws JsonEncodingException {
        if (value == null) {
            return;
        }
        nextField();
        fieldName(name);
        fieldValue(value);
    }

    public void writeField(final String name, final Date value) throws JsonEncodingException {
        if (value == null) {
            return;
        }
        nextField();
        fieldName(name);
        fieldValue(F_XSD_DATE.format(value));
    }

    public void writeField(final String name, final ThrowableProxy value) throws JsonEncodingException {
        if (value == null) {
            return;
        }
        nextField();
        fieldName(name);
        fieldValue(value);
    }

    public void writeField(final String name, final int value) throws JsonEncodingException {
        nextField();
        fieldName(name);
        encode(value);
    }

    private void fieldValue(final String value) throws JsonEncodingException {
        write('"');
        encode(value);
        write('"');
    }

    private void fieldValue(final ThrowableProxy value) throws JsonEncodingException {
        new ContentEncoder(this.writer).encode(value);
    }

    private void encode(final int value) throws JsonEncodingException {
        write(value);
    }

    public void encode(final ThrowableProxy t) throws JsonEncodingException {
        encode(t, 0);
    }

    private void fieldName(final String name) throws JsonEncodingException {
        write('"');
        encode(name);
        write('"');
        write(':');
    }

    private void nextField() throws JsonEncodingException {
        if (this.first) {
            this.first = false;
        } else {
            write(',');
        }
    }

    private void encode(final String s) throws JsonEncodingException {
        final int len = s.length();
        for (int i = 0; i < len; i++) {
            final char ch = s.charAt(i);
            switch (ch) {
            case '"':
                write("\\\"");
                break;
            case '\\':
                write("\\\\");
                break;
            case '\b':
                write("\\b");
                break;
            case '\f':
                write("\\f");
                break;
            case '\n':
                write("\\n");
                break;
            case '\r':
                write("\\r");
                break;
            case '\t':
                write("\\t");
                break;
            case '/':
                write("\\/");
                break;
            default:
                // Reference: http://www.unicode.org/versions/Unicode5.1.0/
                if (ch >= '\u0000' && ch <= '\u001F' || ch >= '\u007F' && ch <= '\u009F' || ch >= '\u2000' && ch <= '\u20FF') {
                    final String ss = Integer.toHexString(ch);
                    write("\\u");
                    for (int k = 0; k < 4 - ss.length(); k++) {
                        write('0');
                    }
                    write(ss.toUpperCase());
                } else {
                    write(ch);
                }
            }
        }
    }

    private void encode(final ThrowableProxy throwable, final int commonFrames) throws JsonEncodingException {
        write('{');
        writeField("class", throwable.getClassName());
        if (throwable.getMessage() != null) {
            writeField("message", throwable.getMessage());
        }

        if (commonFrames > 0) {
            writeField("more", commonFrames);
        }
        nextField();
        fieldName("trace");
        encode(throwable.getStackTrace(), commonFrames);

        final ThrowableProxy cause = throwable.getCause();
        if (cause != null) {
            nextField();
            fieldName("cause");
            new ContentEncoder(this.writer).encode(cause, throwable.getCommonFramesWithCause());
        }
        write('}');
    }

    private void encode(final StackTraceProxy trace, final int commonFrames) throws JsonEncodingException {
        write('"');
        final String[] frames = trace.getFrames();
        final int m = frames.length - 1 - commonFrames;
        for (int i = 0; i < m; i++) {
            encode(frames[i]);
            write("\\n");
        }
        encode(frames[m]);
        write('"');
    }

    private void write(final char ch) throws JsonEncodingException {
        try {
            this.writer.write(ch);
        } catch (final IOException e) {
            throw new JsonEncodingException("Unable to write char " + ch, e);
        }
    }

    private void write(final int i) throws JsonEncodingException {
        write(String.valueOf(i));
    }

    private void write(final String s) throws JsonEncodingException {
        try {
            this.writer.write(s);
        } catch (final IOException e) {
            throw new JsonEncodingException("Unable to write string " + s, e);
        }
    }
}
