/*
 * Created on 2013.09.02.
 * 
 * Copyright 2013 progos.hu All rights reserved. SAMEBUG
 * PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * Author: rp
 */

package com.samebug.notifier;

import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.util.Date;
import java.util.UUID;

import com.samebug.notifier.exceptions.BadAppKey;
import com.samebug.notifier.exceptions.BadServerAddress;
import com.samebug.notifier.exceptions.JsonEncodingException;
import com.samebug.notifier.exceptions.MultipleConfigFileException;
import com.samebug.notifier.exceptions.NoConfigFileException;
import com.samebug.notifier.exceptions.NotifierException;
import com.samebug.notifier.exceptions.WriteError;
import com.samebug.notifier.proxy.DefaultThrowableProxy;
import com.samebug.notifier.proxy.ThrowableProxy;

/**
 * The Samebug Notifier.
 * <p>
 * The Samebug Notifier class lets you to instantiate a notifier with custom configuration
 * without the need of configuration files. Also, it makes explicit error sending possible.
 * 
 * @author poroszd
 *
 */
public class SamebugNotifier implements INotifier {
    private final Configuration config;
    private final Connection connection;

    /**
     * Create a notifier with the default configuration 
     */
    public SamebugNotifier() throws NoConfigFileException, MultipleConfigFileException, BadAppKey, BadServerAddress {
        this.config = Configuration.fromProperties();
        if (config.getDebug()) {
            System.err.println("Notifier initiated with this config:\n" + config);
        }
        this.connection = new Connection(this.config);
    }

    /**
     * Create a notifier with only the application key set.
     */
    public SamebugNotifier(String appKey) throws BadAppKey {
        config = new Configuration(appKey);
        if (config.getDebug()) {
            System.err.println("Notifier initiated with this config:\n" + config);
        }
        connection = new Connection(config);
    }

    /**
     * Create a notifier with the specified configuration.
     */
    public SamebugNotifier(Configuration config) {
        this.config = config;
        if (config.getDebug()) {
            System.err.println("Notifier initiated with this config:\n" + config);
        }
        connection = new Connection(config);
    }
    
    public Configuration getConfiguration() {
        return config;
    }

    public UUID notify(final String message, final Throwable throwable) throws NotifierException {
        return notify(message, throwable, new Date());
    }

    public UUID notify(final String message, final ThrowableProxy throwable) throws NotifierException {
        return notify(message, throwable, new Date());
    }

    public UUID notify(final String message, final Throwable throwable, final Date timestamp) throws NotifierException {
        return notify(message, new DefaultThrowableProxy(throwable), timestamp);
    }

    public UUID notify(final String message, final ThrowableProxy throwable, final Date timestamp) throws NotifierException {
        if (config.getDebug()) {
            System.err.println("Attempt to send notification");
        }
        final HttpURLConnection http = connection.createConnection();
        sendReport(message, throwable, timestamp, http);
        return connection.processResponse(http);
    }

    private void sendReport(final String message, final ThrowableProxy throwable, final Date timestamp, final HttpURLConnection connnection) throws WriteError, JsonEncodingException {
        OutputStreamWriter writer = null;
        try {
            writer = new OutputStreamWriter(connnection.getOutputStream());
            Encoder encoder = new Encoder(config, writer);
            encoder.encode(message, throwable, timestamp);
            writer.flush();
        } catch (final IOException e) {
            throw new WriteError("Unable send error notification to " + config.getServerURL(), e);
        } finally {
            if (writer != null) {
                try {
                    writer.close();
                } catch (final IOException e) {
                    if (config.getDebug()) {
                        System.err.println("Failed to close output stream:\n");
                        e.printStackTrace(System.err);
                    }
                }
            }
        }
    }
}
