package com.samebug.notifier;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Enumeration;
import java.util.Properties;
import java.util.UUID;

import com.samebug.notifier.exceptions.BadAppKey;
import com.samebug.notifier.exceptions.BadConfigFile;
import com.samebug.notifier.exceptions.BadServerAddress;
import com.samebug.notifier.exceptions.MultipleConfigFileException;
import com.samebug.notifier.exceptions.NoConfigFileException;

/**
 * Configuration parameters of samebug notifier.
 * <p>
 * It contains the following parameters:
 * <ul>
 * <li>appKey: the application key, mandatory</li>
 * <li>version: application version, default is <code>null</code></li>
 * <li>serverURL: url of the recorder, default is
 * <code>http://recorder.samebug.io</code></li>
 * </ul>
 * 
 * @author poroszd
 * 
 */
public class Configuration {
    public static final String DEFAULT_PROPERTY_RESOURCE = "samebug.properties";
    public static final String DEFAULT_SERVER = "http://recorder.samebug.io";
    public static final String DEFAULT_DEBUG = "false";

    private UUID appKey;
    private String version;
    private URL serverURL;
    private boolean debug;

    /**
     * Construct a configuration with the specified application key.
     * <p>
     * The other parameters are left on default.
     * 
     * @param appKey
     *            the key of your registered Samebug application, as found on
     *            the documentation site of your app. Must not be
     *            <code>null</code>
     * @throws BadAppKey
     *             when appKey is not parseable as a UUID
     */
    public Configuration(final String appKey) throws BadAppKey {
        setAppKey(appKey);
    }

    private Configuration(final String appKey, final String version, final String serverAddress, final String debug) throws BadAppKey, BadServerAddress {
        setAppKey(appKey);
        setVersion(version);
        setServer(serverAddress);
        setDebug(debug);
    }

    private void setAppKey(final String appKey) throws BadAppKey {
        if (appKey == null) throw new BadAppKey("Application key is missing!");
        try {
            this.appKey = UUID.fromString(appKey);
        } catch (final IllegalArgumentException e) {
            throw new BadAppKey(appKey + " is not parsable as application key");
        }
        assert (this.appKey != null);
    }

    /**
     * Set your application version.
     * <p>
     * <code>null</code> means an unspecified version.
     * 
     * @param version
     *            arbitrary string
     */
    public void setVersion(final String version) {
        this.version = version;
    }

    /**
     * Set your samebug API endpoint.
     * 
     * @param serverAddress
     *            an URL string of a samebug server
     * 
     * @throws BadServerAddress
     *             when serverAddress is not a valid URL
     */
    public void setServer(final String serverAddress) throws BadServerAddress {
        if (serverAddress == null) throw new BadServerAddress("Server address must not be null.");
        try {
            this.serverURL = new URI(serverAddress).toURL();
        } catch (final URISyntaxException e) {
            throw new BadServerAddress(serverAddress + " is not a valid URI");
        } catch (final MalformedURLException e) {
            throw new BadServerAddress(serverAddress + " is not a valid URL");
        }
        assert (this.serverURL != null);
    }

    /**
     * Set if samebug notifier should print debug messages.
     * 
     * @param debug
     *            "false" disables debug messages, any else string enables it.
     */
    public void setDebug(final String debug) {
        if (debug == "false") {
            this.debug = false;
        } else {
            this.debug = true;
        }
    }

    public UUID getAppKey() {
        return appKey;
    }

    public String getVersion() {
        return version;
    }

    public URL getServerURL() {
        return serverURL;
    }

    public boolean getDebug() {
        return debug;
    }

    /**
     * Factory method of the default configuration.
     * <p>
     * This method will build a configuration based on values taken from many
     * location. Each value is assigned in this order:
     * <ol>
     * <li>command line property parameter (e.g.
     * -Dsamebug.key=0092029d-6f46-46a8-a65c-3e2dc22e5f0a)</li>
     * <li>content of the file or resource specified on command line
     * (-Dsamebug.config.file=/home/poroszd/my_app/samebug.properties or
     * -Dsamebug.config.resource=my_app.properties)</li>
     * <li>content of the default samebug property resource (i.e.
     * samebug.properties in the classpath)</li>
     * <li>the default value of the property, if not mandatory</li>
     * 
     * @return a configuration
     */
    public static Configuration fromProperties() throws NoConfigFileException, MultipleConfigFileException, BadAppKey, BadServerAddress {
        int specified = 0;
        String resource = System.getProperty("samebug.config.resource");
        if (resource != null) specified += 1;
        String file = System.getProperty("samebug.config.file");
        if (file != null) specified += 1;
        if (specified == 0) return fromDefault();
        else if (specified > 1) throw new MultipleConfigFileException("You specified more than one property file for samebug:\nconfig.file=" + file + "\nconfig.resource=" + resource);
        else {
            if (resource != null) return fromPropertyResource(resource);
            else return fromPropertyFile(file);
        }
    }

    /**
     * Factory method to create a configuration with the default values.
     */
    public static Configuration fromDefault() throws NoConfigFileException, MultipleConfigFileException, BadAppKey, BadServerAddress {
        try {
            // try if samebug.properties exists and unique
            openResource(DEFAULT_PROPERTY_RESOURCE);
            return fromPropertyResource(DEFAULT_PROPERTY_RESOURCE);
        } catch (final Exception e) {
            // In this case, we still try to create a configuration from command
            // line properties
            return fromProps(new Properties());
        }
    }

    /**
     * Factory method to create a configuration based on the content of a
     * resource property file.
     * 
     * @param resourceName
     *            the name of the resource
     * @return a configuration
     */
    public static Configuration fromPropertyResource(final String resourceName) throws NoConfigFileException, MultipleConfigFileException, BadServerAddress, BadAppKey {
        Properties props = new Properties();
        InputStream propFile = null;
        try {
            propFile = openResource(resourceName);
        } catch (final IOException e) {
            throw new NoConfigFileException();
        }
        try {
            props.load(propFile);
        } catch (final IOException e) {
            throw new BadConfigFile();
        }
        return fromProps(props);
    }

    /**
     * Factory method to create a configuration based on the content of a
     * property file, given with its absolute path.
     * 
     * @param fileName
     *            the name of the file
     * @return a configuration
     */
    public static Configuration fromPropertyFile(final String fileName) throws NoConfigFileException, MultipleConfigFileException, BadServerAddress, BadAppKey {
        Properties props = new Properties();
        InputStream propFile;
        try {
            propFile = new FileInputStream(fileName);
        } catch (FileNotFoundException e1) {
            throw new NoConfigFileException();
        }
        try {
            props.load(propFile);
        } catch (final IOException e) {
            throw new BadConfigFile();
        }
        return fromProps(props);
    }

    private static Configuration fromProps(final Properties props) {
        props.putAll(System.getProperties());

        String appKey = props.getProperty("samebug.key");
        String version = props.getProperty("samebug.version");
        String serverAddress = props.getProperty("samebug.server", DEFAULT_SERVER);
        String debug = props.getProperty("samebug.debug", DEFAULT_DEBUG);

        return new Configuration(appKey, version, serverAddress, debug);
    }

    protected static InputStream openResource(final String resourceName) throws NoConfigFileException, MultipleConfigFileException, IOException {
        Enumeration<URL> urls = Configuration.class.getClassLoader().getResources(resourceName);
        if (!urls.hasMoreElements()) throw new NoConfigFileException("No configuration resource found with name " + resourceName);
        URL onlyConfig = urls.nextElement();
        if (urls.hasMoreElements()) throw new MultipleConfigFileException("Multiple configuration resource found in the classpath with name " + resourceName);
        return onlyConfig.openStream();
    }

    public String toString() {
        return "samebug.key = " + appKey + "\n" + "samebug.version = " + version + "\n" + "samebug.server = " + serverURL + "\n" + "samebug.debug = " + debug;
    }
}
