/*
 * Created on 2013.09.02.
 * 
 * Copyright 2013 progos.hu All rights reserved. SAMEBUG
 * PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 *
 * Author: rp
 */

package com.samebug.notifier;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.HttpURLConnection;
import java.net.SocketTimeoutException;
import java.util.Date;
import java.util.UUID;

import com.samebug.notifier.encoder.ContentEncoder;
import com.samebug.notifier.exceptions.ConnectionError;
import com.samebug.notifier.exceptions.ConnectionTimeout;
import com.samebug.notifier.exceptions.JsonEncodingException;
import com.samebug.notifier.exceptions.NoAppKeyDefinedException;
import com.samebug.notifier.exceptions.NoResponse;
import com.samebug.notifier.exceptions.NoURI;
import com.samebug.notifier.exceptions.NotifierException;
import com.samebug.notifier.exceptions.ReadError;
import com.samebug.notifier.exceptions.RecorderError;
import com.samebug.notifier.exceptions.UnknownProtocol;
import com.samebug.notifier.exceptions.UnsupportedUTF8;
import com.samebug.notifier.exceptions.UrlEncodingError;
import com.samebug.notifier.exceptions.WriteError;
import com.samebug.notifier.proxy.DefaultThrowableProxy;
import com.samebug.notifier.proxy.ThrowableProxy;

class SamebugNotifier extends AbstractNotifier {

	private static final int SC_INTERNAL_SERVER_ERROR = 500;

	private static final int SC_UNAUTHORIZED = 401;

	/**
	 * Create a new SamebugNotifier instance, configured by the property file
	 * named 'samebug.properties' in the classpath.
	 * 
	 * In case of any error (properties file not found, multiple file found,
	 * wrong format of properties, missing obligatory entries) the constructed
	 * notifier will not function as intended, but can be fix by manually
	 * setting the parameters.
	 * 
	 */
	public SamebugNotifier() {
		super();
	}

	/**
	 * Create a new SamebugNotifier with the given application key.
	 * 
	 * @param applicationKey the key associated with your application in Samebug
	 */
	public SamebugNotifier(final String applicationKey) {
		super(applicationKey);
	}

	public UUID notify(final String message, final Throwable throwable) throws NotifierException {
		return notify(message, throwable, new Date());
	}

	public UUID notify(final String message, final ThrowableProxy throwable) throws NotifierException {
		return notify(message, throwable, new Date());
	}

	public UUID notify(final String message, final Throwable throwable, final Date timestamp) throws NotifierException {
		return notify(message, new DefaultThrowableProxy(throwable), timestamp);
	}

	public UUID notify(final String message, final ThrowableProxy throwable, final Date timestamp) throws NotifierException {
		// Open conn
		final HttpURLConnection connection = connect();

		// Send report
		sendReport(message, throwable, timestamp, connection);

		// Process response
		return processResponse(connection);
	}

	protected HttpURLConnection connect() throws UnknownProtocol, ConnectionTimeout, ConnectionError, UnsupportedUTF8, UrlEncodingError, NoAppKeyDefinedException {
		if (key == null) {
			throw new NoAppKeyDefinedException("Application key is not set in Samebug Notifier");
		}
		final HttpURLConnection connection = createConnection();
		try {
			connection.setConnectTimeout(connectionTimeout);
			connection.setReadTimeout(readTimeout);
			connection.setRequestProperty("Content-Type", "application/json;charset=utf-8");
			connection.setRequestProperty("charset", "utf-8");
			connection.setRequestMethod("POST");
			connection.setRequestProperty("Accept", "text/plain");
			connection.setRequestProperty("X-Samebug-Key", key);
			connection.setRequestProperty("X-Samebug-Platform", SAMEBUG_PLATFORM);
			connection.setRequestProperty("X-Samebug-Version", SAMEBUG_VERSION);
			connection.setDoOutput(true);
			connection.connect();
		} catch (final SocketTimeoutException e) {
			throw new ConnectionTimeout("Cannot connect to " + this.server, e);
		} catch (final IOException e) {
			throw new ConnectionError("IO error while connecting to " + this.server, e);
		}
		return connection;
	}

	private void serialize(final String message, final ThrowableProxy throwable, final Date timestamp, final Writer writer) throws JsonEncodingException {
		final ContentEncoder e = new ContentEncoder(writer);
		e.startObject();
		e.writeField("version", this.version);
		e.writeField("timestamp", timestamp);
		e.writeField("message", message);
		e.writeField("exception", throwable);
		e.endObject();
	}

	protected UUID processResponse(final HttpURLConnection conn) throws RecorderError, NoResponse, NoURI, ReadError {
		// Get response code
		try {
			final int rc = conn.getResponseCode();
			final String rm = conn.getResponseMessage();
			switch (rc) {
			case SC_INTERNAL_SERVER_ERROR:
			case SC_UNAUTHORIZED:
			default:
				if (rc < 200 || rc >= 300) {
					throw new RecorderError(rc, rm);
				}
			}
		} catch (final IOException e) {
			throw new NoResponse("Unable to process response from " + this.server, e);
		}

		// Read response
		BufferedReader rd = null;
		try {
			rd = new BufferedReader(new InputStreamReader(conn.getInputStream()));
			String line;
			if ((line = rd.readLine()) != null) {
				return UUID.fromString(line);
			}
			throw new NoURI("No report URI.");
		} catch (final SocketTimeoutException e) {
			throw new ReadError("Unable to process response from " + this.server, e);
		} catch (final IOException e) {
			throw new ReadError("Unable to process response from " + this.server, e);
		} finally {
			if (rd != null) {
				try {
					rd.close();
				} catch (final IOException e) {
					// SKIP
				}
			}
		}
	}

	private void sendReport(final String message, final ThrowableProxy throwable, final Date timestamp, final HttpURLConnection connnection) throws WriteError, JsonEncodingException {
		OutputStreamWriter writer = null;
		try {
			writer = new OutputStreamWriter(connnection.getOutputStream());
			serialize(message, throwable, timestamp, writer);
			writer.flush();
		} catch (final IOException e) {
			throw new WriteError("Unable send error notification to " + this.server, e);
		} finally {
			if (writer != null) {
				try {
					writer.close();
				} catch (final IOException e) {
					// SKIP
				}
			}
		}
	}
}
