/*
 * Copyright (C) 2013 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.runningmessage.kotlinx.widget

import android.content.Context
import android.support.annotation.ColorInt
import android.support.annotation.ColorRes
import android.support.v4.content.ContextCompat
import android.support.v4.view.*
import android.util.AttributeSet
import android.util.Log
import android.view.MotionEvent
import android.view.View
import android.view.ViewConfiguration
import android.view.ViewGroup
import android.view.animation.Animation
import android.view.animation.Animation.AnimationListener
import android.view.animation.DecelerateInterpolator
import android.view.animation.Transformation
import android.widget.AbsListView
import android.widget.ListView

/**
 *
 *     1. support pull down to refresh and enable to custom pull refresh anim
 *     2. support show remind pop view after pull refresh
 *
 * the blew links to google source:
 *
 *
 * The SwipeRefreshLayout should be used whenever the user can refresh the
 * contents of a view via a vertical swipe gesture. The activity that
 * instantiates this view should add an OnRefreshListener to be notified
 * whenever the swipe to refresh gesture is completed. The SwipeRefreshLayout
 * will notify the listener each and every time the gesture is completed again;
 * the listener is responsible for correctly determining when to actually
 * initiate a refresh of its content. If the listener determines there should
 * not be a refresh, it must call setRefreshing(false) to cancel any visual
 * indication of a refresh. If an activity wishes to show just the progress
 * animation, it should call setRefreshing(true). To disable the gesture and
 * progress animation, call setEnabled(false) on the view.
 *
 *
 * This layout should be made the parent of the view that will be refreshed as a
 * result of the gesture and can only support one direct child. This view will
 * also be made the target of the gesture and will be forced to match both the
 * width and the height supplied in this layout. The SwipeRefreshLayout does not
 * provide accessibility events; instead, a menu item must be provided to allow
 * refresh of the content wherever this gesture is used.
 *
 */
abstract class AbsSwipeRefreshLayout<ProgressView, RemindView>
/**
 * Constructor that is called when inflating SwipeRefreshLayout from XML.
 *
 * @param context
 * @param attrs
 */
@JvmOverloads constructor(context: Context, attrs: AttributeSet? = null) :
        ViewGroup(context, attrs), NestedScrollingParent, NestedScrollingChild
        where ProgressView : View,
              ProgressView : SwipeRefreshProgress,
              RemindView : View,
              RemindView : SwipeRefreshRemind {

    private var mTarget: View? = null // the target of the gesture
    internal var mListener: OnRefreshListener? = null
    internal var mRefreshing = false
    private val mTouchSlop: Int = ViewConfiguration.get(context).scaledTouchSlop
    private var mTotalDragDistance = -1f

    // If nested scrolling is enabled, the total amount that needed to be
    // consumed by this as the nested scrolling parent is used in place of the
    // overscroll determined by MOVE events in the onTouch handler
    private var mTotalUnconsumed: Float = 0.toFloat()
    private val mNestedScrollingParentHelper: NestedScrollingParentHelper
    private val mNestedScrollingChildHelper: NestedScrollingChildHelper
    private val mParentScrollConsumed = IntArray(2)
    private val mParentOffsetInWindow = IntArray(2)
    private var mNestedScrollInProgress: Boolean = false

    internal var mCurrentTargetOffsetTop: Int = 0
    private var mCurrentRemindOffsetTop: Int? = null

    private var mInitialMotionY: Float = 0.toFloat()
    private var mInitialDownY: Float = 0.toFloat()
    private var mIsBeingDragged: Boolean = false
    private var mActivePointerId = INVALID_POINTER
    // Whether this item is scaled up rather than clipped
    internal var mScale: Boolean = false
    // Whether the target pull follow spinner when pulling
    private var mTargetPull: Boolean = false

    // Target is returning to its start offset because it was cancelled or a
    // refresh was triggered.
    private var mReturningToStart: Boolean = false
    private val mDecelerateInterpolator: DecelerateInterpolator

    internal lateinit var mProgressView: ProgressView
    private var mProgressViewIndex = -1

    protected var mFrom: Int = 0

    internal var mStartingScale: Float = 0.toFloat()

    /**
     * @return The offset in pixels from the top of this view at which the progress spinner should
     * appear.
     */
    var progressViewStartOffset: Int = 0
        protected set

    /**
     * @return The offset in pixels from the top of this view at which the progress spinner should
     * come to rest after a successful swipe gesture.
     */
    var progressViewEndOffset: Int = 0
        internal set

    var remindViewStartOffset: Int = 0
    var remindViewEndOffset: Int = 0

    private var mScaleDownAnimation: Animation? = null

    private var mScaleDownToStartAnimation: Animation? = null

    internal var mNotify: Boolean = false
    private var mAutoNotify = false

    // Whether the client has set a custom starting position;
    internal var mUsingCustomStart: Boolean = false

    private var mShowRemind = false
    private var mRemindMessage = ""
    private var mRemindView: RemindView? = null
    private var mRemindTime = REMIND_DURATION
    private var mIsShowingRemind = false

    private var mChildScrollUpCallback: OnChildScrollUpCallback<ProgressView, RemindView>? = null

    private val mRefreshListener = object : Animation.AnimationListener {
        override fun onAnimationStart(animation: Animation?) {}

        override fun onAnimationRepeat(animation: Animation?) {}

        override fun onAnimationEnd(animation: Animation?) {
            if (mRefreshing) {

                mProgressView.startAnimRefreshing()

                mCurrentTargetOffsetTop = mProgressView.top

                if (mNotify) {

                    if (mListener == null) {
                        isRefreshing = false
                    } else {
                        mListener?.onRefresh(mAutoNotify)
                    }
                }
            } else {
                if (mShowRemind && mRemindMessage.isNotEmpty()) {
                    ensureRemind()

                    if (mRemindView != null) {
                        showRemind()
                    } else {
                        reset()
                    }

                } else {
                    reset()
                }
            }
        }
    }

    private fun showRemind() {

        mIsShowingRemind = true

        mProgressView.clearAnimation()
        mProgressView.stopAnimRefreshing()
        mProgressView.visibility = View.GONE


        mRemindView?.visibility = View.VISIBLE
        mRemindView?.message = mRemindMessage
        mRemindMessage = ""// only show once and clear the message

        val taskAfterAnim = Runnable {

            mCurrentRemindOffsetTop = mRemindView?.top
            postDelayed({
                animateOffsetToRemindStartPosition(object : AnimationListener {
                    override fun onAnimationRepeat(animation: Animation?) {

                    }

                    override fun onAnimationEnd(animation: Animation?) {
                        mCurrentRemindOffsetTop = null
                        reset()
                        mIsShowingRemind = false
                    }

                    override fun onAnimationStart(animation: Animation?) {

                    }
                })
            }, mRemindTime)

        }

        if (mRemindView?.customAnimShow(taskAfterAnim) != true) {
            animateOffsetToRemindPosition(object : AnimationListener {
                override fun onAnimationRepeat(animation: Animation?) {

                }

                override fun onAnimationEnd(animation: Animation?) {
                    taskAfterAnim.run()
                }

                override fun onAnimationStart(animation: Animation?) {

                }
            })
        } else {
            setRemindOffsetTopAndBottom(remindViewEndOffset - (mRemindView?.top
                    ?: remindViewEndOffset))
        }
    }

    private fun animateOffsetToRemindPosition(listener: AnimationListener?) {
        mAnimateToRemindPosition.reset()
        mAnimateToRemindPosition.duration = ANIMATE_TO_REMIND_DURATION.toLong()
        mAnimateToRemindPosition.interpolator = mDecelerateInterpolator
        if (listener != null) {
            mRemindView?.setAnimationListener(listener)
        }
        mRemindView?.clearAnimation()
        mRemindView?.startAnimation(mAnimateToRemindPosition)
    }

    private fun animateOffsetToRemindStartPosition(listener: AnimationListener?) {
        mAnimateToRemindStartPosition.reset()
        mAnimateToRemindStartPosition.duration = ANIMATE_TO_REMIND_START_DURATION.toLong()
        mAnimateToRemindStartPosition.interpolator = mDecelerateInterpolator
        if (listener != null) {
            mRemindView?.setAnimationListener(listener)
        }
        mRemindView?.clearAnimation()
        mRemindView?.startAnimation(mAnimateToRemindStartPosition)
    }

    /**
     * @return Whether the SwipeRefreshWidget is actively showing refresh
     * progress.
     */
    /**
     * Notify the widget that refresh state has changed. Do not call this when
     * refresh is triggered by a swipe gesture.
     *
     * @param refreshing Whether or not the view should show refresh progress.
     */
    // scale and show
    /* notify */ var isRefreshing: Boolean
        get() = mRefreshing
        set(refreshing) = if (refreshing && mRefreshing != refreshing) {
            mRefreshing = refreshing
            var endTarget = if (!mUsingCustomStart) {
                progressViewEndOffset + progressViewStartOffset
            } else {
                progressViewEndOffset
            }
            if (mScale) {
                setAnimationProgress(1f)
            }

            setTargetOffsetTopAndBottom(endTarget - mCurrentTargetOffsetTop)
            mNotify = false
            if (mAutoNotify) mNotify = true
            mProgressView.visibility = View.VISIBLE
            mProgressView.autoToAnimRefreshing(mRefreshListener)
        } else {
            setRefreshing(refreshing, false)
        }

    /***
     * @param refreshing see [isRefreshing]
     * @param notify whether to notify the [OnRefreshListener] to call [OnRefreshListener.onRefresh]
     */
    fun notifyRefresh(refreshing: Boolean, notify: Boolean = false) {

        if (!isEnabled || mReturningToStart || mRefreshing || mIsShowingRemind) {
            // Fail fast if we're not in a state where a auto notify refresh is possible
            return
        }

        mAutoNotify = notify
        isRefreshing = refreshing
    }

    private val mAnimateToCorrectPosition = object : Animation() {
        public override fun applyTransformation(interpolatedTime: Float, t: Transformation) {
            var endTarget = if (!mUsingCustomStart) {
                progressViewEndOffset - Math.abs(progressViewStartOffset)
            } else {
                progressViewEndOffset
            }
            var targetTop = mFrom + ((endTarget - mFrom) * interpolatedTime).toInt()
            val offset = targetTop - mProgressView.top
            setTargetOffsetTopAndBottom(offset)
            mProgressView.releaseToAnimRefreshing(interpolatedTime)
        }
    }

    private val mAnimateToStartPosition = object : Animation() {
        public override fun applyTransformation(interpolatedTime: Float, t: Transformation) {
            moveToStart(interpolatedTime)
        }
    }

    private val mAnimateToRemindPosition = object : Animation() {
        override fun applyTransformation(interpolatedTime: Float, t: Transformation?) {
            var endTarget = remindViewEndOffset
            var from = remindViewStartOffset
            var targetTop = from + ((endTarget - from) * interpolatedTime).toInt()
            val offset = targetTop - (mRemindView?.top ?: 0)
            setRemindOffsetTopAndBottom(offset)
        }
    }
    private val mAnimateToRemindStartPosition = object : Animation() {
        override fun applyTransformation(interpolatedTime: Float, t: Transformation?) {
            var endTarget = remindViewStartOffset
            var from = remindViewEndOffset
            var targetTop = from + ((endTarget - from) * interpolatedTime).toInt()
            val offset = targetTop - (mRemindView?.top ?: 0)
            setRemindOffsetTopAndBottom(offset)

            mCurrentTargetOffsetTop = targetTop
        }
    }

    internal fun reset() {
        mProgressView.clearAnimation()
        mProgressView.stopAnimRefreshing()
        mProgressView.visibility = View.GONE

        // Return the progress view to its start position
        if (mScale) {
            setAnimationProgress(0f /* animation complete and view is hidden */)
        } else {
            setTargetOffsetTopAndBottom(progressViewStartOffset - mCurrentTargetOffsetTop)
        }
        mCurrentTargetOffsetTop = mProgressView.top

        if (mTargetPull) {
            mCurrentTargetOffsetTop = progressViewStartOffset
            requestLayout()
        }

        if (mRemindView != null) {
            mRemindView?.clearAnimation()
            mRemindView?.visibility = View.GONE
        }
    }

    override fun setEnabled(enabled: Boolean) {
        super.setEnabled(enabled)
        if (!enabled) {
            reset()
        }
    }

    override fun onDetachedFromWindow() {
        super.onDetachedFromWindow()
        reset()
    }

    /**
     * The refresh indicator starting and resting position is always positioned
     * near the top of the refreshing content. This position is a consistent
     * location, but can be adjusted in either direction based on whether or not
     * there is a toolbar or actionbar present.
     *
     *
     * **Note:** Calling this will reset the position of the refresh indicator to
     * `start`.
     *
     *
     * @param scale Set to true if there is no view at a higher z-order than where the progress
     * spinner is set to appear. Setting it to true will cause indicator to be scaled
     * up rather than clipped.
     * @param start The offset in pixels from the top of this view at which the
     * progress spinner should appear.
     * @param end The offset in pixels from the top of this view at which the
     * progress spinner should come to rest after a successful swipe
     * gesture.
     */
    fun setProgressViewOffset(scale: Boolean, start: Int, end: Int) {
        mScale = scale
        progressViewStartOffset = start
        progressViewEndOffset = end
        mUsingCustomStart = true
        reset()
        mRefreshing = false
    }

    fun setProgressScale(scale: Boolean) {
        mScale = scale
        mProgressView.invalidate()
    }

    fun setTargetPull(pull: Boolean) {
        mTargetPull = pull
    }

    fun setShowRemind(show: Boolean = true, message: String = "") {
        mShowRemind = show
        mRemindMessage = message
    }

    fun setRemindTime(time: Long) {
        mRemindTime = time
    }

    /**
     * The refresh indicator resting position is always positioned near the top
     * of the refreshing content. This position is a consistent location, but
     * can be adjusted in either direction based on whether or not there is a
     * toolbar or actionbar present.
     *
     * @param scale Set to true if there is no view at a higher z-order than where the progress
     * spinner is set to appear. Setting it to true will cause indicator to be scaled
     * up rather than clipped.
     * @param end The offset in pixels from the top of this view at which the
     * progress spinner should come to rest after a successful swipe
     * gesture.
     */
    fun setProgressViewEndTarget(scale: Boolean, end: Int) {
        progressViewEndOffset = end
        mScale = scale
        mProgressView.invalidate()
    }

    fun setProgressStyle(size: Long) {
        mProgressView.setStyle(size)
    }

    init {

        setWillNotDraw(false)
        mDecelerateInterpolator = DecelerateInterpolator(DECELERATE_INTERPOLATION_FACTOR)

        val metrics = resources.displayMetrics

        mProgressView = createProgressView()
        addView(mProgressView)

        isChildrenDrawingOrderEnabled = true
        // the absolute offset has to take into account that the progress view starts at an offset
        progressViewEndOffset = (DEFAULT_PROGRESS_END_OFFSET * metrics.density).toInt()
        mTotalDragDistance = progressViewEndOffset.toFloat()
        mNestedScrollingParentHelper = NestedScrollingParentHelper(this)

        mNestedScrollingChildHelper = NestedScrollingChildHelper(this)
        isNestedScrollingEnabled = true

        mCurrentTargetOffsetTop = -mProgressView.viewHeight
        progressViewStartOffset = mCurrentTargetOffsetTop
        moveToStart(1.0f)

        val a = context.obtainStyledAttributes(attrs, LAYOUT_ATTRS)
        isEnabled = a.getBoolean(0, true)
        a.recycle()
    }

    override fun getChildDrawingOrder(childCount: Int, i: Int): Int {
        return when {
            mProgressViewIndex < 0 -> i
            i == childCount - 1 -> // Draw the selected child last
                mProgressViewIndex
            i >= mProgressViewIndex -> // Move the children after the selected child earlier one
                i + 1
            else -> // Keep the children before the selected child the same
                i
        }
    }

    protected abstract fun createProgressView(): ProgressView

    protected open fun createRemindView(): RemindView? = null

    /**
     * Set the listener to be notified when a refresh is triggered via the swipe
     * gesture.
     */

    fun setOnRefreshListener(listener: OnRefreshListener?) {
        mListener = listener
    }

    /**
     * Pre API 11, this does an alpha animation.
     * @param progress
     */
    internal fun setAnimationProgress(progress: Float) {
        mProgressView.scaleX = progress
        mProgressView.scaleY = progress
    }

    private fun setRefreshing(refreshing: Boolean, notify: Boolean, autoNotify: Boolean = false) {
        mAutoNotify = autoNotify
        if (mRefreshing != refreshing) {
            mNotify = notify
            ensureTarget()
            mRefreshing = refreshing
            if (mRefreshing) {
                animateOffsetToCorrectPosition(mCurrentTargetOffsetTop, mRefreshListener)
            } else {
                startScaleDownAnimation(mRefreshListener)
            }
        }
    }

    internal fun startScaleDownAnimation(listener: Animation.AnimationListener?) {
        mScaleDownAnimation = object : Animation() {
            public override fun applyTransformation(interpolatedTime: Float, t: Transformation) {
                setAnimationProgress(1 - interpolatedTime)
            }
        }
        mScaleDownAnimation?.duration = SCALE_DOWN_DURATION.toLong()
        mProgressView.setAnimationListener(listener)
        mProgressView.clearAnimation()
        mProgressView.startAnimation(mScaleDownAnimation)
    }

    @Deprecated("Use {@link #setProgressBackgroundColorSchemeResource(int)}")
    fun setProgressBackgroundColor(colorRes: Int) {
        setProgressBackgroundColorSchemeResource(colorRes)
    }

    /**
     * Set the background color of the progress spinner disc.
     *
     * @param colorRes Resource id of the color.
     */
    fun setProgressBackgroundColorSchemeResource(@ColorRes colorRes: Int) {
        setProgressBackgroundColorSchemeColor(ContextCompat.getColor(context, colorRes))
    }

    /**
     * Set the background color of the progress spinner disc.
     *
     * @param color
     */
    fun setProgressBackgroundColorSchemeColor(@ColorInt color: Int) {
        mProgressView.setBackgroundColor(color)
    }


    @Deprecated("Use {@link #setColorSchemeResources(int...)}")
    fun setColorScheme(@ColorRes vararg colors: Int) {
        setColorSchemeResources(*colors)
    }

    /**
     * Set the color resources used in the progress animation from color resources.
     * The first color will also be the color of the bar that grows in response
     * to a user swipe gesture.
     *
     * @param colorResIds
     */
    fun setColorSchemeResources(@ColorRes vararg colorResIds: Int) {
        val context = context
        val colorRes = IntArray(colorResIds.size)
        for (i in colorResIds.indices) {
            colorRes[i] = ContextCompat.getColor(context, colorResIds[i])
        }
        setColorSchemeColors(*colorRes)
    }

    /**
     * Set the colors used in the progress animation. The first
     * color will also be the color of the bar that grows in response to a user
     * swipe gesture.
     *
     * @param colors
     */
    fun setColorSchemeColors(@ColorInt vararg colors: Int) {
        ensureTarget()
        //TODO m:lorss
        //mProgress.setColorSchemeColors(*colors)
    }

    private fun ensureTarget() {
        // Don't bother getting the parent height if the parent hasn't been laid
        // out yet.
        if (mTarget == null) {
            for (i in 0 until childCount) {
                val child = getChildAt(i)
                if (child != mProgressView && child != mRemindView) {
                    mTarget = child
                    break
                }
            }
        }
    }

    private fun ensureRemind() {
        if (mRemindView == null) {
            mRemindView = createRemindView()

            mRemindView?.let {
                addView(it)
                remindViewStartOffset = -(mRemindView?.viewHeight ?: 0)
                remindViewEndOffset = 0
            }
        }
    }

    /**
     * Set the distance to trigger a sync in dips
     *
     * @param distance
     */
    fun setDistanceToTriggerSync(distance: Int) {
        mTotalDragDistance = distance.toFloat()
    }

    override fun onLayout(changed: Boolean, left: Int, top: Int, right: Int, bottom: Int) {
        val width = measuredWidth
        val height = measuredHeight
        if (childCount == 0) {
            return
        }
        if (mTarget == null) {
            ensureTarget()
        }
        if (mTarget == null) {
            return
        }

        val progressViewWidth = mProgressView.measuredWidth
        val progressViewHeight = mProgressView.measuredHeight

        val child = mTarget
        val childLeft = paddingLeft

        var fixChildTop = mCurrentTargetOffsetTop + progressViewHeight
        if (!mTargetPull || fixChildTop < 0) fixChildTop = 0

        if (mCurrentRemindOffsetTop != null && mRemindView != null) {
            var fixChildTop2 = (mCurrentRemindOffsetTop ?: 0) + (mRemindView?.viewHeight ?: 0)
            if (fixChildTop2 < 0) fixChildTop2 = 0
            if (fixChildTop2 > fixChildTop) fixChildTop = fixChildTop2
        }
        val childTop = paddingTop + fixChildTop
        val childWidth = width - paddingLeft - paddingRight
        val childHeight = height - paddingTop - paddingBottom
        child?.layout(childLeft, childTop, childLeft + childWidth, childTop + childHeight)

        mProgressView.layout(
                width / 2 - progressViewWidth / 2, mCurrentTargetOffsetTop,
                width / 2 + progressViewWidth / 2, mCurrentTargetOffsetTop + progressViewHeight
        )

        val remindViewWidth = mRemindView?.measuredWidth ?: 0
        val remindViewHeight = mRemindView?.measuredHeight ?: 0
        mRemindView?.layout(
                width / 2 - remindViewWidth / 2, mCurrentRemindOffsetTop ?: 0,
                width / 2 + remindViewWidth / 2, (mCurrentRemindOffsetTop ?: 0) + remindViewHeight
        )
    }

    public override fun onMeasure(widthMeasureSpec: Int, heightMeasureSpec: Int) {
        super.onMeasure(widthMeasureSpec, heightMeasureSpec)
        if (mTarget == null) {
            ensureTarget()
        }
        if (mTarget == null) {
            return
        }
        mTarget?.measure(
                View.MeasureSpec.makeMeasureSpec(
                        measuredWidth - paddingLeft - paddingRight,
                        View.MeasureSpec.EXACTLY
                ), View.MeasureSpec.makeMeasureSpec(
                measuredHeight - paddingTop - paddingBottom, View.MeasureSpec.EXACTLY
        )
        )

        mRemindView?.measure(
                View.MeasureSpec.makeMeasureSpec(mRemindView?.viewWidth
                        ?: 0, View.MeasureSpec.EXACTLY),
                View.MeasureSpec.makeMeasureSpec(mRemindView?.viewHeight
                        ?: 0, View.MeasureSpec.EXACTLY)
        )

        mProgressView.measure(
                View.MeasureSpec.makeMeasureSpec(mProgressView.viewWidth, View.MeasureSpec.EXACTLY),
                View.MeasureSpec.makeMeasureSpec(mProgressView.viewHeight, View.MeasureSpec.EXACTLY)
        )
        mProgressViewIndex = -1
        // Get the index of the progress view.
        for (index in 0 until childCount) {
            if (getChildAt(index) === mProgressView) {
                mProgressViewIndex = index
                break
            }
        }
    }

    /**
     * @return Whether it is possible for the child view of this layout to
     * scroll up. Override this if the child view is a custom view.
     */
    fun canChildScrollUp(): Boolean {
        if (mChildScrollUpCallback != null) {
            return mChildScrollUpCallback?.canChildScrollUp(this, mTarget) ?: false
        }
        return if (mTarget is ListView) {
            ListViewCompat.canScrollList((mTarget as ListView?)!!, -1)
        } else mTarget?.canScrollVertically(-1) ?: false
    }

    /**
     * Set a callback to override [AbsSwipeRefreshLayout.canChildScrollUp] method. Non-null
     * callback will return the value provided by the callback and ignore all internal logic.
     * @param callback Callback that should be called when canChildScrollUp() is called.
     */
    fun setOnChildScrollUpCallback(callback: OnChildScrollUpCallback<ProgressView, RemindView>?) {
        mChildScrollUpCallback = callback
    }

    override fun onInterceptTouchEvent(ev: MotionEvent): Boolean {
        ensureTarget()

        val action = ev.actionMasked
        val pointerIndex: Int

        if (mReturningToStart && action == MotionEvent.ACTION_DOWN) {
            mReturningToStart = false
        }

        if (!isEnabled || mReturningToStart || canChildScrollUp()
                || mRefreshing || mNestedScrollInProgress
        ) {
            // Fail fast if we're not in a state where a swipe is possible
            return false
        }

        when (action) {
            MotionEvent.ACTION_DOWN -> try {
                setTargetOffsetTopAndBottom(progressViewStartOffset - mProgressView.top)
                mActivePointerId = ev.getPointerId(0)
                mIsBeingDragged = false

                pointerIndex = ev.findPointerIndex(mActivePointerId)
                if (pointerIndex < 0) {
                    return false
                }
                mInitialDownY = ev.getY(pointerIndex)
            } catch (e: Throwable) {
                // nothing to do
            }

            MotionEvent.ACTION_MOVE -> {
                if (mActivePointerId == INVALID_POINTER) {
                    Log.e(LOG_TAG, "Got ACTION_MOVE event but don't have an active pointer id.")
                    return false
                }

                pointerIndex = ev.findPointerIndex(mActivePointerId)
                if (pointerIndex < 0) {
                    return false
                }
                val y = ev.getY(pointerIndex)
                startDragging(y)
            }

            MotionEvent.ACTION_POINTER_UP -> onSecondaryPointerUp(ev)

            MotionEvent.ACTION_UP, MotionEvent.ACTION_CANCEL -> {
                mIsBeingDragged = false
                mActivePointerId = INVALID_POINTER
            }
        }

        return mIsBeingDragged
    }

    override fun requestDisallowInterceptTouchEvent(b: Boolean) {
        // if this is a List < L or another view that doesn't support nested
        // scrolling, ignore this request so that the vertical scroll event
        // isn't stolen
        if (android.os.Build.VERSION.SDK_INT < 21 && mTarget is AbsListView || mTarget != null && !ViewCompat.isNestedScrollingEnabled(
                        mTarget!!
                )
        ) {
            // Nope.
        } else {
            super.requestDisallowInterceptTouchEvent(b)
        }
    }

    // NestedScrollingParent

    override fun onStartNestedScroll(child: View, target: View, nestedScrollAxes: Int): Boolean {
        return (isEnabled && !mReturningToStart && !mRefreshing
                && nestedScrollAxes and ViewCompat.SCROLL_AXIS_VERTICAL != 0)
    }

    override fun onNestedScrollAccepted(child: View, target: View, axes: Int) {
        // Reset the counter of how much leftover scroll needs to be consumed.
        mNestedScrollingParentHelper.onNestedScrollAccepted(child, target, axes)
        // Dispatch up to the nested parent
        startNestedScroll(axes and ViewCompat.SCROLL_AXIS_VERTICAL)
        mTotalUnconsumed = 0f
        mNestedScrollInProgress = true
    }

    override fun onNestedPreScroll(target: View, dx: Int, dy: Int, consumed: IntArray) {
        // If we are in the middle of consuming, a scroll, then we want to move the spinner back up
        // before allowing the list to scroll
        if (dy > 0 && mTotalUnconsumed > 0) {
            if (dy > mTotalUnconsumed) {
                consumed[1] = dy - mTotalUnconsumed.toInt()
                mTotalUnconsumed = 0f
            } else {
                mTotalUnconsumed -= dy.toFloat()
                consumed[1] = dy
            }
            moveSpinner(mTotalUnconsumed)
        }

        // If a client layout is using a custom start position for the progress
        // view, they mean to hide it again before scrolling the child view
        // If we get back to mTotalUnconsumed == 0 and there is more to go, hide
        // the progress so it isn't exposed if its blocking content is moved
        if (mUsingCustomStart && dy > 0 && mTotalUnconsumed == 0f
                && Math.abs(dy - consumed[1]) > 0
        ) {
            mProgressView.visibility = View.GONE
        }

        // Now let our nested parent consume the leftovers
        val parentConsumed = mParentScrollConsumed
        if (dispatchNestedPreScroll(dx - consumed[0], dy - consumed[1], parentConsumed, null)) {
            consumed[0] += parentConsumed[0]
            consumed[1] += parentConsumed[1]
        }
    }

    override fun getNestedScrollAxes(): Int {
        return mNestedScrollingParentHelper.nestedScrollAxes
    }

    override fun onStopNestedScroll(target: View) {
        mNestedScrollingParentHelper.onStopNestedScroll(target)
        mNestedScrollInProgress = false
        // Finish the spinner for nested scrolling if we ever consumed any
        // unconsumed nested scroll
        if (mTotalUnconsumed > 0) {
            finishSpinner(mTotalUnconsumed)
            mTotalUnconsumed = 0f
        }
        // Dispatch up our nested parent
        stopNestedScroll()
    }

    override fun onNestedScroll(
            target: View, dxConsumed: Int, dyConsumed: Int,
            dxUnconsumed: Int, dyUnconsumed: Int
    ) {
        // Dispatch up to the nested parent first
        dispatchNestedScroll(
                dxConsumed, dyConsumed, dxUnconsumed, dyUnconsumed,
                mParentOffsetInWindow
        )

        // This is a bit of a hack. Nested scrolling works from the bottom up, and as we are
        // sometimes between two nested scrolling views, we need a way to be able to know when any
        // nested scrolling parent has stopped handling events. We do that by using the
        // 'offset in window 'functionality to see if we have been moved from the event.
        // This is a decent indication of whether we should take over the event stream or not.
        val dy = dyUnconsumed + mParentOffsetInWindow[1]
        if (dy < 0 && !canChildScrollUp() && !mIsShowingRemind) {
            mTotalUnconsumed += Math.abs(dy).toFloat()
            moveSpinner(mTotalUnconsumed)
        }
    }

    // NestedScrollingChild

    override fun setNestedScrollingEnabled(enabled: Boolean) {
        mNestedScrollingChildHelper.isNestedScrollingEnabled = enabled
    }

    override fun isNestedScrollingEnabled(): Boolean {
        return mNestedScrollingChildHelper.isNestedScrollingEnabled
    }

    override fun startNestedScroll(axes: Int): Boolean {
        return mNestedScrollingChildHelper.startNestedScroll(axes)
    }

    override fun stopNestedScroll() {
        mNestedScrollingChildHelper.stopNestedScroll()
    }

    override fun hasNestedScrollingParent(): Boolean {
        return mNestedScrollingChildHelper.hasNestedScrollingParent()
    }

    override fun dispatchNestedScroll(
            dxConsumed: Int, dyConsumed: Int, dxUnconsumed: Int,
            dyUnconsumed: Int, offsetInWindow: IntArray?
    ): Boolean {
        return mNestedScrollingChildHelper.dispatchNestedScroll(
                dxConsumed, dyConsumed,
                dxUnconsumed, dyUnconsumed, offsetInWindow
        )
    }

    override fun dispatchNestedPreScroll(
            dx: Int,
            dy: Int,
            consumed: IntArray?,
            offsetInWindow: IntArray?
    ): Boolean {
        return mNestedScrollingChildHelper.dispatchNestedPreScroll(
                dx, dy, consumed, offsetInWindow
        )
    }

    override fun onNestedPreFling(
            target: View, velocityX: Float,
            velocityY: Float
    ): Boolean {
        return dispatchNestedPreFling(velocityX, velocityY)
    }

    override fun onNestedFling(
            target: View, velocityX: Float, velocityY: Float,
            consumed: Boolean
    ): Boolean {
        return dispatchNestedFling(velocityX, velocityY, consumed)
    }

    override fun dispatchNestedFling(
            velocityX: Float,
            velocityY: Float,
            consumed: Boolean
    ): Boolean {
        return mNestedScrollingChildHelper.dispatchNestedFling(velocityX, velocityY, consumed)
    }

    override fun dispatchNestedPreFling(velocityX: Float, velocityY: Float): Boolean {
        return mNestedScrollingChildHelper.dispatchNestedPreFling(velocityX, velocityY)
    }

    private fun moveSpinner(overscrollTop: Float) {

        val originalDragPercent = overscrollTop / mTotalDragDistance

        val dragPercent = Math.min(1f, Math.abs(originalDragPercent))
        val adjustedPercent = Math.max(dragPercent - .4, 0.0).toFloat() * 5 / 3
        val extraOS = Math.abs(overscrollTop) - mTotalDragDistance
        val slingshotDist = (if (mUsingCustomStart)
            progressViewEndOffset - progressViewStartOffset
        else
            progressViewEndOffset).toFloat()
        val tensionSlingshotPercent =
                Math.max(0f, Math.min(extraOS, slingshotDist * 2) / slingshotDist)
        val tensionPercent = (tensionSlingshotPercent / 4 - Math.pow(
                (tensionSlingshotPercent / 4).toDouble(), 2.0
        )).toFloat() * 2f
        val extraMove = slingshotDist * tensionPercent * 2f

        val targetY = progressViewStartOffset + (slingshotDist * dragPercent + extraMove).toInt()

        // where 1.0f is a full progress
        if (mProgressView.visibility != View.VISIBLE) {
            mProgressView.visibility = View.VISIBLE
        }
        if (!mScale) {
            mProgressView.scaleX = 1f
            mProgressView.scaleY = 1f
        }

        if (mScale) {
            setAnimationProgress(Math.min(1f, overscrollTop / mTotalDragDistance))
        }

        mProgressView.moveSpinner(
                overscrollTop,
                mTotalDragDistance,
                adjustedPercent,
                tensionPercent,
                targetY
        )


        setTargetOffsetTopAndBottom(targetY - mCurrentTargetOffsetTop)
    }

    private fun finishSpinner(overscrollTop: Float, autoNotify: Boolean = false) {
        if (overscrollTop > mTotalDragDistance) {
            setRefreshing(true, true /* notify */, autoNotify)
        } else {
            // cancel refresh
            mRefreshing = false
            var listener: Animation.AnimationListener? = null
            if (!mScale) {
                listener = object : Animation.AnimationListener {

                    override fun onAnimationStart(animation: Animation?) {}

                    override fun onAnimationEnd(animation: Animation?) {
                        if (!mScale) {
                            startScaleDownAnimation(null)
                        }
                    }

                    override fun onAnimationRepeat(animation: Animation?) {}

                }
            }
            animateOffsetToStartPosition(mCurrentTargetOffsetTop, listener)
        }
        mProgressView.finishSpinner(overscrollTop, mTotalDragDistance)
    }

    override fun onTouchEvent(ev: MotionEvent): Boolean {
        val action = ev.actionMasked
        var pointerIndex = -1

        if (mReturningToStart && action == MotionEvent.ACTION_DOWN) {
            mReturningToStart = false
        }

        if (!isEnabled || mReturningToStart || canChildScrollUp()
                || mRefreshing || mNestedScrollInProgress || mIsShowingRemind
        ) {
            // Fail fast if we're not in a state where a swipe is possible
            return false
        }

        when (action) {
            MotionEvent.ACTION_DOWN -> try {
                mActivePointerId = ev.getPointerId(0)
                mIsBeingDragged = false
            } catch (e: Throwable) {
                // nothing to do
            }

            MotionEvent.ACTION_MOVE -> {
                pointerIndex = ev.findPointerIndex(mActivePointerId)
                if (pointerIndex < 0) {
                    Log.e(LOG_TAG, "Got ACTION_MOVE event but have an invalid active pointer id.")
                    return false
                }

                val y = ev.getY(pointerIndex)
                startDragging(y)

                if (mIsBeingDragged) {
                    val overscrollTop = (y - mInitialMotionY) * DRAG_RATE
                    if (overscrollTop > 0) {
                        moveSpinner(overscrollTop)
                    } else {
                        return false
                    }
                }
            }
            MotionEvent.ACTION_POINTER_DOWN -> {
                pointerIndex = ev.actionIndex
                if (pointerIndex < 0) {
                    Log.e(
                            LOG_TAG,
                            "Got ACTION_POINTER_DOWN event but have an invalid action index."
                    )
                    return false
                }
                try {
                    mActivePointerId = ev.getPointerId(pointerIndex)
                } catch (e: Throwable) {
                    // nothing to do
                }
            }

            MotionEvent.ACTION_POINTER_UP -> onSecondaryPointerUp(ev)

            MotionEvent.ACTION_UP -> {
                pointerIndex = ev.findPointerIndex(mActivePointerId)
                if (pointerIndex < 0) {
                    Log.e(LOG_TAG, "Got ACTION_UP event but don't have an active pointer id.")
                    return false
                }

                if (mIsBeingDragged) {
                    val y = ev.getY(pointerIndex)
                    val overscrollTop = (y - mInitialMotionY) * DRAG_RATE
                    mIsBeingDragged = false
                    finishSpinner(overscrollTop)
                }
                mActivePointerId = INVALID_POINTER
                return false
            }
            MotionEvent.ACTION_CANCEL -> return false
        }

        return true
    }

    private fun startDragging(y: Float) {
        val yDiff = y - mInitialDownY
        if (yDiff > mTouchSlop && !mIsBeingDragged) {
            mInitialMotionY = mInitialDownY + mTouchSlop
            mIsBeingDragged = true
            mProgressView.startDragging()
        }
    }

    private fun animateOffsetToCorrectPosition(from: Int, listener: AnimationListener?) {
        mFrom = from
        mAnimateToCorrectPosition.reset()
        mAnimateToCorrectPosition.duration = ANIMATE_TO_TRIGGER_DURATION.toLong()
        mAnimateToCorrectPosition.interpolator = mDecelerateInterpolator
        if (listener != null) {
            mProgressView.setAnimationListener(listener)
        }
        mProgressView.clearAnimation()
        mProgressView.startAnimation(mAnimateToCorrectPosition)
    }

    private fun animateOffsetToStartPosition(from: Int, listener: AnimationListener?) {
        if (mScale) {
            // Scale the item back down
            startScaleDownReturnToStartAnimation(from, listener)
        } else {
            mFrom = from
            mAnimateToStartPosition.reset()
            mAnimateToStartPosition.duration = ANIMATE_TO_START_DURATION.toLong()
            mAnimateToStartPosition.interpolator = mDecelerateInterpolator
            if (listener != null) {
                mProgressView.setAnimationListener(listener)
            }
            mProgressView.clearAnimation()
            mProgressView.startAnimation(mAnimateToStartPosition)
        }
    }

    internal fun moveToStart(interpolatedTime: Float) {
        var targetTop = mFrom + ((progressViewStartOffset - mFrom) * interpolatedTime).toInt()
        val offset = targetTop - mProgressView.top
        setTargetOffsetTopAndBottom(offset)
    }

    private fun startScaleDownReturnToStartAnimation(
            from: Int,
            listener: Animation.AnimationListener?
    ) {
        mFrom = from
        mStartingScale = mProgressView.scaleX
        mScaleDownToStartAnimation = object : Animation() {
            public override fun applyTransformation(interpolatedTime: Float, t: Transformation) {
                val targetScale = mStartingScale + -mStartingScale * interpolatedTime
                setAnimationProgress(targetScale)
                moveToStart(interpolatedTime)
            }
        }
        mScaleDownToStartAnimation?.duration = SCALE_DOWN_DURATION.toLong()
        if (listener != null) {
            mProgressView.setAnimationListener(listener)
        }
        mProgressView.clearAnimation()
        mProgressView.startAnimation(mScaleDownToStartAnimation)
    }

    internal fun setTargetOffsetTopAndBottom(offset: Int) {
        mProgressView.bringToFront()
        ViewCompat.offsetTopAndBottom(mProgressView, offset)
        mCurrentTargetOffsetTop = mProgressView.top
    }

    internal fun setRemindOffsetTopAndBottom(offset: Int) {
        if (mRemindView != null) {
            mRemindView?.bringToFront()
            ViewCompat.offsetTopAndBottom(mRemindView, offset)
            mCurrentRemindOffsetTop = mRemindView?.top
        }
    }

    private fun onSecondaryPointerUp(ev: MotionEvent) {
        try {
            val pointerIndex = ev.actionIndex
            val pointerId = ev.getPointerId(pointerIndex)
            if (pointerId == mActivePointerId) {
                // This was our active pointer going up. Choose a new
                // active pointer and adjust accordingly.
                val newPointerIndex = if (pointerIndex == 0) 1 else 0
                mActivePointerId = ev.getPointerId(newPointerIndex)
            }
        } catch (e: Throwable) {
            // nothing to do
        }
    }

    /**
     * Classes that wish to be notified when the swipe gesture correctly
     * triggers a refresh should implement this interface.
     */
    interface OnRefreshListener {
        /**
         * Called when a swipe gesture triggers a refresh.
         */
        fun onRefresh(autoNotify: Boolean)
    }

    /**
     * Classes that wish to override [AbsSwipeRefreshLayout.canChildScrollUp] method
     * behavior should implement this interface.
     */
    interface OnChildScrollUpCallback<ProgressView, RemindView>
            where ProgressView : View,
                  ProgressView : SwipeRefreshProgress,
                  RemindView : View,
                  RemindView : SwipeRefreshRemind {
        /**
         * Callback that will be called when [AbsSwipeRefreshLayout.canChildScrollUp] method
         * is called to allow the implementer to override its behavior.
         *
         * @param parent SwipeRefreshLayout that this callback is overriding.
         * @param child The child view of SwipeRefreshLayout.
         *
         * @return Whether it is possible for the child view of parent layout to scroll up.
         */
        fun canChildScrollUp(
                parent: AbsSwipeRefreshLayout<ProgressView, RemindView>,
                child: View?
        ): Boolean
    }

    companion object {

        private val LOG_TAG = AbsSwipeRefreshLayout::class.java?.getSimpleName()

        private const val DECELERATE_INTERPOLATION_FACTOR = 2f
        private const val INVALID_POINTER = -1
        private const val DRAG_RATE = .5f

        private const val SCALE_DOWN_DURATION = 150
        private const val ANIMATE_TO_TRIGGER_DURATION = 200
        private const val ANIMATE_TO_START_DURATION = 200
        private const val ANIMATE_TO_REMIND_DURATION = 200
        private const val REMIND_DURATION = 2000L
        private const val ANIMATE_TO_REMIND_START_DURATION = 200


        // Default offset in dips from the top of the view to where the progress spinner should stop
        private const val DEFAULT_PROGRESS_END_OFFSET = 64

        private val LAYOUT_ATTRS = intArrayOf(android.R.attr.enabled)
    }
}
/**
 * Simple constructor to use when creating a SwipeRefreshLayout from code.
 *
 * @param context
 */
