package com.raybritton.htmltableparser;

import android.content.Context;
import android.text.Html;
import android.text.method.LinkMovementMethod;
import android.util.TypedValue;
import android.widget.TableLayout;
import android.widget.TableRow;
import android.widget.TableRow.LayoutParams;
import android.widget.TextView;

import com.raybritton.htmltableparser.models.Cell;
import com.raybritton.htmltableparser.models.Row;
import com.raybritton.htmltableparser.models.Table;

public class TableLayoutConverter {
    private final Context ctx;

    private final int ROW_PADDING;

    public TableLayoutConverter(Context ctx) {
        this.ctx = ctx;
        ROW_PADDING = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, 2, ctx.getResources().getDisplayMetrics());
    }

    public TableLayout makeTableLayout(Table table) {
        TableLayout tableLayout = new TableLayout(ctx);
        tableLayout.setBackgroundColor(table.getBackgroundColor());
        tableLayout.setPadding(ROW_PADDING, ROW_PADDING, ROW_PADDING, ROW_PADDING);
        if(table.getHeader() != null) {
            tableLayout.addView(makeRow(table.getHeader()));
        }

        for (int i = 0; i < table.getRowCount(); i++) {
            TableRow tableRow = makeRow(table.getRow(i));
            tableLayout.addView(tableRow);
        }

        if(table.getFooter() != null) {
            tableLayout.addView(makeRow(table.getFooter()));
        }

        return tableLayout;
    }

    private TableRow makeRow(Row row) {
        TableRow tableRow = new TableRow(ctx);
        tableRow.setBackgroundColor(row.getBackgroundColor());
        for (int i = 0; i < row.getCellCount(); i++) {
            Cell cell = row.getCell(i);
            int padding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, row.getCell(i).getPaddingDp(), ctx.getResources().getDisplayMetrics());
            int margin = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, row.getCell(i).getMarginDp(), ctx.getResources().getDisplayMetrics());
            TextView textView = new TextView(ctx);
            textView.setText(Html.fromHtml(cell.getText()));
            textView.setPadding(padding, padding, padding, padding);
            textView.setBackgroundColor(cell.getBackgroundColor());
            textView.getPaint().setFakeBoldText(cell.isBold());
            textView.setGravity(cell.getGravity());
            textView.setMovementMethod(new LinkMovementMethod());
            if (cell.getWidthDp() >= 0) {
                int width = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, cell.getWidthDp(), ctx.getResources().getDisplayMetrics());
                LayoutParams params = new LayoutParams(width, LayoutParams.WRAP_CONTENT);
                params.bottomMargin = margin;
                params.topMargin = margin;
                params.leftMargin = margin;
                params.rightMargin = margin;
                textView.setLayoutParams(params);
            } else if (cell.getWidthPerc() >= 0) {
                LayoutParams params = new LayoutParams(0, LayoutParams.WRAP_CONTENT, cell.getWidthPerc());
                params.bottomMargin = margin;
                params.topMargin = margin;
                params.leftMargin = margin;
                params.rightMargin = margin;
                textView.setLayoutParams(params);
            } else {
                LayoutParams params = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
                params.bottomMargin = margin;
                params.topMargin = margin;
                params.leftMargin = margin;
                params.rightMargin = margin;
                textView.setLayoutParams(params);
            }
            tableRow.addView(textView);
        }
        return tableRow;
    }
}
