package com.raybritton.htmltableparser;

import android.graphics.Color;
import android.util.Log;
import android.view.Gravity;

import com.raybritton.htmltableparser.models.Cell;
import com.raybritton.htmltableparser.models.Row;
import com.raybritton.htmltableparser.models.Table;

import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;

import java.util.ArrayList;
import java.util.List;

public class HtmlTableParser {

    /**
     * @param html valid html with any number of tables in it
     * @return a list of tables parsed from the html
     */
    public static List<Table> parse(String html) {
        List<Table> tables = new ArrayList<>();
        Document document = Jsoup.parse(html);
        Elements tableElements = document.select("table");
        for (Element table : tableElements) {
            tables.add(parse(table));
        }
        return tables;
    }

    private static Table parse(Element element) {
        Row header = null;
        Row footer = null;
        List<Row> body = new ArrayList<>();

        Elements headerElement = element.getElementsByTag("thead");
        Elements footerElement = element.getElementsByTag("tfoot");
        Elements bodyElement = element.getElementsByTag("tbody");

        Integer cellPadding = null;
        Integer cellMargin = null;

        if (element.hasAttr("cellpadding")) {
            cellPadding = Integer.parseInt(element.attr("cellpadding"));
        }
        if (element.hasAttr("cellmargin")) {
            cellMargin = Integer.parseInt(element.attr("cellmargin"));
        }

        if (headerElement != null && !headerElement.isEmpty()) {
            header = parseRow(headerElement.first(), element, cellPadding, cellMargin);
            headerElement.remove();
        }

        if (footerElement != null && !footerElement.isEmpty()) {
            footer = parseRow(footerElement.first(), element, cellPadding, cellMargin);
            footerElement.remove();
        }

        Elements rows;
        if (bodyElement != null && !bodyElement.isEmpty()) {
            rows = bodyElement.first().getElementsByTag("tr");
        } else {
            rows = element.getElementsByTag("tr");
        }
        for (Element row : rows) {
            body.add(parseRow(row, element, cellPadding, cellMargin));
        }

        return new Table(new ImmutableList<>(body), header, footer);
    }

    private static Row parseRow(Element rowElement, Element tableElement, Integer cellPadding, Integer cellMargin) {
        List<Cell> cells = new ArrayList<>();

        Elements cellElements = rowElement.select("td, th");

        for (Element element : cellElements) {
            cells.add(parseCell(element, rowElement, cellPadding, cellMargin));
        }

        Row row = new Row(new ImmutableList<>(cells));

        if (rowElement.hasAttr("bgcolor")) {
            row.setBackgroundColor(Color.parseColor(rowElement.attr("bgcolor")));
        }

        return row;
    }

    private static Cell parseCell(Element cellElement, Element rowElement, Integer cellPadding, Integer cellMargin) {
        Cell cell = new Cell(cellElement.html());
        int vertAlign = Gravity.CENTER_VERTICAL;
        int horzAlign = Gravity.START;
        if (cellElement.tagName().equals("th")) {
            cell.setBold(true);
            horzAlign = Gravity.CENTER_HORIZONTAL;
        }
        if (cellElement.hasAttr("bgcolor")) {
            try {
                cell.setBackgroundColor(Color.parseColor(cellElement.attr("bgcolor")));
            } catch (IllegalArgumentException e) {
                cell.setBackgroundColor(Color.TRANSPARENT);
                Log.e("PARSER", "Unsupported colour: " + cellElement.attr("bgcolor"));
            }
        }
        if (cellPadding != null) {
            cell.setPaddingDp(cellPadding);
        }
        if (cellMargin != null) {
            cell.setMarginDp(cellMargin);
        }
        if (cellElement.hasAttr("width")) {
            String value = cellElement.attr("width");
            if(value.endsWith("%")) {
                value = value.substring(0, value.length() - 1);
                cell.setWidthPerc(Integer.parseInt(value) / 100.0f);
            } else {
                cell.setWidthDp(Integer.parseInt(cellElement.attr("width")));
            }
        }
        if (cellElement.hasAttr("align")) {
            switch(cellElement.attr("align")) {
                case "left":
                    horzAlign = Gravity.START;
                    break;
                case "right":
                    horzAlign = Gravity.END;
                    break;
                case "center":
                case "justify":
                    horzAlign = Gravity.CENTER_HORIZONTAL;
                    break;
            }
        }
        if (cellElement.hasAttr("valign")) {
            switch (cellElement.attr("valign")) {
                case "top":
                    vertAlign = Gravity.TOP;
                    break;
                case "middle":
                    vertAlign = Gravity.CENTER_VERTICAL;
                    break;
                case "bottom":
                    vertAlign = Gravity.BOTTOM;
                    break;
            }
        }
        cell.setGravity(vertAlign | horzAlign);

        return cell;
    }
}
