package com.raybritton.htmltableparser;

import android.text.Html;
import android.text.SpannableStringBuilder;
import android.text.Spanned;
import android.util.SparseIntArray;
import android.util.TypedValue;
import android.view.View;
import android.widget.TextView;

import com.raybritton.htmltableparser.models.Cell;
import com.raybritton.htmltableparser.models.Row;
import com.raybritton.htmltableparser.models.Table;

/**
 * When this is used the TextViews paint (and text appearance) must be set before calling makeCharSequence()
 * and it can not be changed afterwards, otherwise there may be display issues.
 */
public class CharSequenceConverter {
    private static final int DEFAULT = -1;
    private SparseIntArray htmlWidths = new SparseIntArray();

    public CharSequence makeCharSequence(TextView view, Table table) {
        view.setLineSpacing(0, 1.2f);
        int viewWidth = view.getWidth();
        SpannableStringBuilder builder = new SpannableStringBuilder();
        htmlWidths.clear();

        for (int rowIdx = 0; rowIdx < table.getRowCount(); rowIdx++) {
            for (int cellIdx = 0; cellIdx < table.getRow(rowIdx).getCellCount(); cellIdx++) {
                Cell cell = table.getRow(rowIdx).getCell(cellIdx);
                int width = DEFAULT;
                int padding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, cell.getPaddingDp(), view.getResources().getDisplayMetrics());
                int margin = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, cell.getMarginDp(), view.getResources().getDisplayMetrics());
                if (cell.getWidthDp() >= 0) {
                    width = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, cell.getWidthDp(), view.getResources().getDisplayMetrics());
                } else if (cell.getWidthPerc() >= 0) {
                    width = (int) (viewWidth * cell.getWidthPerc());
                }
                if (width < 1) {
                    width = (int) view.getPaint().measureText(cell.getText()) + padding + padding + margin + margin;
                } else {
                    width += (padding + padding + margin + margin);
                }
                if (htmlWidths.indexOfKey(cellIdx) < 0 || htmlWidths.get(cellIdx, 0) < width) {
                    htmlWidths.put(cellIdx, width);
                }
            }
        }

        if (table.getHeader() != null) {
            parseRow(view, builder, table.getHeader());
        }
        for (int i = 0; i < table.getRowCount(); i++) {
            parseRow(view, builder, table.getRow(i));
        }
        if (table.getFooter() != null) {
            parseRow(view, builder, table.getFooter());
        }

        return builder;
    }

    private void parseRow(View view, SpannableStringBuilder builder, Row row){
        for (int j = 0; j < row.getCellCount(); j++) {
            Cell cell = row.getCell(j);
            int start = builder.length();
            builder.append(Html.fromHtml(cell.getText()));
            int end = builder.length();

            int padding = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, cell.getPaddingDp(), view.getResources().getDisplayMetrics());
            int margin = (int) TypedValue.applyDimension(TypedValue.COMPLEX_UNIT_DIP, cell.getMarginDp(), view.getResources().getDisplayMetrics());

            builder.setSpan(new TableColumnSpan(htmlWidths.get(j), padding, margin, cell), start, end, Spanned.SPAN_INCLUSIVE_EXCLUSIVE);
        }
        builder.append("\n");
    }
}
