package com.random_guys.pica

import android.content.Context
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import androidx.core.content.ContextCompat
import com.random_guys.rv.BaseViewHolder
import com.random_guys.rv.RV
import de.hdodenhof.circleimageview.CircleImageView

class SelectContactAdapter(
    private val mContext: Context,
    private var mContactSelectedListener: ContactSelectedListener
) : RV<BaseViewHolder>() {

    private val contacts: ArrayList<MainContact> = ArrayList()

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): BaseViewHolder {
        val view: View?
        return when (viewType) {
            VIEW_TYPE_LOCAL -> {
                view = LayoutInflater.from(mContext).inflate(R.layout.contact_item, parent, false)
                ViewHolder(view, mContactSelectedListener)
            }
            VIEW_TYPE_SERVER -> {
                view = LayoutInflater.from(mContext).inflate(R.layout.server_contact_item, parent, false)
                ServerContactViewHolder(view, mContactSelectedListener)
            }
            else -> {
                view = LayoutInflater.from(mContext).inflate(R.layout.layout_empty_view, parent, false)
                EmptyViewHolder(view)
            }
        }
    }

    fun get(position: Int): MainContact = contacts[position]

    fun addMany(contacts: Collection<MainContact>) {
        this.contacts.addAll(contacts.sortedBy { it.mainContactType.name })
        notifyDataSetChanged()
    }

    fun clear() {
        contacts.clear()
        notifyDataSetChanged()
    }

    override fun onBindViewHolder(holder: BaseViewHolder, position: Int) {
        holder.onBind(position)
    }

    override fun getItemViewType(position: Int): Int {
        return if (contacts.size > 0) {
            when (contacts[position].mainContactType) {
                MainContact.MainContactType.Local -> VIEW_TYPE_LOCAL
                MainContact.MainContactType.Server -> VIEW_TYPE_SERVER
                MainContact.MainContactType.None -> VIEW_TYPE_EMPTY
            }
        } else {
            VIEW_TYPE_EMPTY
        }
    }

    override fun getItemCount(): Int {
        return if (contacts.size > 0) {
            contacts.size
        } else {
            1
        }
    }

    inner class ServerContactViewHolder(itemView: View, private var contactSelectedListener: ContactSelectedListener) : BaseViewHolder(itemView) {
        private val contactTitleLabelIconView: TitleLabelIconView =
            itemView.findViewById(R.id.contact)
        private val logoImageView = itemView.findViewById<CircleImageView>(R.id.logo_icon)

        override fun onBind(position: Int) {
            super.onBind(adapterPosition)

            with(contacts[adapterPosition].serverContact) {
                this?.let {
                    contactTitleLabelIconView.title = name
                    contactTitleLabelIconView.initials = name.initials()
                    contactTitleLabelIconView.label = id
                    logoImageView.visibility = View.VISIBLE

                    when {
                        name.initials().contains(Regex("[A-B]")) -> contactTitleLabelIconView.drawable = ContextCompat.getDrawable(mContext, R.drawable.transaction_initials)!!
                        name.initials().contains(Regex("[C-H]")) -> contactTitleLabelIconView.drawable = ContextCompat.getDrawable(mContext, R.drawable.transaction_initials_peachy_pink)!!
                        else -> contactTitleLabelIconView.drawable = ContextCompat.getDrawable(mContext, R.drawable.transaction_initials_carolina_blue)!!
                    }
                }
            }

            contactTitleLabelIconView.setOnClickListener {
                contactSelectedListener.onContactSelected(it, adapterPosition)
            }
        }
    }

    inner class ViewHolder(
        itemView: View,
        private var contactSelectedListener: ContactSelectedListener
    ) :
        BaseViewHolder(itemView) {

        private val contactTitleLabelIconView: TitleLabelIconView =
            itemView.findViewById(R.id.contact)
        private val logoImageView = itemView.findViewById<CircleImageView>(R.id.logo_icon)

        override fun onBind(position: Int) {
            super.onBind(adapterPosition)

            with(contacts[adapterPosition].localContact) {
                this?.let {
                    contactTitleLabelIconView.title = name
                    contactTitleLabelIconView.initials = name.initials()
                    contactTitleLabelIconView.label = id
                    logoImageView.visibility = View.INVISIBLE
                    when {
                        name.initials().contains(Regex("[A-B]")) -> contactTitleLabelIconView.drawable = ContextCompat.getDrawable(mContext, R.drawable.transaction_initials)!!
                        name.initials().contains(Regex("[C-H]")) -> contactTitleLabelIconView.drawable = ContextCompat.getDrawable(mContext, R.drawable.transaction_initials_peachy_pink)!!
                        else -> contactTitleLabelIconView.drawable = ContextCompat.getDrawable(mContext, R.drawable.transaction_initials_carolina_blue)!!
                    }
                }
            }

            contactTitleLabelIconView.setOnClickListener {
                contactSelectedListener.onContactSelected(it, adapterPosition)
            }
        }
    }

    inner class EmptyViewHolder(itemView: View) : BaseViewHolder(itemView)

    companion object {

        private const val VIEW_TYPE_SERVER = 2
        private const val VIEW_TYPE_LOCAL = 1
        private const val VIEW_TYPE_EMPTY = 0
    }
}