package com.qfpay.clientstat.upload;

import android.app.AlarmManager;
import android.app.PendingIntent;
import android.app.job.JobInfo;
import android.app.job.JobScheduler;
import android.content.ComponentName;
import android.content.Context;
import android.os.Build;

import com.qfpay.clientstat.utils.Logger;

import java.io.File;
import java.util.Calendar;
import java.util.Date;

/**
 * 定时上传策略
 * 需指定上传时间点，和间隔时间
 */
public class TimingUpload implements UploadStrategy {
    private static final String TAG = "TimingUpload";
    //默认日志上传时间间隔，ms
    private static final int DEFAULT_UPLOAD_PERIOD = 24 * 60 * 60 * 1000;

    /**
     * 第一次上传时间
     */
    private Date firstTime;

    /**
     * 上传间隔
     */
    private int period;

    /**
     * 上下文
     */
    private Context context;

    private volatile boolean isSetAlarmTask = false;

    /**
     * 构造函数
     *
     * @param context   上下文
     * @param firstTime 任务第一次执行时间点  默认23:00:00
     * @param period    间隔周期 默认24小时
     */
    public TimingUpload(Context context, Date firstTime, int period) {
        if (context == null) {
            throw new IllegalArgumentException("the context must not be null.");
        }
        if (firstTime == null) {
            Calendar calendar = Calendar.getInstance();
            calendar.set(Calendar.HOUR_OF_DAY, 23);
            calendar.set(Calendar.MINUTE, 0);
            calendar.set(Calendar.SECOND, 0);
            firstTime = calendar.getTime();
        }
        if (period <= 0) {
            period = DEFAULT_UPLOAD_PERIOD;
        }
        this.context = context;
        this.firstTime = firstTime;
        this.period = period;

    }

    @Override
    public boolean isSatisfied(File eventFile) {
        if (!isSetAlarmTask) {
            isSetAlarmTask = true;
            PlatformCompat.scheduleUploadJob(context, firstTime, period);
        }
        return false;
    }


    private static class PlatformCompat {
        /**
         * 开启定时任务
         * 依据不同系统版本采用不同实现
         * LOLLIPOP及以上系统采用JobSchedule
         * LOLLIPOP一下采用AlarmManager
         *
         * @param context   上下文
         * @param firstTime 第一次上传时间
         * @param period    上传周期
         */
        static void scheduleUploadJob(Context context, Date firstTime, int period) {
            //JobSchedule无法实现定点启动任务
            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                ComponentName jobServiceComponent = new ComponentName(context, UploadJobService.class);
                JobInfo.Builder jobBuilder = new JobInfo.Builder(250, jobServiceComponent);
                //任意网络可执行
                jobBuilder.setRequiredNetworkType(JobInfo.NETWORK_TYPE_ANY);
                //设置任务最晚执行时间
//                long delay = firstTime.getTime() - System.currentTimeMillis();
//                jobBuilder.setOverrideDeadline(delay);
                //设置任务周期执行间隔
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                    jobBuilder.setPeriodic(period, period);
                } else if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
                    jobBuilder.setPeriodic(period);
                }
                JobScheduler jobScheduler = (JobScheduler) context.getSystemService(Context.JOB_SCHEDULER_SERVICE);
                if (jobScheduler == null) {
                    Logger.e(TAG, "the job scheduler is null.");
                    return;
                }
                try {
                    int scheduleResult = jobScheduler.schedule(jobBuilder.build());
                    if (scheduleResult == JobScheduler.RESULT_FAILURE) {
                        Logger.e(TAG, "start upload file job scheduler failed, please check.");
                    } else {
                        Logger.d(TAG, "start upload file job scheduler success.");
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    Logger.e(TAG, "start upload file job scheduler error: %s", e.getMessage());
                }
            } else {
                PendingIntent pendingIntent = PendingIntent.getService(context, 0, UploadEventFileIntentService.getCallingIntent(context), 0);
                int alarmType = AlarmManager.RTC_WAKEUP;
                AlarmManager alarmManager = (AlarmManager) context.getSystemService(Context.ALARM_SERVICE);
                if (alarmManager == null) {
                    Logger.e(TAG, "the alarm manager is null.");
                    return;
                }
                if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.KITKAT) {
                    alarmManager.setWindow(alarmType, firstTime.getTime(), period, pendingIntent);
                }
                alarmManager.setRepeating(alarmType, firstTime.getTime(), period, pendingIntent);
                Logger.d(TAG, "set one repeat alarm task, first start at %s, and the interval time is %d", firstTime.toString(), period);
            }
        }
    }
}