package com.qfpay.clientstat.internal;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.os.Process;

import com.qfpay.clientstat.config.StatConfig;
import com.qfpay.clientstat.event.Event;
import com.qfpay.clientstat.upload.UploadStrategy;
import com.qfpay.clientstat.utils.DeviceUtil;
import com.qfpay.clientstat.utils.Logger;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import in.joye.urlconnection.client.Response;

/**
 * 默认事件处理器
 * <p>
 * Created by joye on 2017/9/21.
 */

public final class DefaultEventHandler implements EventHandler {
    private static final String TAG = "DefaultEventHandler";
    private static final String THREAD_NAME_WRITE_FILE = "client-stat-write-file-thread";
    private Handler mEventHandler;
    private EventWriter eventWriter = null;

    public DefaultEventHandler() {
        Logger.i(TAG, "create new DefaultEventHandler instance.");
    }

    @Override
    public void handleEvent(Context context, Event event, StatConfig statConfig) {
        context = context.getApplicationContext();
        if (this.mEventHandler == null) {
            createEventHandler(context, statConfig);
        }
        Message eventMsg = mEventHandler.obtainMessage();
        eventMsg.obj = event;
        mEventHandler.sendMessage(eventMsg);
    }

    @Override
    public EventWriter getEventWriter() {
        return eventWriter;
    }

    /**
     * 事件写入处理器
     */
    static class EventWriteHandler extends Handler {
        private EventWriter eventWriter;
        private List<UploadStrategy> uploadStrategyList;
        private Context context;
        private List<File> uploadingFiles;

        public EventWriteHandler(Looper looper, Context context, StatConfig statConfig, EventWriter eventWriter) {
            super(looper);
            this.context = context;
            this.eventWriter = eventWriter;
            this.uploadStrategyList = statConfig.getUploadStrategies();
            this.uploadingFiles = new ArrayList<>();
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                Event event = (Event) msg.obj;
                if (event == null) {
                    Logger.e(TAG, "receive empty msg, and the event is null, just return.");
                    return;
                }
                File eventFile = eventWriter.writeEvent(event);
                for (UploadStrategy uploadStrategy : uploadStrategyList) {
                    boolean satisfied = uploadStrategy.isSatisfied(eventFile);
                    if (satisfied) {
                        File newFile = eventWriter.newFile();
                        Logger.i(TAG, "【%s】the event file {'%s'} is satisfied upload condition {'%s'}, upload now.", DeviceUtil.getProcessName(context), eventFile.getAbsoluteFile(), uploadStrategy.getClass().getSimpleName());

                        uploadEventFiles(newFile);
                        break;
                    }
                }

            } catch (IOException e) {
                Logger.e(TAG, "write event to file failed, the error is %s.", e.getMessage());
            }
            super.handleMessage(msg);
        }

        /**
         * 上传统计文件
         *
         * @param writingFile 正在写入的文件
         */
        private void uploadEventFiles(File writingFile){
            //检查网络状况
            if (!DeviceUtil.isNetAvailable(context)) {
                Logger.w(TAG, "uploadEventFiles(): the current net is not available, just return.");
                return;
            }
            //检查文件合法性
            if (writingFile == null || !writingFile.exists()) {
                Logger.e(TAG, "uploadEventFiles(): the writing file is null or not exists, this should not happened. just continue.");
            }
            Logger.i(TAG, "There are %s files is uploading.", uploadingFiles.size());
            List<File> allFiles = eventWriter.getAllFiles();
            if (allFiles != null && allFiles.size() > 0) {
                //去除正在写入的文件
                allFiles.remove(writingFile);

                List<File> preparedFiles = new ArrayList<>();
                //遍历所有文件，判断是否需要上传
                for (File file : allFiles) {
                    if (file.isFile() && !uploadingFiles.contains(file)) {
                        preparedFiles.add(file);
                    }
                }
                Logger.i(TAG, "There are %s new file need to upload.", preparedFiles.size());
                //如果存在新文件需要上传，添加到上传列表，并启动上传
                if (preparedFiles.size() > 0) {
                    uploadingFiles.addAll(preparedFiles);
                    startUpload(preparedFiles);
                }
            }
        }

        private void startUpload(List<File> files){
            for(final File eventFile : files) {
                //过滤空文件
                if(eventFile.length() == 0) {
                    try {
                        uploadingFiles.remove(eventFile);
                        eventWriter.deleteFile(eventFile);
                    } catch (IOException e) {
                        Logger.e(TAG, "delete event file fail. the reason is %s", e.getMessage());
                    }
                    continue;
                }
                EventFileUploader.uploadEventFileAsync(eventFile, new in.joye.urlconnection.client.Callback<UploadResult>() {
                    @Override
                    public void success(UploadResult uploadResult, Response response) {
                        Logger.i(TAG, "upload file success, the result is %s.", uploadResult);
                        if (uploadResult != null && uploadResult.isUploadSuccess()) {
                            try {
                                uploadingFiles.remove(eventFile);
                                eventWriter.deleteFile(eventFile);
                                Logger.i(TAG, "delete event file success.");
                            } catch (IOException e) {
                                Logger.e(TAG, "delete event file fail. the reason is %s", e.getMessage());
                            }
                        }
                    }

                    @Override
                    public void failure(int statusCode, String error) {
                        Logger.e(TAG, "upload file failed, the reason is %s", error);
                    }
                });
            }
        }
    }

    private void createEventHandler(Context context, StatConfig statConfig) {
        if (mEventHandler == null) {
            synchronized (DefaultEventHandler.class) {
                if (mEventHandler == null) {
                    eventWriter = new EventFileWriter(context, statConfig);
                    HandlerThread handlerThread = new HandlerThread(THREAD_NAME_WRITE_FILE, Process.THREAD_PRIORITY_BACKGROUND);
                    handlerThread.start();
                    mEventHandler = new EventWriteHandler(handlerThread.getLooper(), context, statConfig, eventWriter);
                }
            }
        }
    }
}
