package com.qfpay.clientstat.internal;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Environment;

import com.qfpay.clientstat.utils.DeviceUtil;
import com.qfpay.clientstat.utils.FileUtil;
import com.qfpay.clientstat.utils.Logger;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import static android.os.Build.VERSION.SDK_INT;

/**
 * 默认的文件提供器
 * 文件命名规则为 进程id-时间戳.event
 * <p>
 * Created by joye on 2017/9/22.
 */

public final class DefaultEventFileProvider implements EventFileProvider {
    private static final String TAG = "DefaultEventFileProvide";
    public static final String EVENT_FILE_SUFFIX = ".event";
    private static final String EVENT_FILE_DIR = "client_stat";
    private Context mContext;
    private String mRelativeDir;
    private volatile boolean writeExternalStorageGranted = false;
    private String mProcessName;

    public DefaultEventFileProvider(Context context) {
        this(context, EVENT_FILE_DIR);
    }

    public DefaultEventFileProvider(Context context, String dirRelative) {
        if (context == null) throw new IllegalArgumentException("context must not be null.");
        if (dirRelative == null) {
            dirRelative = EVENT_FILE_DIR;
            Logger.e(TAG, "custom event file dir is null, just use default dir {'%s'}", EVENT_FILE_DIR);
        }
        this.mContext = context.getApplicationContext();
        this.mRelativeDir = dirRelative;
        this.mProcessName = DeviceUtil.getProcessName(context);
        Logger.d(TAG, "current process name is %s", mProcessName);
    }

    @Override
    public List<File> list() {
        List<File> allEventFiles = new ArrayList<>();
        FilenameFilter filenameFilter = eventFileFilter();
        if (hasStoragePermission(mContext)) {
            File[] externalFiles = externalStorageDirectory().listFiles(filenameFilter);
            if (externalFiles != null) {
                allEventFiles.addAll(Arrays.asList(externalFiles));
            }
        }

        File[] appFiles = appStorageDirectory().listFiles(filenameFilter);
        if (appFiles != null) {
            allEventFiles.addAll(Arrays.asList(appFiles));
        }

        Logger.d(TAG, "list all event files, the all file's size is %d.", allEventFiles.size());
        return allEventFiles;
    }

    @Override
    public synchronized File newEventFile() throws IOException {
        File dir = null;
        if (hasStoragePermission(mContext)) {
            dir = externalStorageDirectory();
        } else {
            dir = appStorageDirectory();
        }
        if (!dir.exists()) {
            boolean mkDirSuc = dir.mkdirs();
            if (!mkDirSuc) {
                Logger.e(TAG, "create dir '%s' failed.", dir.getAbsoluteFile());
            }
        }
        File newExternalFile = new File(dir, mProcessName + "-" + System.currentTimeMillis() + EVENT_FILE_SUFFIX);

        boolean createFileSuc = newExternalFile.createNewFile();
        if (!createFileSuc) {
            Logger.e(TAG, "create file '%s' failed.", newExternalFile.getAbsoluteFile());
        } else {
            Logger.i(TAG, "create new event file {'%s'}", newExternalFile.getAbsoluteFile());
        }
        return newExternalFile;
    }

    @Override
    public void clearEventFiles() {
        if (hasStoragePermission(mContext)) {
            File externalDir = externalStorageDirectory();
            if (externalDir.exists()) {
                File[] externalFiles = externalDir.listFiles();
                if (externalFiles != null) {
                    for (File file : externalFiles) {
                        delEventFile(file);
                    }
                    Logger.d(TAG, "delete external event file size is %d", externalFiles.length);
                }
            }
        }

        File appDir = appStorageDirectory();
        if (appDir.exists()) {
            File[] appFiles = appDir.listFiles();
            if (appFiles != null) {
                for (File file : appFiles) {
                    delEventFile(file);
                }
                Logger.d(TAG, "delete app event file size is %d", appFiles.length);
            }
        }
    }

    @Override
    public void delEventFile(File file) {
        if (file == null || !file.exists()) {
            Logger.e(TAG, "the file is null or not exist.");
            return;
        }
        FileUtil.delFile(file);
    }

    private File externalStorageDirectory() {
        File downloadsDir = Environment.getExternalStoragePublicDirectory(Environment.DIRECTORY_DOWNLOADS);
        return new File(downloadsDir, File.separator + "client-stat-" + mContext.getPackageName() + mRelativeDir + File.separator);
    }

    private File appStorageDirectory() {
        File appFilesDir = mContext.getFilesDir();
        return new File(appFilesDir, mRelativeDir);
    }

    //是否有存储权限
    private boolean hasStoragePermission(Context context) {
        if (SDK_INT < Build.VERSION_CODES.M) {
            return true;
        }
        //一旦赋予的读写权限，在当前的进程的声明周期中不会变化，所以可以缓存状态
        if (writeExternalStorageGranted) {
            return true;
        }

        writeExternalStorageGranted = context.checkSelfPermission(Manifest.permission.WRITE_EXTERNAL_STORAGE) == PackageManager.PERMISSION_GRANTED;
        return writeExternalStorageGranted;
    }

    //过滤属于当前进程统计文件
    private FilenameFilter eventFileFilter() {
        return new FilenameFilter() {
            @Override
            public boolean accept(File file, String fileName) {
                if (fileName != null) {
                    String[] part = fileName.split("-");
                    if (part.length > 0) {
                        String firstPart = part[0];
                        return firstPart.equalsIgnoreCase(mProcessName);
                    }
                }
                return false;
            }
        };
    }
}
