package com.qfpay.clientstat;

import android.content.Context;
import android.support.v4.util.ArrayMap;

import com.qfpay.clientstat.config.StatConfig;
import com.qfpay.clientstat.event.Event;
import com.qfpay.clientstat.internal.DefaultEventHandler;
import com.qfpay.clientstat.internal.EventHandler;
import com.qfpay.clientstat.utils.Logger;

import java.util.Map;

/**
 * 统计逻辑代理
 * <p>
 * Created by joye on 2017/9/24.
 */

public class StatProxy {
    private static final String TAG = "StatProxy";
    private com.qfpay.clientstat.config.StatConfig statConfig = com.qfpay.clientstat.config.StatConfig.getInstance();
    //事件处理器
    private EventHandler eventHandler;
    //存放时长事件统计
    private ArrayMap<String, Long> hashMap = new ArrayMap<>();

    StatProxy() {
        this.eventHandler = new DefaultEventHandler();
    }

    void setStatConfig(com.qfpay.clientstat.config.StatConfig statConfig) {
        this.statConfig = statConfig;
    }

    /**
     * 基础统计方法
     *
     * @param context
     * @param event   统计事件
     * @param params  额外参数
     */
    void onEvent(Context context, String event, Map<String, String> params) {
        if (null == context) {
            Logger.e(TAG, "unexpected null context in onEvent");
            return;
        }
        eventHandler.handleEvent(context, createEvent(context, event, params, 1, Event.EVENT_TYPE_COUNT), getStatConfig());
    }

    /**
     * 时长统计开始
     * see {@link #onEventEnd}
     *
     * @param context
     * @param event   事件Id (唯一)
     */
    void onEventBegin(Context context, String event) {
        if (null == context) {
            Logger.e(TAG, ("unexpected null context in onEventBegin"));
            return;
        }
        addToMap(event, System.currentTimeMillis());
    }


    /**
     * 时长统计结束
     *
     * @param context
     * @param event   事件key
     * @param params  额外参数
     */
    void onEventEnd(Context context, String event, Map<String, String> params) {
        Long start = removeFromMap(event);

        if (start == null || start <= 0) {
            Logger.w(TAG, "please call onEventBegin '" + event + "' before onPageEnd");
            return;
        }
        long duration = System.currentTimeMillis() - start;

        if (duration <= Integer.MAX_VALUE && duration >= Integer.MIN_VALUE) {
            eventHandler.handleEvent(context, createEvent(context, event, params, duration, Event.EVENT_TYPE_CALCULATE), getStatConfig());
        } else {
            Logger.w(TAG, "the event " + event + " time consuming is not accurate " + duration + " !");
        }
    }

    /**
     * 计算事件
     *
     * @param context
     * @param event
     * @param params
     * @param value
     */
    void onEventValue(Context context, String event, Map<String, String> params, int value) {
        eventHandler.handleEvent(context, createEvent(context, event, params, value, Event.EVENT_TYPE_CALCULATE), getStatConfig());
    }

    /**
     * 页面暂停
     *
     * @param context
     */
    void onPause(Context context) {
        //// TODO: 2017/9/26 need to implement
    }

    /**
     * 页面恢复
     *
     * @param context
     */
    void onResume(Context context) {
        //// TODO: 2017/9/26 need to implement
    }

    public com.qfpay.clientstat.config.StatConfig getStatConfig() {
        return statConfig;
    }

    private Event createEvent(Context context, String key, Map<String, String> params, long value, int type) {
        Event e = new Event(context);
        e.setEvent_key(key);
        e.setParams(params);
        e.setEvent_type(type);
        if (type == Event.EVENT_TYPE_COUNT) {
            e.setEvent_value(1);
        } else {
            e.setEvent_value(value);
        }
        e.setEvent_time(System.currentTimeMillis());
        e.getDeviceInfo().setLnglat(statConfig.getLnglat());
        e.getDeviceInfo().setUser_id(statConfig.getUserId());
        e.getDeviceInfo().setTime_diff(statConfig.getTimeDiff());
        return e;
    }

    /**
     * 添加到map 用于onEventBegin， onResume 统计
     * 用于页面时长统计和事件开始统计
     *
     * @param key
     * @param value
     */
    private void addToMap(String key, long value) {
        synchronized (hashMap) {
            hashMap.put(key, value);
        }
    }

    /**
     * 从map中移除
     *
     * @param key
     * @return
     */
    private Long removeFromMap(String key) {
        synchronized (hashMap) {
            return hashMap.remove(key);
        }
    }

    public EventHandler getEventHandler() {
        return eventHandler;
    }
}
