package com.qfpay.clientstat.internal;

import android.content.Context;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.os.Process;

import com.qfpay.clientstat.StatProxy;
import com.qfpay.clientstat.event.Event;
import com.qfpay.clientstat.upload.UploadStrategy;
import com.qfpay.clientstat.utils.DeviceUtil;
import com.qfpay.clientstat.utils.Logger;

import java.io.File;
import java.io.IOException;
import java.util.List;

import in.joye.urlconnection.client.Response;

/**
 * 默认事件处理器
 * <p>
 * Created by joye on 2017/9/21.
 */

public class DefaultEventHandler implements EventHandler {
    private static final String TAG = "DefaultEventHandler";
    private static final String THREAD_NAME_WRITE_FILE = "client-stat-write-file-thread";
    private Handler mEventHandler;
    private EventWriter eventWriter = null;

    public DefaultEventHandler() {
        Logger.i(TAG, "create new DefaultEventHandler instance.");
    }

    @Override
    public void handleEvent(Context context, Event event, StatProxy statProxy) {
        if (this.mEventHandler == null) {
            createEventHandler(context, statProxy);
        }
        Message eventMsg = mEventHandler.obtainMessage();
        eventMsg.obj = event;
        mEventHandler.sendMessage(eventMsg);
    }

    @Override
    public EventWriter getEventWriter() {
        return eventWriter;
    }

    /**
     * 事件写入处理器
     */
    static class EventWriteHandler extends Handler {
        private EventWriter eventWriter;
        private List<UploadStrategy> uploadStrategyList;
        private Context context;

        public EventWriteHandler(Looper looper, Context context, StatProxy statProxy, EventWriter eventWriter) {
            super(looper);
            this.context = context;
            this.eventWriter = eventWriter;
            this.uploadStrategyList = statProxy.getStatConfig().getUploadStrategies();
        }

        @Override
        public void handleMessage(Message msg) {
            try {
                Event event = (Event) msg.obj;
                if (event == null) {
                    Logger.e(TAG, "receive empty msg, and the event is null, just return.");
                    return;
                }
                File eventFile = eventWriter.writeEvent(event);
                for (UploadStrategy uploadStrategy : uploadStrategyList) {
                    boolean satisfied = uploadStrategy.isSatisfied(eventFile);
                    if (satisfied) {
                        eventWriter.newFile();
                        Logger.i(TAG, "【%s】the event file {'%s'} is satisfied upload condition {'%s'}, upload now.", DeviceUtil.getProcessName(context), eventFile.getAbsoluteFile(), uploadStrategy.getClass().getSimpleName());
                        uploadEventFile(eventFile);
                        break;
                    }
                }
            } catch (IOException e) {
                Logger.e(TAG, "write event to file failed, the error is %s.", e.getMessage());
            }
            super.handleMessage(msg);
        }

        private void uploadEventFile(final File eventFile) {
            //检查文件合法性
            if (eventFile == null || !eventFile.exists()) {
                Logger.e(TAG, "uploadEventFile(): the event file is null or not exists, just return.");
                return;
            }
            //检查网络状况
            if (!DeviceUtil.isNetAvailable(context)) {
                Logger.e(TAG, "uploadEventFile(): the current net is not available, just return.");
                return;
            }
            //开始上传文件
            Logger.i(TAG, "start upload event file {'%s'}", eventFile.getAbsoluteFile());
            EventFileUploader.uploadEventFileAsync(eventFile, new in.joye.urlconnection.client.Callback<UploadResult>() {
                @Override
                public void success(UploadResult uploadResult, Response response) {
                    Logger.i(TAG, "upload file success, the result is %s.", uploadResult);
                    if (uploadResult != null && uploadResult.isUploadSuccess()) {
                        try {
                            eventWriter.deleteFile(eventFile);
                            Logger.i(TAG, "delete event file success.");
                        } catch (IOException e) {
                            e.printStackTrace();
                            Logger.i(TAG, "delete event file fail.");
                        }
                    }
                }

                @Override
                public void failure(int statusCode, String error) {
                    Logger.e(TAG, "upload file failed, the reason is %s", error);
                }
            });
        }
    }

    private void createEventHandler(Context context, StatProxy statProxy) {
        if (mEventHandler == null) {
            synchronized (DefaultEventHandler.class) {
                if (mEventHandler == null) {
                    eventWriter = new EventFileWriter(context, statProxy);
                    HandlerThread handlerThread = new HandlerThread(THREAD_NAME_WRITE_FILE, Process.THREAD_PRIORITY_BACKGROUND);
                    handlerThread.start();
                    mEventHandler = new EventWriteHandler(handlerThread.getLooper(), context, statProxy, eventWriter);
                }
            }
        }
    }
}
