package org.qas.qtest.api.services.user;

import org.qas.api.AuthClientException;
import org.qas.api.AuthServiceException;
import org.qas.api.ClientConfiguration;
import org.qas.api.Request;
import org.qas.api.internal.util.google.base.Throwables;
import org.qas.qtest.api.auth.DefaultQTestCredentialsProviderChain;
import org.qas.qtest.api.auth.QTestCredentials;
import org.qas.qtest.api.auth.QTestCredentialsProvider;
import org.qas.qtest.api.auth.StaticQTestCredentialsProvider;
import org.qas.qtest.api.internal.QTestApiWebServiceClient;
import org.qas.qtest.api.services.user.model.AssignToProjectRequest;
import org.qas.qtest.api.services.user.model.GetUserProfileRequest;
import org.qas.qtest.api.services.user.model.User;
import org.qas.qtest.api.services.user.model.UserProfile;
import org.qas.qtest.api.services.user.model.transform.BatchAssignUsersRequestMarshaller;
import org.qas.qtest.api.services.user.model.transform.GetUserProfileRequestMarshaller;
import org.qas.qtest.api.services.user.model.transform.ListUserJsonUnmarshaller;
import org.qas.qtest.api.services.user.model.transform.UserProfileJsonUnmarshaller;

import java.util.List;

/**
 * UserProfileServiceClient
 *
 * @author trongle
 * @since 1.4.2
 */
public class UserProfileServiceClient extends QTestApiWebServiceClient<UserProfileServiceClient>
  implements UserProfileService {
  /**
   * Constructs a new client to invoke service method on UserProfileService using
   * the default qTest credentials provider and default client configuration options.
   */
  public UserProfileServiceClient() {
    this(new DefaultQTestCredentialsProviderChain(), new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on UserProfileService using
   * the default qTest credentials provider and client configuration options.
   *
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to UserService
   */
  public UserProfileServiceClient(ClientConfiguration clientConfiguration) {
    this(new DefaultQTestCredentialsProviderChain(), clientConfiguration);
  }

  /**
   * Constructs a new client to invoke service method on UserProfileService using
   * the specified qTest credentials.
   *
   * @param credentials The qTest credentials which will provide
   *                    credentials to authenticate request with qTest services.
   */
  public UserProfileServiceClient(QTestCredentials credentials) {
    this(credentials, new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on UserProfileService using
   * the specified qTest credentials and client configuration options.
   *
   * @param credentials         The qTest credentials which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to UserProfileService
   */
  public UserProfileServiceClient(QTestCredentials credentials, ClientConfiguration clientConfiguration) {
    super(clientConfiguration);
    this.credentialsProvider = new StaticQTestCredentialsProvider(credentials);
    init();
  }

  /**
   * Constructs a new client to invoke service method on UserProfileService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   */
  public UserProfileServiceClient(QTestCredentialsProvider credentialsProvider) {
    this(credentialsProvider, new ClientConfiguration());
  }

  /**
   * Constructs a new client to invoke service method on UserService using
   * the specified qTest credentials provider and client configuration options.
   *
   * @param credentialsProvider The qTest credentials provider which will provide
   *                            credentials to authenticate request with qTest services.
   * @param clientConfiguration The client configuration options controlling how this
   *                            client connects to UserService
   */
  public UserProfileServiceClient(QTestCredentialsProvider credentialsProvider, ClientConfiguration clientConfiguration) {
    super(clientConfiguration);
    this.credentialsProvider = credentialsProvider;

    init();
  }

  @Override
  public UserProfile get(GetUserProfileRequest getUserProfileRequest) throws AuthServiceException {
    try {
      Request request = new GetUserProfileRequestMarshaller().marshall(getUserProfileRequest);
      return invoke(request, UserProfileJsonUnmarshaller.getInstance());
    } catch (Exception ex) {
      Throwables.propagateIfInstanceOf(ex, AuthClientException.class);
      throw new AuthClientException("An error occurs during get user profile from qTest service", ex);
    }
  }

  @Override
  public List<User> batchAssignUsers(AssignToProjectRequest assignToProjectRequest) throws AuthServiceException {
    try {
      Request request = new BatchAssignUsersRequestMarshaller().marshall(assignToProjectRequest);
      return invoke(request, ListUserJsonUnmarshaller.getInstance());
    } catch (Exception ex) {
      Throwables.propagateIfInstanceOf(ex, AuthClientException.class);
      throw new AuthClientException("An error occurs during assigning user to project", ex);
    }
  }
}
