package org.qas.qtest.api.services.execution.model;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.qas.qtest.api.internal.model.FieldValue;
import org.qas.qtest.api.internal.model.QTestBaseModel;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * TestSuite
 *
 * @author Dzung Nguyen
 * @version $Id TestSuite 2014-03-28 07:07:30z dungvnguyen $
 * @since 1.0
 */
public class TestSuite extends QTestBaseModel<TestSuite> {
  @JsonProperty("id")
  private Long id;

  @JsonProperty("name")
  private String name;

  @JsonProperty("pid")
  private String pid;

  @JsonProperty("description")
  private String description;

  @JsonProperty("order")
  private Integer order;

  @JsonProperty("properties")
  private List<FieldValue> fieldValues;

  public TestSuite() {
  }

  /**
   * @return the test suite identifier.
   */
  public Long getId() {
    return id;
  }

  /**
   * Sets the test suite identifier.
   *
   * @param id the given test suite id to set.
   */
  public TestSuite setId(Long id) {
    this.id = id;
    return this;
  }

  /**
   * Sets the test suite identifier.
   *
   * @param id the given test suite id to set.
   */
  public TestSuite withId(Long id) {
    setId(id);
    return this;
  }
  
  public String getPid() {
    return pid;
  }
  
  public TestSuite setPid(String pid) {
    this.pid = pid;
    return this;
  }
  
  public TestSuite withPid(String pid) {
    return setPid(pid);
  }
  
  /**
   * @return the test suite name.
   */
  public String getName() {
    return name;
  }

  /**
   * Sets test suite name.
   *
   * @param name the given test suite name to set.
   */
  public TestSuite setName(String name) {
    this.name = name;
    return this;
  }

  /**
   * Sets test suite name.
   *
   * @param name the given test suite name to set.
   */
  public TestSuite withName(String name) {
    setName(name);
    return this;
  }

  /**
   * @return the test suite description.
   */
  public String getDescription() {
    return description;
  }

  /**
   * Sets test suite description.
   *
   * @param description the given test suite description to set.
   */
  public TestSuite setDescription(String description) {
    this.description = description;
    return this;
  }

  /**
   * Sets test suite description.
   *
   * @param description the given test suite description to set.
   */
  public TestSuite withDescription(String description) {
    setDescription(description);
    return this;
  }

  /**
   * @return the test suite order.
   */
  public Integer getOrder() {
    return order;
  }

  /**
   * Sets the test suite order.
   *
   * @param order the given test suite order to set.
   */
  public TestSuite setOrder(Integer order) {
    this.order = order;
    return this;
  }

  /**
   * Sets the test suite order and return current test suite.
   *
   * @param order the given test suite order to set.
   * @return the current test suite instance.
   */
  public TestSuite withOrder(Integer order) {
    setOrder(order);
    return this;
  }

  /**
   * @return the list of field value objects.
   */
  public List<FieldValue> getFieldValues() {
    if (null == fieldValues) {
      return Collections.emptyList();
    }
    return fieldValues;
  }

  /**
   * Sets the list of field value objects.
   *
   * @param values the given field value objects to set.
   */
  public TestSuite setFieldValues(List<FieldValue> values) {
    this.fieldValues = new ArrayList<>();
    if (null == values) {
      return this;
    }
    for (FieldValue fieldValue : values) {
      addFieldValue(fieldValue);
    }
    return this;
  }

  /**
   * Sets the list of field value objects.
   *
   * @param values the given field value objects to set.
   * @return the test-suite instance.
   */
  public TestSuite withFieldValues(List<FieldValue> values) {
    setFieldValues(values);
    return this;
  }

  /**
   * Adds the property value to requirement and return itself.
   *
   * @param value the given field value to add.
   * @return the test-suite instance.
   */
  public TestSuite addFieldValue(FieldValue value) {
    if (value == null || value.getValue() == null) {
      return this;
    }
    if (fieldValues == null) {
      fieldValues = new ArrayList<>();
    }
    fieldValues.add(value);
    return this;
  }

  @Override
  protected TestSuite clone() {
    TestSuite that = new TestSuite();
    that.setPropertiesFrom(this);
    return that;
  }

  @Override
  public String elementName() {
    return "test-suite";
  }

  @Override
  public String jsonElementName() {
    return "test_suite";
  }
}
