package org.qas.qtest.api.services.design.model;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.qas.qtest.api.internal.model.FieldValue;
import org.qas.qtest.api.internal.model.QTestBaseModel;
import org.qas.qtest.api.services.attachment.model.Attachment;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * TestCase
 *
 * @author Dzung Nguyen
 * @version $Id TestCase 2014-03-28 07:06:30z dungvnguyen $
 * @since 1.0
 */
public class TestCase extends QTestBaseModel<TestCase> {
  @JsonProperty("id")
  private Long id;

  @JsonProperty("name")
  private String name;

  @JsonProperty("description")
  private String description;

  @JsonProperty("precondition")
  private String precondition;

  @JsonProperty("order")
  private Integer order;

  @JsonProperty("pid")
  private String pid;

  @JsonProperty("test_steps")
  private List<TestStep> testSteps;

  @JsonProperty("test_case_version_id")
  private Long testCaseVersionId;

  @JsonProperty("attachments")
  private List<Attachment> attachments;

  @JsonProperty("properties")
  private List<FieldValue> fieldValues;

  @JsonProperty("parent_id")
  private Long parentId;

  public TestCase() {
  }

  /**
   * @return the test case id.
   */
  public Long getId() {
    return id;
  }

  /**
   * Sets the test case id.
   *
   * @param id the given test case identifier to set.
   * @return the {@link TestCase} instance.
   */
  public TestCase setId(Long id) {
    this.id = id;
    return this;
  }

  /**
   * Sets the test case id.
   *
   * @param id the given test case identifier to set.
   * @return the {@link TestCase} instance.
   */
  public TestCase withId(Long id) {
    setId(id);
    return this;
  }

  /**
   * @return the test case name.
   */
  public String getName() {
    return name;
  }

  /**
   * Sets the test case name.
   *
   * @param name the given test case name to set.
   * @return the {@link TestCase} instance.
   */
  public TestCase setName(String name) {
    this.name = name;
    return this;
  }

  /**
   * Sets the test case name.
   *
   * @param name the given test case name to set.
   * @return the {@link TestCase} instance.
   */
  public TestCase withName(String name) {
    setName(name);
    return this;
  }

  /**
   * @return the test case description.
   */
  public String getDescription() {
    return description;
  }

  /**
   * Sets the test case description.
   *
   * @param description the given test case description to set.
   * @return the {@link TestCase} instance.
   */
  public TestCase setDescription(String description) {
    this.description = description;
    return this;
  }

  /**
   * Sets the test case description.
   *
   * @param description the given test case description to set.
   * @return the {@link TestCase} instance.
   */
  public TestCase withDescription(String description) {
    setDescription(description);
    return this;
  }

  /**
   * @return the test case precondition.
   */
  public String getPrecondition() {
    return precondition;
  }

  /**
   * Sets the test case precondition.
   *
   * @param precondition the given test case precondition.
   * @return the {@link TestCase} instance.
   */
  public TestCase setPrecondition(String precondition) {
    this.precondition = precondition;
    return this;
  }

  /**
   * Sets the test case precondition.
   *
   * @param precondition the given test case precondition.
   * @return the {@link TestCase} instance.
   */
  public TestCase withPrecondition(String precondition) {
    setPrecondition(precondition);
    return this;
  }

  /**
   * @return the test case object order.
   */
  public Integer getOrder() {
    return order;
  }

  /**
   * Sets the test case object order.
   *
   * @param order the given test case object order.
   * @return the {@link TestCase} instance.
   */
  public TestCase setOrder(Integer order) {
    this.order = order;
    return this;
  }

  /**
   * Sets the test case object order.
   *
   * @param order the given test case object order.
   * @return this
   */
  public TestCase withOrder(Integer order) {
    setOrder(order);
    return this;
  }

  /**
   * @return the format id of test case.
   */
  public String getPid() {
    return pid;
  }

  /**
   * Sets the format id of test case.
   *
   * @param pid the given format id of test case to set.
   * @return this
   */
  public TestCase setPid(String pid) {
    this.pid = pid;
    return this;
  }

  /**
   * Sets the format id of test case.
   *
   * @param pid the given format id of test case to set.
   * @return this
   */
  public TestCase withPid(String pid) {
    setPid(pid);
    return this;
  }

  /**
   * @return the list of test steps.
   */
  public List<TestStep> getTestSteps() {
    if (null == testSteps) {
      return Collections.emptyList();
    }
    return testSteps;
  }

  /**
   * Sets the list of test steps.
   *
   * @param testSteps the given list of test steps to set.
   * @return this
   */
  public TestCase setTestSteps(List<TestStep> testSteps) {
    this.testSteps = testSteps;
    return this;
  }

  /**
   * Sets the list of test steps.
   *
   * @param testSteps the given list of test steps to set.
   * @return this
   */
  public TestCase withTestSteps(List<TestStep> testSteps) {
    setTestSteps(testSteps);
    return this;
  }

  /**
   * Adds test step to test case.
   *
   * @param testStep the given test step.
   * @return the {@link TestCase} object.
   */
  public TestCase addTestStep(TestStep testStep) {
    if (testSteps == null) {
      testSteps = new ArrayList<>();
    }
    testSteps.add(testStep);
    return this;
  }

  /**
   * @return the test case version id.
   */
  public Long getTestCaseVersionId() {
    return testCaseVersionId;
  }

  /**
   * Sets the test case version id.
   *
   * @param testCaseVersionId the given test case version id to set.
   * @return this
   */
  public TestCase setTestCaseVersionId(Long testCaseVersionId) {
    this.testCaseVersionId = testCaseVersionId;
    return this;
  }

  /**
   * Sets the test case version id.
   *
   * @param testCaseVersionId the given test case version id to set.
   * @return this
   */
  public TestCase withTestCaseVersionId(Long testCaseVersionId) {
    setTestCaseVersionId(testCaseVersionId);
    return this;
  }

  /**
   * @return the list of testcase's attachments.
   */
  public List<Attachment> getAttachments() {
    if (null == attachments) {
      return Collections.emptyList();
    }
    return attachments;
  }

  /**
   * Sets the list of testcase's attachments.
   *
   * @param attachments the given list of testcase's attachment to set.
   * @return this
   */
  public TestCase setAttachments(List<Attachment> attachments) {
    this.attachments = attachments;
    return this;
  }

  /**
   * Sets the list of testcase's attachments.
   *
   * @param attachments the given list of testcase's attachment to set.
   * @return this
   */
  public TestCase withAttachments(List<Attachment> attachments) {
    setAttachments(attachments);
    return this;
  }

  /**
   * Adds attachment to test case.
   *
   * @param attachment the given attachment to add to test case.
   * @return the {@link TestCase} object.
   */
  public TestCase addAttachment(Attachment attachment) {
    if (attachments == null) {
      attachments = new ArrayList<>();
    }
    attachments.add(attachment);
    return this;
  }

  /**
   * @return the list of field value objects.
   */
  public List<FieldValue> getFieldValues() {
    if (fieldValues == null) {
      return Collections.emptyList();
    }
    return fieldValues;
  }

  /**
   * Sets the list of field value objects.
   *
   * @param values the given field value objects to set.
   * @return this
   */
  public TestCase setFieldValues(List<FieldValue> values) {
    this.fieldValues = new ArrayList<>();
    if (null == values) {
      return this;
    }
    for (FieldValue fieldValue : values) {
      addFieldValue(fieldValue);
    }
    return this;
  }

  /**
   * Sets the list of field value objects.
   *
   * @param values the given field value objects to set.
   * @return the test-case instance.
   */
  public TestCase withFieldValues(List<FieldValue> values) {
    setFieldValues(values);
    return this;
  }

  /**
   * Adds the property value to requirement and return itself.
   *
   * @param value the given field value to add.
   * @return the test-case instance.
   */
  public TestCase addFieldValue(FieldValue value) {
    if (value == null || value.getValue() == null) {
      return this;
    }
    if (fieldValues == null) {
      fieldValues = new ArrayList<>();
    }
    fieldValues.add(value);
    return this;
  }

  /**
   * @return the parent identifier.
   */
  public Long getParentId() {
    return parentId;
  }

  /**
   * Sets the test-case parent identifier.
   *
   * @param parentId the given test-case parent identifier value to set.
   * @return this
   */
  public TestCase setParentId(Long parentId) {
    this.parentId = parentId;
    return this;
  }

  /**
   * Sets the test-case parent identifier.
   *
   * @param parentId the given parent identifier value to set.
   * @return the test-case instance.
   */
  public TestCase withParentId(Long parentId) {
    setParentId(parentId);
    return this;
  }

  @Override
  protected TestCase clone() {
    TestCase that = new TestCase();

    that.setPropertiesFrom(this);

    return that;
  }

  @Override
  public String elementName() {
    return "test-case";
  }

  @Override
  public String jsonElementName() {
    return "test_case";
  }
}
