package org.qas.qtest.api.internal.model;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.qas.api.http.HttpMethod;
import org.qas.api.internal.PropertyContainer;

/**
 * Link
 *
 * @author Dzung Nguyen
 * @version $Id Link 2014-03-28 04:46:30z dungvnguyen $
 * @since 1.0
 */
public class Link extends PropertyContainer {
  @JsonProperty("rel")
  private String relation;

  @JsonProperty("href")
  private String href;

  @JsonIgnore
  private HttpMethod method;

  /**
   * Constructs {@link Link} object.
   */
  public Link() {
    setRelation("self");
    setMethod(HttpMethod.GET);
  }

  /**
   * @return the link relation.
   */
  public String getRelation() {
    return relation;
  }

  /**
   * Sets the link relation.
   *
   * @param relation the given link relation to set.
   * @return the current instance.
   */
  public Link setRelation(String relation) {
    this.relation = relation;
    return this;
  }

  /**
   * Sets the link relation and return itself.
   *
   * @param relation the given link relation to set.
   * @return the current {@link Link} object.
   */
  public Link withRelation(String relation) {
    setRelation(relation);
    return this;
  }

  /**
   * @return the HTTP method required for the related call.
   */
  @JsonIgnore
  public HttpMethod getMethod() {
    return method;
  }

  /**
   * Sets the HTTP method required for the related call.
   *
   * @param method the HTTP method to set.
   * @return the current instance.
   */
  public Link setMethod(HttpMethod method) {
    this.method = (method == null ? HttpMethod.GET : method);
    return this;
  }

  /**
   * Sets the HTTP method required for the related call.
   *
   * @param method the HTTP method to set.
   * @return the current {@link Link} object.
   */
  public Link withMethod(HttpMethod method) {
    setMethod(method);
    return this;
  }

  /**
   * @return the URL of related HATEOAS link you can use for subsequent call.
   */
  public String getHref() {
    return href;
  }

  /**
   * Sets the URL of related HATEOAS link.
   *
   * @param href the given URL of related HATEOAS link to set.
   * @return the current instance.
   */
  public Link setHref(String href) {
    this.href = href;
    return this;
  }

  /**
   * Sets the URL of related HATEOAS link.
   *
   * @param href the given URL of related HATEOAS link to set.
   * @return current {@link Link} object.
   */
  public Link withHref(String href) {
    setHref(href);
    return this;
  }

  @Override
  public Link clone() {
    Link that = new Link();

    // copy current property.
    that.setPropertiesFrom(this);

    return that;
  }

  @Override
  @JsonIgnore
  public String elementName() {
    return "link";
  }
}
