package org.qas.api.http;

import org.qas.api.ApiServiceRequest;
import org.qas.api.Request;
import org.qas.api.internal.util.google.base.Preconditions;
import org.qas.api.internal.util.google.base.Strings;

import java.io.InputStream;
import java.net.URI;
import java.util.Collections;
import java.util.Map;

/**
 * AbstractHttpRequest
 *
 * @author Dzung Nguyen
 * @version $Id AbstractHttpRequest 2014-03-27 11:44:30z dungvnguyen $
 * @since 1.0
 */
public abstract class AbstractHttpRequest<T> implements HttpRequest<T> {
  //~ class properties ========================================================
  protected final Request request;

  //~ class members ===========================================================
  protected AbstractHttpRequest(Request request) {
    Preconditions.checkNotNull(request);
    this.request = request;
  }

  @Override
  public HttpMethod getHttpMethod() {
    return request.getHttpMethod();
  }

  @Override
  public void setHttpMethod(HttpMethod httpMethod) {
    request.setHttpMethod(httpMethod);
  }

  @Override
  public InputStream getContent() {
    return request.getContent();
  }

  @Override
  public void setContent(InputStream content) {
    request.setContent(content);
  }

  @Override
  public int getTimeOffset() {
    return request.getTimeOffset();
  }

  @Override
  public void setTimeOffset(int timeOffset) {
    request.setTimeOffset(timeOffset);
  }

  @Override
  public String getResourcePath() {
    return request.getResourcePath();
  }

  @Override
  public void setResourcePath(String resourcePath) {
    request.setResourcePath(resourcePath);
  }

  @Override
  public String getServiceName() {
    return request.getServiceName();
  }

  @Override
  public URI getEndpoint() {
    return request.getEndpoint();
  }

  @Override
  public Map<String, String> getHeaders() {
    return Collections.unmodifiableMap(request.getHeaders());
  }

  @Override
  public void setHeader(String name, String value) {
    request.setHeader(name, value);
  }

  @Override
  public void setHeaders(Map<String, String> headers) {
    request.setHeaders(headers);
  }

  @Override
  public Map<String, String> getParameters() {
    return Collections.unmodifiableMap(request.getParameters());
  }

  @Override
  public void setParameters(Map<String, String> parameters) {
    request.setParameters(parameters);
  }

  @Override
  public void addParameter(String name, String value) {
    request.addParameter(name, value);
  }

  @Override
  public ApiServiceRequest getOriginalRequest() {
    return request.getOriginalRequest();
  }

  @Override
  public String toString() {
    StringBuilder builder = new StringBuilder("<http-request>\n");

    // build request object.
    builder.append("\t<service-name value=\"").append(getServiceName()).append("\"/>\n");
    builder.append("\t<http-method value=\"").append(getHttpMethod().name()).append("\"/>\n");
    builder.append("\t<endpoint value=\"").append(getEndpoint().toString()).append("\"/>\n");
    builder.append("\t<content-type value=\"").append(getContentType()).append("\"/>\n");
    builder.append("\t<content-length value=\"").append(getContentLength()).append("\"/>\n");

    // append resource path.
    String resourcePath = getResourcePath();
    if (!Strings.isNullOrEmpty(resourcePath)) {
      builder.append("\t<resource-path value=\"").append(resourcePath).append("\"/>\n");
    }

    builder.append("\t<time-offset value=\"").append(getTimeOffset()).append("\"/>\n");
    builder.append("\t<headers>").append(getHeaders().toString()).append("</headers>\n");
    builder.append("\t<parameters>").append(getParameters().toString()).append("</parameters>\n");

    builder.append("</http-request>");

    return builder.toString();
  }
}
