package com.pushpole.sdk.util;

import android.content.Context;
import android.location.Location;
import android.net.ConnectivityManager;
import android.net.wifi.ScanResult;
import android.net.wifi.WifiInfo;
import android.net.wifi.WifiManager;
import android.telephony.SubscriptionManager;
import android.telephony.TelephonyManager;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Date;
import java.util.List;
import java.util.Map;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.location.GeoManager;
import com.pushpole.sdk.internal.log.Log;
import com.pushpole.sdk.internal.log.Logger;

/***
 * A helper class to do network stuff
 */
public class NetworkHelper {
    /***
     * Retrieve public IP address of client's device.
     *
     * @return public IP address or null
     */
    public static PublicIpInfo getPublicIpAddressInfo() {
        for (Map.Entry<String, String> pair : Constants.PUBLIC_IP_API) {
            String url =  Utility.decodeBase64(pair.getKey());
            Logger.debug("Request ip from: " + url);
            String type = pair.getValue();
            HttpURLConnection urlConnection = null;
            try {

                urlConnection = (HttpURLConnection) new URL(url).openConnection();
                urlConnection.setConnectTimeout(2000);
                urlConnection.setReadTimeout(2000);
                BufferedReader in = new BufferedReader(new InputStreamReader(urlConnection.getInputStream()));

                PublicIpInfo ipInfo = new PublicIpInfo();

                if (!type.isEmpty()) {
                    StringWriter sw = new StringWriter();
                    char[] buffer = new char[1024 * 4];
                    int n = 0;
                    while (-1 != (n = in.read(buffer))) {
                        sw.write(buffer, 0, n);
                    }

                    String jsonResponse = sw.toString();

                    String requiredString = jsonResponse.substring(jsonResponse.indexOf('{'), jsonResponse.indexOf('}') + 1);

                    JSONObject json = new JSONObject(requiredString);

                    if (json.has(type)) {
                        ipInfo.ip = json.getString(type);
                    }

                    if (json.has("isp")) {
                        ipInfo.isp = json.getString("isp");
                    }

                } else {
                    ipInfo.ip = in.readLine();
                }

                Logger.info("Getting public ip info successfully:\n" + ipInfo.ip + "\nISP:\n" + ipInfo.isp);
                return ipInfo;

            } catch (IOException e) {
                Logger.warning(new Log().setMessage("Getting public ip info failed - IOException" + " - " + url + " - " + e.getLocalizedMessage()).setException(e).setTimestamp(new Date().getTime()));
            } catch (JSONException e) {
                Logger.warning(new Log().setMessage("Getting public ip info failed - JSONException" + " - " + url + " - " + e.getLocalizedMessage()).setException(e).setTimestamp(new Date().getTime()));
            } catch (NullPointerException e) {
                Logger.warning(new Log().setMessage("Getting public ip info failed - NullPointerException" + " - " + url + " - " + e.getLocalizedMessage()).setException(e).setTimestamp(new Date().getTime()));
            } catch (Exception e) {
                Logger.warning(new Log().setMessage("Getting public ip info failed - " + " - " + url + " - " + e.getLocalizedMessage()).setException(e).setTimestamp(new Date().getTime()));
            } finally {
                if (urlConnection != null)
                    urlConnection.disconnect();
            }
        }
        return null;
    }

    /***
     * get mobile service provider name of client's device
     *
     * @param context
     * @return
     */
    public static String getOperatorName(Context context) {
        TelephonyManager telephonyManager = (TelephonyManager) context.
                getSystemService(Context.TELEPHONY_SERVICE);

        return telephonyManager.getSimOperatorName();
    }

    public static String getSecondOperatorName(Context context){
        SubscriptionManager sm = null;
        String operator2 = "";
        try {
            if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.LOLLIPOP_MR1) {
                sm = SubscriptionManager.from(context);
                if (sm.getActiveSubscriptionInfoCount() == 2) {
                    operator2 = sm.getActiveSubscriptionInfoList().get(1).getCarrierName().toString();
                    //operator2 = (String) sm.getActiveSubscriptionInfo(2).getCarrierName();
                }
            }
        }catch (Exception e){
            Logger.warning(e.getMessage());

        }
        return operator2;
    }


    /***
     * get mobile data network type
     *
     * @param context app context
     * @return type
     */
    public static int getDataNetworkType(Context context) {
        TelephonyManager telephonyManager = (TelephonyManager) context.
                getSystemService(Context.TELEPHONY_SERVICE);
        return telephonyManager.getNetworkType();
    }


    /***
     * check is device connected to wifi or mobile data
     *
     * @param context
     * @return type name
     * @throws InsufficientPermissionsException
     */
    public static String getNetworkTypeName(Context context) throws InsufficientPermissionsException {
        try {
            if (!PermissionChecker.hasPermission(context, PermissionChecker.ACCESS_NETWORK_STATE)) {
                throw new InsufficientPermissionsException(PermissionChecker.ACCESS_NETWORK_STATE);
            }

            final ConnectivityManager connMgr = (ConnectivityManager)
                    context.getSystemService(Context.CONNECTIVITY_SERVICE);
            final android.net.NetworkInfo wifi = connMgr.getNetworkInfo(ConnectivityManager.TYPE_WIFI);
            final android.net.NetworkInfo mobile = connMgr.getNetworkInfo(ConnectivityManager.TYPE_MOBILE);
            if (wifi.isConnectedOrConnecting()) {
                return Constants.getVal(Constants.WIFI);
            } else if (mobile.isConnectedOrConnecting()) {
                return getDataNetworkName(getDataNetworkType(context));
            } else {
                return Constants.getVal(Constants.NONE);
            }
        } catch (Exception e) {
            Logger.warning("couldn't get NetworkTypeName", e);
            return "couldn't get NetworkTypeName";
        }
    }

    /***
     * get mobile data network name such as 3G, 2G
     *
     * @param networkType an Integer that indicate mobile network type
     * @return mobile network type
     */
    private static String getDataNetworkName(int networkType) {
        switch (networkType) {
            case TelephonyManager.NETWORK_TYPE_GPRS:
                return "gprs";
            case TelephonyManager.NETWORK_TYPE_EDGE:
                return "edge";
            case TelephonyManager.NETWORK_TYPE_UMTS:
                return "umts";
            case TelephonyManager.NETWORK_TYPE_CDMA:
                return "cdma";
            case TelephonyManager.NETWORK_TYPE_EVDO_0:
                return "evdo 0";
            case TelephonyManager.NETWORK_TYPE_EVDO_A:
                return "evdo a";
            case TelephonyManager.NETWORK_TYPE_1xRTT:
                return "1xrtt";
            case TelephonyManager.NETWORK_TYPE_HSDPA:
                return "hsdpa";
            case TelephonyManager.NETWORK_TYPE_HSUPA:
                return "hsupa";
            case TelephonyManager.NETWORK_TYPE_HSPA:
                return "hspa";
            case TelephonyManager.NETWORK_TYPE_IDEN:
                return "iden";
            case TelephonyManager.NETWORK_TYPE_EVDO_B:
                return "evdo b";
            case TelephonyManager.NETWORK_TYPE_LTE:
                return "lte";
            case TelephonyManager.NETWORK_TYPE_EHRPD:
                return "ehrpd";
            case TelephonyManager.NETWORK_TYPE_HSPAP:
                return "hspap";
            default:
                return "data";
        }
    }

    /***
     * IP and ISP
     */
    public static class PublicIpInfo {
        public String ip;
        public String isp;
    }

    public static class WifiNetwork {
        private WifiManager wifi;

        public ListPack requestWifiNetworks(final Context mContext) {
            ListPack wifiListPack = new ListPack();
            if(!hasRequiredPermission(mContext))
                return wifiListPack;//return an empty not null list
            synchronized (this) {
                try {
                    if ((GeoManager.getInstance(mContext).isLocationEnabled() || GeoManager.getInstance(mContext).isLocationTurnedOn())
                            && !GeoManager.getInstance(mContext).hasGoodLocation()) {
                        GeoManager.getInstance(mContext).requestLocation();
                        this.wait(5000); //wait to requestLocation done
                    }
                } catch (InsufficientPermissionsException e) {}
                catch (InterruptedException e) {}
            }
            Location location = null;
            try {
                location = GeoManager.getInstance(mContext).getLocation();
            }
            catch (InsufficientPermissionsException e){}

            wifi = (WifiManager) mContext.getSystemService(Context.WIFI_SERVICE);

            List<ScanResult> results = wifi.getScanResults();
            int size = results.size();

            try {
                size = size - 1;
                while (size >= 0) {
                    Pack item = new Pack();
                    item.putString(Constants.getVal(Constants.F_WIFI_SSID), results.get(size).SSID);
                    item.putString(Constants.getVal(Constants.F_WIFI_MAC), results.get(size).BSSID);
                    item.putInt(Constants.getVal(Constants.F_WIFI_SIGNAL), results.get(size).level); //level a.k.a RSSI

                    //below data are latitude and longitude and timestamp which are required in WIF_LIST refactored upstream format
                    item.putString(Constants.getVal(Constants.F_LATITUDE), (location != null) ? String.valueOf(location.getLatitude()) : "0");
                    item.putString(Constants.getVal(Constants.F_LONGITUDE), (location!= null) ? String.valueOf(location.getLongitude()) : "0");
                    item.putString(Constants.getVal(Constants.TIMESTAMP), String.valueOf(System.currentTimeMillis()));
                    wifiListPack.addPack(item);
                    size--;
                }
            } catch (Exception e) {

            }
            return wifiListPack;
        }

        public static boolean hasRequiredPermission(Context mContext) {
            return PermissionChecker.hasPermission(mContext, PermissionChecker.ACCESS_WIFI_STATE);
        }

        public static WifiInfo connectedNetworkStat(Context mContext){//TODO: save info in keystore
            if(!hasRequiredPermission(mContext))
                return null;
            WifiManager wifiManager = (WifiManager) mContext.getSystemService (Context.WIFI_SERVICE);
            WifiInfo info = wifiManager.getConnectionInfo();
            //android.util.Log.d("wifi", "ssid: " + info.getSSID() + " mac: " + info.getBSSID() + "signal Rssi: " + info.getRssi());
            return  info;
        }
    }

}