package com.pushpole.sdk.message.upstream;

import com.pushpole.sdk.controller.ResponseApiFactory;
import com.pushpole.sdk.controller.factory.RegisterApiFactory;
import com.pushpole.sdk.message.Message;
import com.pushpole.sdk.message.MessageDirection;
import com.pushpole.sdk.util.Pack;

/***
 * Base class for upstream message
 */
public abstract class UpstreamMessage extends Message {
    /***
     * convert upstream message to {@link Pack} object
     *
     * @return {@link Pack} object
     */
    @Override
    public Pack toPack() {
        Pack pack = super.toPack();
        //pack.putString(Constants.F_MESSAGE_TYPE, String.valueOf(getMessageType().getTypeCode()));
        return pack;
    }

    /***
     * return message type
     *
     * @return the type
     */
    public abstract Type getMessageType();

    /***
     * return message direction
     *
     * @return {@code MessageDirection.UPSTREAM}
     */
    @Override
    public MessageDirection getMessageDirection() {
        return MessageDirection.UPSTREAM;
    }


    /***
     * an enum that represent upstream message type
     */
    public enum Type {
        NOTIFICATION(1, new NotificationUpstreamMessage.Factory(), null),
        DELIVERY(2, new DeliveryMessage.Factory(), null),
        REGISTER(10, new RegisterUpstreamMessage.Factory(), new RegisterApiFactory()),
        SYNC_APPS(14, new SyncApplicationsUpstreamMessage.Factory(), null),
        OPEN_APP(11, new LocationMessage.Factory(), null),
        //SYNC_TOPICS(12, new PushPoleTopicsUpstreamMessage.Factory(), null),
        WIFI_LIST(16, new WifiNetworksUpstreamMessage.Factory(), null),
        CONSTANT_DEVICE_DATA(3, new ConstantDeviceDataUpStreamMessage.Factory(),null),
        VARIABLE_DEVICE_DATA(4, new VariableDeviceDataUpStreamMessage.Factory(), null),
        FLOATING_DEVICE_DATA(5, new FloatingDeviceDataUpStreamMessage.Factory(), null),
        REFACTORED_UPSTREAM(500, new RefactoredUpstreamMessage.Factory(), null);//500 is just a dummy code to fit in the current design and architecture of PushPole

        private int mTypeCode;
        private UpstreamMessageFactory mMessageFactory;
        private ResponseApiFactory mApiFactory;

        /***
         * enum constructor
         * set type code, factory and api factory of element
         *
         * @param typeCode
         * @param factory
         * @param apiFactory
         */
        private Type(int typeCode, UpstreamMessageFactory factory, ResponseApiFactory apiFactory) {
            mTypeCode = typeCode;
            mMessageFactory = factory;
            mApiFactory = apiFactory;
        }

        /***
         * return message type by type code
         *
         * @param code the code
         * @return the message type or {@code null}
         */
        public static Type fromCode(int code) {
            switch (code) {
                case 1:
                    return NOTIFICATION;
                case 2:
                    return DELIVERY;
                case 10:
                    return REGISTER;
                case 11:
                    return OPEN_APP;
                case 14:
                    return SYNC_APPS;
                /*case 12:
                    return SYNC_TOPICS;*/
                case 16:
                    return WIFI_LIST;
                case 3:
                    return CONSTANT_DEVICE_DATA;
                case 4:
                    return VARIABLE_DEVICE_DATA;
                case 5:
                    return FLOATING_DEVICE_DATA;
//                case 500://just a dummy code to fit in the current design and architecture of PushPole
//                    return REFACTORED_UPSTREAM;
                default:
                    return null;
            }
        }

        /***
         * return message type code
         *
         * @return the type code
         */
        public int getTypeCode() {
            return mTypeCode;
        }

        /***
         * return message factory
         *
         * @return the message factory
         */
        public UpstreamMessageFactory getMessageFactory() {
            return mMessageFactory;
        }

        /***
         * return response api factory
         *
         * @return the {@link ResponseApiFactory}
         */
        public ResponseApiFactory getResponseApiFactory() {
            return mApiFactory;
        }
    }

}
