package com.pushpole.sdk;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageManager;
import android.os.Bundle;

import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.LogLevel;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.util.PushPoleFailedException;

/***
 * A singleton class that contains client's  GCM variables and specification
 */
public class SenderInfo {
    public final static int NO_TOKEN = 0;
    public final static int TOKEN_RECEIVED = 1;
    public final static int TOKEN_SYNCED = 2;
    //public final static int TOKEN_IS_SYNCING = 3;
    public final static String SENDER_ID = "$sender_id";
    public final static String APP_TOKEN = "$app_token";
    public static final String TAG = SenderInfo.class.getSimpleName();
    private final static String INSTANCE_ID = "$instance_id";
    private final static String TOKEN = "$token";
    private final static String TOKEN_STATE = "$token_state";
    private volatile static SenderInfo mInstance;
    private static LogLevel logLevel = null;
    private Context mContext;

    private SenderInfo(Context context) throws NullPointerException {
        // Need to set context before setting attributes
        mContext = context;
    }

    /***
     * get single instance of {@link SenderInfo}, for using this class your should always get
     * instance of {@link SenderInfo} from this method
     *
     * @param context
     * @return single instance of {@link SenderInfo}
     */
    public static SenderInfo getInstance(Context context) {
        if (mInstance == null) {
            synchronized (SenderInfo.class) {
                if (mInstance == null) {
                    mInstance = new SenderInfo(context);
                }
            }
        }
        mInstance.mContext = context.getApplicationContext();
        return mInstance;
    }

    /**
     * @param mContext
     * @return
     * @throws PushPoleFailedException
     */

    public static String readSenderIdFromManifest(Context mContext) throws PushPoleFailedException {
        String token = null;
        try {
            ApplicationInfo ai = mContext.getPackageManager().getApplicationInfo(mContext.getPackageName(), PackageManager.GET_META_DATA);
            Bundle bundle = ai.metaData;
            token = bundle.getString(PlainConstants.CONFIG_TOKENS);
        } catch (PackageManager.NameNotFoundException e) {
            throw new PushPoleFailedException("packageName " + mContext.getPackageName() + " can not be found");
        }

        if (token == null)
            throw new PushPoleFailedException("provided token is null");
        if (token.isEmpty())
            throw new PushPoleFailedException("provided token is empty");

        String senderId = null;

        try {
            if(token.contains("_")) { //checking for old format which was apptoken_senderId
                String[] tokens = token.split("_");
                senderId = tokens[1];
            }
           /* else
                senderId = token;*/
        } catch (Exception e) {
            throw new PushPoleFailedException("provided token problem - bad token - " + e.getLocalizedMessage());

        }

        if (senderId == null)
            throw new PushPoleFailedException("provided senderId is null");
        if (senderId.isEmpty())
            throw new PushPoleFailedException("provided senderId is empty");


        // remove any whitespace
        senderId = senderId.replaceAll("\\s+", "");

        // Check senderID is a valid Long number
        try {
            Long.parseLong(senderId);
        } catch (NumberFormatException e) {
            throw new PushPoleFailedException("provided senderId is not a valid integer");
        }
        return senderId;
    }

    /***
     * get a unique token that send from GCM to client when GCM registration is successful
     *
     * @return unique GCM token
     */
    public String getToken() {
        return KeyStore.getInstance(mContext).getString(TOKEN, null);
    }

    /***
     * set a unique token that send from GCM to client when GCM registration is successful
     *
     * @param token unique GCM token
     */
    public void setToken(String token) {
        setToken(mContext, token);
    }

    /***
     * set a unique token that send from GCM to client when GCM registration is successful
     *
     * @param context app context
     * @param token   unique GCM token
     */
    public void setToken(Context context, String token) {
        KeyStore.getInstance(context).putString(TOKEN, token);
    }

    /***
     * get tokenState that indicate registration status
     *
     * @return a token that indicate registration status
     */
    public int getTokenState() {
        return KeyStore.getInstance(mContext).getInt(TOKEN_STATE, NO_TOKEN);
    }

    /***
     * Set tokenState that indicate registration status
     *
     * @param tokenState a token that indicate registration status
     */
    public void setTokenState(int tokenState) {
        setTokenState(mContext, tokenState);
    }

    /***
     * Set tokenState that indicate registration status
     *
     * @param context    app context
     * @param tokenState a token that indicate registration status
     */
    public void setTokenState(Context context, int tokenState) {
        KeyStore.getInstance(context).putInt(TOKEN_STATE, tokenState);
    }

    public String getSenderIdSimple() {
        try {
            return getSenderId();
        } catch (PushPoleFailedException e) {

        }

        return "bad senderId";
    }

    /***
     * get instanceID from sharedPref
     *
     * @return a required unique ID for registering to GCM
     */
    public String getSenderId() throws PushPoleFailedException {
        String senderIdFromSharedPref = null;
        senderIdFromSharedPref = KeyStore.getInstance(mContext).getString(SENDER_ID, null);

        if (!tokenIsValid(senderIdFromSharedPref)) {
            String senderIdFromManifest = readSenderIdFromManifest(mContext);

            // store senderId in sharedPref
            setSenderId(senderIdFromManifest);

            // read again from sharedPref to check it's really stored
            senderIdFromSharedPref = KeyStore.getInstance(mContext).getString(SENDER_ID, null);
            if (!tokenIsValid(senderIdFromSharedPref)) {
                Logger.error("Check format of senderId from sharedPref failed - senderId is: " + senderIdFromSharedPref + " - set senderId with senderIdFromManifest", new LogData(
                        "senderIdFromManifest", senderIdFromManifest));
                return senderIdFromManifest;
            }
            Logger.debug("senderId successfully retrieved from manifest and sharedPref");
        }
        return senderIdFromSharedPref;
    }

    public static void updateSenderIdInKeystore()throws PushPoleFailedException {
        String senderIdFromManifest = readSenderIdFromManifest(mInstance.mContext);
        mInstance.setSenderId(senderIdFromManifest);
    }

    /***
     * set instanceID in sharedPref
     *
     * @param senderId a required unique ID for registering to GCM
     */
    public void setSenderId(String senderId) {
        KeyStore.getInstance(mContext).putString(SENDER_ID, senderId);
    }

    /***
     * get instanceID from sharedPref
     *
     * @return a required unique ID for registering to GCM
     */
    public String getInstanceId() {
        return KeyStore.getInstance(mContext).getString(INSTANCE_ID, null);
    }

    /***
     * Set instanceID
     *
     * @param instanceId a required unique ID for registering to GCM
     */
    public void setInstanceId(String instanceId) {
        setInstanceId(mContext, instanceId);
    }

    /***
     * set instanceID in sharedPref
     *
     * @param context
     * @param instanceId a required unique ID for registering to GCM
     */
    public void setInstanceId(Context context, String instanceId) {
        KeyStore.getInstance(mContext).putString(INSTANCE_ID, instanceId);
    }

    /***
     * set app token in sharedPref
     *
     * @param appToken a required unique token for registering to pushpole server
     */
    public void setAppToken(String appToken) {
        KeyStore.getInstance(mContext).putString(APP_TOKEN, appToken);

    }

    public String getAppId() {
        return mContext.getPackageName();
    }

    /***
     * check token already initialized
     *
     * @return
     */

    public boolean tokenIsValid(String token) {
        return ((token != null) && (!token.isEmpty()));
    }

    public boolean isRegistrationComplete() {
        return getTokenState() == TOKEN_SYNCED;
    }

    public LogLevel getLogCatLevel() {

        ApplicationInfo ai = null;
        try {
            ai = mContext.getPackageManager().getApplicationInfo(mContext.getPackageName(), PackageManager.GET_META_DATA);
        } catch (PackageManager.NameNotFoundException e) {
            e.printStackTrace();
            return null;
        }
        Bundle bundle = ai.metaData;
        String logCatLevel = bundle.getString("pushpole_log_level", "").toLowerCase();

        if ("debug".equals(logCatLevel)) {
            logLevel = LogLevel.DEBUG;
        } else if ("info".equals(logCatLevel)) {
            logLevel = LogLevel.INFO;
        } else if ("warning".equals(logCatLevel)) {
            logLevel = LogLevel.WARN;
        } else if ("error".equals(logCatLevel)) {
            logLevel = LogLevel.ERROR;
        } else if ("fatal".equals(logCatLevel)) {
            logLevel = LogLevel.FATAL;
        }

        return logLevel;
    }
}
