package com.pushpole.sdk.controller.controllers;

import android.content.Context;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.FirebaseAppNotAvailableException;
import com.pushpole.sdk.controller.DownstreamApiController;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.downstream.DownstreamMessage;
import com.pushpole.sdk.message.downstream.UpdateSubscriptionsDownstreamMessage;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.task.tasks.SubscribeTask;
import com.pushpole.sdk.topic.Topic;
import com.pushpole.sdk.topic.TopicStore;
import com.pushpole.sdk.topic.TopicSubscriber;
import com.pushpole.sdk.util.ListPack;
import com.pushpole.sdk.util.Pack;

/***
 * A class for handling {@link UpdateSubscriptionsDownstreamMessage}
 */
public class SubscriptionController implements DownstreamApiController {
    public final static String SUBSCRIBE = "\u0086\u0088u\u0086v\u0085|ux"; //"subscribe";
    public final static String UNSUBSCRIBE = "\u0088\u0081\u0086\u0088u\u0086v\u0085|ux"; //"unsubscribe";

    private Context mContext;

    public SubscriptionController(Context context) {
        mContext = context;
    }

    /**
     * handler method for subscribe and unsubscribe from channels that specified in downstream message
     * if some channels still remained for subscription and unsubscription, reschedule them to run in future
     *
     * @param downstreamMessage the message
     */
    @Override
    public void handleDownstreamMessage(DownstreamMessage downstreamMessage) {
        UpdateSubscriptionsDownstreamMessage message = (UpdateSubscriptionsDownstreamMessage) downstreamMessage;

        List<Topic> remainingSubscribe = attemptTopicSubscription(Constants.getVal(SUBSCRIBE), message.getSubscribeTo());
        List<Topic> remainingUnsubscribe = attemptTopicSubscription(Constants.getVal(UNSUBSCRIBE), message.getUnsubscribeFrom());

        if (remainingSubscribe.size() > 0) {
            Logger.warning("Subscribing topics failed, scheduling task");
            Pack data = new Pack();
            data.putString(Constants.getVal(Constants.TASK_ACTION), String.valueOf(Constants.getVal(SUBSCRIBE)));
            data.putString(Constants.getVal(Constants.TOPICS), topicListToString(remainingSubscribe));
            TaskManager.getInstance(mContext).scheduleTask(SubscribeTask.class, data);
        }

        if (remainingUnsubscribe.size() > 0) {
            Logger.warning("Unsubscribing topics failed, scheduling task");
            Pack data = new Pack();
            data.putString(Constants.getVal(Constants.TASK_ACTION), Constants.getVal((UNSUBSCRIBE)) ) ;
            data.putString(Constants.getVal(Constants.TOPICS), topicListToString(remainingUnsubscribe));
            TaskManager.getInstance(mContext).scheduleTask(SubscribeTask.class, data);
        }

    }

    /***
     * try to subscribe and unsubscribe from list of channels
     *
     * @param action {@code SUBSCRIBE} and {@code UNSUBSCRIBE}
     * @param topics list of channels
     * @return list of topics that can't be subscribed or unsubscribed
     */
    public List<Topic> attemptTopicSubscription(String action, List<Topic> topics) {
        TopicSubscriber subscriber = new TopicSubscriber(mContext);
        List<Topic> failedTopics = new ArrayList<>();

        for (Topic topic : topics) {
            try {
                if (action.equals(Constants.getVal(SUBSCRIBE))) {
                    Logger.info("Subscribing to topic", new LogData(
                            "Topic", topic.getCode()
                    ));

                    subscriber.subscribe(topic.getCode());
                    TopicStore.getInstance().addTopic(mContext, topic);
                } else if (action.equals(Constants.getVal(UNSUBSCRIBE))) {
                    Logger.info("Unsubscribing from topic", new LogData(
                            "Topic", topic.getCode()
                    ));

                    subscriber.unsubscribe(topic.getCode());
                    TopicStore.getInstance().removeTopic(mContext, topic);
                }
            } catch (IOException e) {
                Logger.warning("Topic subscription/unsubscription failed - " + e.getLocalizedMessage(), new LogData(
                        "Topic", topic.getCode()
                ));
                failedTopics.add(topic);
            } catch (IllegalArgumentException e) {
                Logger.warning("Error subscribing/unsubscribing to topic - " + e.getLocalizedMessage(), new LogData(
                        "Error", e.getMessage()
                ));
            } catch (FirebaseAppNotAvailableException e) {
                Logger.warning("Error subscribing/unsubscribing to topic - " + e.getLocalizedMessage(), new LogData(
                        "Error", e.getMessage()
                ));
            }
        }

        return failedTopics;
    }

    public String topicListToString(List<Topic> topics) {
        StringBuilder builder = new StringBuilder();
        for (int i = 0; i < topics.size(); i++) {
            if (i != 0) {
                builder.append(';');
            }
            builder.append(topics.get(i).getCode());
        }
        return builder.toString();
    }

    public List<Topic> topicListFromString(String topicsStr) {
        List<Topic> topics = new ArrayList<>();
        String[] parts = topicsStr.split(";");
        for (int i = 0; i < parts.length; i++) {
            topics.add(new Topic(parts[i]));
        }
        return topics;
    }

    private ListPack successfulSubUnsubTopics(List<Topic> allTopics, List<Topic> failedTopics, ListPack resList){
        if(resList == null)
            resList = new ListPack();

        for(Topic aTopic : allTopics){
            if(!failedTopics.contains(aTopic)){
                resList.addPack(aTopic.toPack());
            }
        }
        return resList;
    }
}
