package com.pushpole.sdk.controller.controllers;

import android.content.Context;
import android.os.Build;
import android.telephony.CellInfo;
import android.telephony.CellInfoCdma;
import android.telephony.CellInfoGsm;
import android.telephony.CellInfoLte;
import android.telephony.CellInfoWcdma;
import android.telephony.CellLocation;
import android.telephony.TelephonyManager;
import android.telephony.cdma.CdmaCellLocation;
import android.telephony.gsm.GsmCellLocation;

import java.util.List;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.controller.DownstreamApiController;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.message.downstream.DownstreamMessage;
import com.pushpole.sdk.message.upstream.UpstreamMessageFactory;
import com.pushpole.sdk.network.UpstreamSender;
import com.pushpole.sdk.task.PushPoleAsyncTask;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.util.ListPack;
import com.pushpole.sdk.util.Pack;

/**
 * Created on 17-01-04, 3:44 PM.
 *
 * @author Akram Shokri
 */
public class MobileCellInfoController implements DownstreamApiController {

    private static final int INVALID_VAL = Integer.MAX_VALUE;
    private static final int ASU_INVALID_VAL = 99;
    private Context mContext;

    public MobileCellInfoController(Context mContext) {
        this.mContext = mContext;
    }

    @Override
    public void handleDownstreamMessage(DownstreamMessage message) {
        Logger.info("Mobile cell request downstream message");
        sendCellInfo();
    }


    public Pack getCellInfo(){
        try {
            Pack floatingAndCellData = new FloatingDeviceDataController(mContext).getFloatingData();
            TelephonyManager telephonyManager = (TelephonyManager) mContext.getApplicationContext().getSystemService(mContext.getApplicationContext().TELEPHONY_SERVICE);
            int size = floatingAndCellData.toJson().length();

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                List<CellInfo> allCells = telephonyManager.getAllCellInfo();
                if (allCells != null && allCells.size() > 0) {
                    ListPack lp = new ListPack();
                    for (CellInfo cell : allCells) {
                        Pack pack = createPackFromCellArray(cell);
                        if((size + pack.toJson().length()) < Constants.MAX_PAYLOAD_SIZE) {//to avoid having data exceed 4k upstream limit
                            lp.addPack(pack);
                            size += pack.toJson().length();
                        }else
                            break;
                    }
                    floatingAndCellData.putListPack("cellsInfo", lp);
                    return floatingAndCellData;
                }
            }
            //if above conditions are not met, send below info
            ListPack lp = new ListPack();
            lp.add(telephonyManager.getCellLocation().toString());
            floatingAndCellData.putListPack("cellsInfo", lp);
            return floatingAndCellData;
        }catch (Exception e){
            //to avoid unwanted errors
            return null;
        }
    }

    public Pack getLocationOfConnectedCell(){
        Pack pack = new Pack();
        try {
            TelephonyManager telephonyManager = (TelephonyManager) mContext.getApplicationContext().getSystemService(mContext.getApplicationContext().TELEPHONY_SERVICE);

            CellLocation cellLocation = telephonyManager.getCellLocation();

            if(cellLocation !=  null){
                if(cellLocation instanceof GsmCellLocation){
                    GsmCellLocation gsm = (GsmCellLocation) cellLocation;
                    pack.putString(Constants.getVal(Constants.CELL_TYPE), "gsm");
                    pack.putInt(Constants.getVal(Constants.CELL_ID), gsm.getCid());
                    pack.putInt(Constants.getVal(Constants.CELL_LAC), gsm.getLac());
                    pack.putInt(Constants.getVal(Constants.CELL_PSC), gsm.getPsc());

                } else if (cellLocation instanceof CdmaCellLocation){
                    CdmaCellLocation cdma = (CdmaCellLocation) cellLocation;
                    pack.putString(Constants.getVal(Constants.CELL_TYPE), "cdma");
                    pack.putInt(Constants.getVal(Constants.CELL_ID), cdma.getBaseStationId());
                    pack.putInt(Constants.getVal(Constants.CELL_LAT), cdma.getBaseStationLatitude());
                    pack.putInt(Constants.getVal(Constants.CELL_LONG), cdma.getBaseStationLongitude());
                }
                return pack;
            }

            if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.JELLY_BEAN_MR1) {
                List<CellInfo> allCells = telephonyManager.getAllCellInfo();
                if (allCells != null && allCells.size() > 0) {
                    for (CellInfo cell : allCells) {
                        if (cell.isRegistered()) {
                            try {
                                CellInfoLte lte = (CellInfoLte) cell;
                                pack.putString(Constants.getVal(Constants.CELL_TYPE), "lte");
                                pack.putInt(Constants.getVal(Constants.CELL_ID), lte.getCellIdentity().getCi());
                                pack.putInt("tac", lte.getCellIdentity().getTac());
                                pack.putInt("pci", lte.getCellIdentity().getPci());

                            } catch (Exception e) {
                                try {
                                    CellInfoWcdma wcmda = (CellInfoWcdma) cell;
                                    pack.putString(Constants.getVal(Constants.CELL_TYPE), "wcmda");
                                    pack.putInt(Constants.getVal(Constants.CELL_ID), wcmda.getCellIdentity().getCid());
                                    pack.putInt(Constants.getVal(Constants.CELL_LAC), wcmda.getCellIdentity().getLac());
                                    pack.putInt(Constants.getVal(Constants.CELL_PSC), wcmda.getCellIdentity().getPsc() );
                                    //pack.putInt("mcc", wcmda.getCellIdentity().getMcc() );
                                    //pack.putInt("mnc", wcmda.getCellIdentity().getMnc() );
                                } catch (Exception e2) {

                                    }
                                }

                            return pack;

                        }//END of IF isRegistered

                    } //END-of_FOR
                }
            }
        }catch( Exception e){

        }
        return pack;
    }

    private Pack createPackFromCellArray(CellInfo cell){
        String equalInvalidVal = "=" + INVALID_VAL;
        if (Build.VERSION.SDK_INT < Build.VERSION_CODES.JELLY_BEAN_MR1)
            return null;

        Pack p = new Pack();
        p.putBool("registered", cell.isRegistered());
       try{
           CellInfoLte lte = (CellInfoLte) cell;
           Pack idp = new Pack();
           checkValue(lte.getCellIdentity().getCi(), idp, "ci");
           checkValue(lte.getCellIdentity().getMcc(), idp, "mcc");
           checkValue(lte.getCellIdentity().getMnc(), idp, "mnc");
           checkValue(lte.getCellIdentity().getPci(), idp, "pci");
           checkValue(lte.getCellIdentity().getTac(), idp, "tac");
           p.putPack("CellIdentityLte", idp);

           Pack ssp = new Pack();
           checkAsuValue(lte.getCellSignalStrength().getAsuLevel(), ssp);
           ssp.putInt("dbm", lte.getCellSignalStrength().getDbm());
           ssp.putInt("level", lte.getCellSignalStrength().getLevel());
           checkValue(lte.getCellSignalStrength().getTimingAdvance(), ssp, "timingAdvance");

           ssp.putString("original", lte.getCellSignalStrength().toString().replace(equalInvalidVal, ""));
           p.putPack("CellSignalStrengthLte", ssp);
           return p;
       }catch (Exception e){
           try {
               CellInfoGsm gsm = (CellInfoGsm) cell;
               Pack idp = new Pack();
               checkValue(gsm.getCellIdentity().getCid(), idp, "cid");
               checkValue(gsm.getCellIdentity().getMcc(), idp, "mcc");
               checkValue(gsm.getCellIdentity().getMnc(), idp, "mnc");
               //idp.putInt("psc", checkValue(gsm.getCellIdentity().getPsc()));
               checkValue(gsm.getCellIdentity().getLac(), idp, "lac");
               p.putPack("CellIdentityGsm", idp);

               Pack ssp = new Pack();
               checkAsuValue(gsm.getCellSignalStrength().getAsuLevel(), ssp);
               ssp.putInt("dbm", gsm.getCellSignalStrength().getDbm());
               ssp.putInt("level", gsm.getCellSignalStrength().getLevel());
               ssp.putString("original", gsm.getCellSignalStrength().toString().replace(equalInvalidVal, ""));
               p.putPack("CellSignalStrengthGsm", ssp);
               return p;
           }catch (Exception e2){
               try {
                   CellInfoCdma cmda = (CellInfoCdma) cell;
                   Pack idp = new Pack();
                   checkValue(cmda.getCellIdentity().getBasestationId(), idp, "basestationId");
                   checkValue(cmda.getCellIdentity().getLatitude(), idp, "latitude");
                   checkValue(cmda.getCellIdentity().getLongitude(), idp, "longitude");
                   checkValue(cmda.getCellIdentity().getNetworkId(), idp, "networkId");
                   checkValue(cmda.getCellIdentity().getSystemId(), idp, "systemId");
                   p.putPack("CellIdentityCmda", idp);

                   Pack ssp = new Pack();
                   checkAsuValue(cmda.getCellSignalStrength().getAsuLevel(), ssp);
                   ssp.putInt("dbm", cmda.getCellSignalStrength().getDbm());
                   ssp.putInt("level", cmda.getCellSignalStrength().getLevel());
                   ssp.putString("original", cmda.getCellSignalStrength().toString().replace(equalInvalidVal, ""));
                   p.putPack("CellSignalStrengthCmda", ssp);
                   return p;
               }catch (Exception e3){
                   try{
                       CellInfoWcdma wcmda = (CellInfoWcdma) cell;
                       Pack idp = new Pack();
                       checkValue(wcmda.getCellIdentity().getCid(), idp, "cid");
                       checkValue(wcmda.getCellIdentity().getMcc(), idp, "mcc");
                       checkValue(wcmda.getCellIdentity().getMnc(), idp, "mnc");
                       checkValue(wcmda.getCellIdentity().getPsc(), idp, "psc");
                       checkValue(wcmda.getCellIdentity().getLac(), idp, "lac");
                       p.putPack("CellIdentityWcmda", idp);

                       Pack ssp = new Pack();
                       checkAsuValue(wcmda.getCellSignalStrength().getAsuLevel(), ssp);
                       ssp.putInt("dbm", wcmda.getCellSignalStrength().getDbm());
                       ssp.putInt("level", wcmda.getCellSignalStrength().getLevel());
                       ssp.putString("original", wcmda.getCellSignalStrength().toString().replace(equalInvalidVal, ""));
                       p.putPack("CellSignalStrengthWcmda", ssp);
                       return p;

                   }catch (Exception e4){
                       return null;
                   }
               }
           }
       }

    }

    public void sendCellInfo(){
        final Pack upstreamPack = new Pack();
        upstreamPack.putPack(Constants.getVal(Constants.CELL_INFO_T), getCellInfo());
        upstreamPack.putString(Constants.getVal(Constants.F_MESSAGE_ID), UpstreamMessageFactory.generateMsgId());
        TaskManager.getInstance(mContext).asyncTask(new PushPoleAsyncTask() {
            @Override
            public void run(Context context) {
                new UpstreamSender(context).sendMessage(upstreamPack);
            }
        });
    }

    private void checkValue(int value, Pack pack, String key){
        if  (value != INVALID_VAL)
            pack.putInt(key, value);
    }

    private void checkAsuValue(int value, Pack pack){
        if  (value != ASU_INVALID_VAL)
            pack.putInt("asuLevel", value);
    }
}
