package com.pushpole.sdk.action.actions;

import android.content.ComponentName;
import android.content.Context;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.pm.PackageManager;
import android.content.pm.ResolveInfo;
import android.net.Uri;

import java.util.ArrayList;
import java.util.List;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.action.Action;
import com.pushpole.sdk.action.ActionFactory;
import com.pushpole.sdk.action.ActionType;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.util.ListPack;
import com.pushpole.sdk.util.Pack;

/**
 * class for creating a custom intent.
 */
public class IntentAction extends Action {
    protected String mData;
    protected List<String> mCategories;
    protected String mAction;
    protected String mPackage;
    protected List<String> mResolvers;

    /**
     * make appropriate {@link Intent}
     * start an activity if any Activity component that should be used to handle this intent exist
     *
     * @param context the application context.
     */
    @Override
    public void execute(Context context) {
        Intent intent = new Intent();
        if (mAction != null && !mAction.isEmpty()) {
            intent.setAction(mAction);
            Logger.info("Executing intent action. Extra: " + mAction);
        }

        if (mCategories != null && mCategories.size() > 0) {
            for (int i = 0; i < mCategories.size(); i++) {
                intent.addCategory(mCategories.get(i));
                Logger.info("Executing intent action. Category: " + mCategories.toString());
            }
        }

        if (mData != null) {
            intent.setData(Uri.parse(mData));
        }

        if (mPackage != null) {
            if(appInstalled(mPackage, context))
                intent.setPackage(mPackage); //set package name if app with this package name is installed on device
        }

        setIntentResolver(context, intent);

        intent.setFlags(Intent.FLAG_ACTIVITY_NEW_TASK);

        if (intent.resolveActivity(context.getPackageManager()) != null) {
            context.startActivity(intent);
        } else {
            Logger.warning("Intent action could not be resolved", new LogData(
                    "Action", mAction,
                    "Data", mData,
                    "Categories", String.valueOf(mCategories)
            ));
        }
    }

    private boolean appInstalled(String pakageName, Context context) {
        PackageManager pm = context.getPackageManager();
        try {
            pm.getPackageInfo(pakageName, PackageManager.GET_ACTIVITIES);
            return true;
        } catch (PackageManager.NameNotFoundException e) {
        }

        return false;
    }

    /**
     * set the {@link Intent} component
     *
     * @param context the app context
     * @param intent  the generated intent
     */
    private void setIntentResolver(Context context, Intent intent) {
        if (mResolvers == null || mResolvers.isEmpty()) {
            return;
        }

        for (String resolver : mResolvers) {
            final List<ResolveInfo> otherApps = context.getPackageManager().queryIntentActivities(intent, 0);
            for (ResolveInfo otherApp : otherApps) {
                if (otherApp.activityInfo.applicationInfo.packageName.equals(resolver)) {

                    ActivityInfo otherAppActivity = otherApp.activityInfo;
                    ComponentName componentName = new ComponentName(
                            otherAppActivity.applicationInfo.packageName,
                            otherAppActivity.name
                    );
                    intent.setComponent(componentName);
                    return;
                }
            }
        }
    }

    /**
     * return type
     *
     * @return {@code ActionType.INTENT}
     */
    @Override
    public ActionType getActionType() {
        return ActionType.INTENT;
    }

    /**
     * serialize to {@link Pack}
     *
     * @return
     */
    @Override
    public Pack toPack() {
        Pack pack = super.toPack();
        if (mAction != null) {
            pack.putString(Constants.getVal(Constants.F_INTENT_ACTION), mAction);
        }

        if (mData != null) {
            pack.putString(Constants.getVal(Constants.F_INTENT_DATA_URI), mData);
        }

        if (mCategories != null) {
            ListPack listPack = new ListPack(mCategories);
            pack.putListPack(Constants.getVal(Constants.F_INTENT_CATEGORY), listPack);
        }

        if (mResolvers != null) {
            ListPack listPack = new ListPack(mResolvers);
            pack.putListPack(Constants.getVal(Constants.F_INTENT_RESOLVERS), listPack);
        }

        if (mPackage != null && !mPackage.isEmpty()){
            pack.putString(Constants.getVal(Constants.MARKET_PACKAGE_NAME), mPackage);
        }
        return pack;
    }

    /**
     * Factory class to generate {@link IntentAction} instance
     */
    public static class Factory extends ActionFactory {

        /**
         * generate {@link IntentAction} instance
         *
         * @param data the data to create the {@code Action} with given as a {@link Pack}
         * @return {@link IntentAction} instance
         */
        @Override
        public Action buildAction(Pack data) {
            IntentAction action = new IntentAction();

            action.mAction = data.getString(Constants.getVal(Constants.F_INTENT_ACTION), null);
            action.mData = data.getString(Constants.getVal(Constants.F_INTENT_DATA_URI), null);

            ListPack categoryList = data.getListPack(Constants.getVal(Constants.F_INTENT_CATEGORY), null);
            if (categoryList != null) {
                action.mCategories = new ArrayList<>();
                for (int i = 0; i < categoryList.size(); i++) {
                    action.mCategories.add(categoryList.getString(i));
                }
            }

            // Read resolvers
            try {
                ListPack resolverList = data.getListPack(Constants.getVal(Constants.F_INTENT_RESOLVERS), null);
                if (resolverList != null) {
                    action.mResolvers = new ArrayList<>();
                    for (int i = 0; i < resolverList.size(); i++) {
                        action.mResolvers.add(resolverList.getString(i));
                    }
                }
            } catch (ClassCastException e) {
                String resolver = data.getString(Constants.getVal(Constants.F_INTENT_RESOLVERS), null);
                if (resolver != null) {
                    action.mResolvers = new ArrayList<>();
                    action.mResolvers.add(resolver);
                }
            }

            return action;
        }
    }
}
