package com.pushpole.sdk.message.upstream;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.message.downstream.DownstreamMessage;
import com.pushpole.sdk.util.Crypto;
import com.pushpole.sdk.util.Pack;

/**
 * A class that represent upstream message for delivery
 * delivery must contains the downstream message ID that has delivery required
 */
public class DeliveryMessage extends UpstreamMessage {
    private String mOriginalMessageId;

    /***
     * return original message ID
     * original message is the downstream message that has delivery required
     *
     * @return the message ID
     */
    public String getOriginalMessageId() {
        return mOriginalMessageId;
    }

    /***
     * return original message ID
     *
     * @param originalMessageId the ID
     */
    public void setOriginalMessageId(String originalMessageId) {
        mOriginalMessageId = originalMessageId;
    }

    /***
     * convert {@link DeliveryMessage} message to {@link Pack} object
     *
     * @return {@link Pack} object
     */
    @Override
    public Pack toPack() {
        Pack pack = super.toPack();
        pack.putString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID), mOriginalMessageId);
        pack.putString(Constants.getVal(Constants.F_STATUS), Constants.getVal(Constants.NOTIF_DELIVERED));
        pack.putString(Constants.getVal(Constants.TIMESTAMP), String.valueOf(System.currentTimeMillis()));

        return pack;
    }

    /***
     * return message type
     *
     * @return {@code Type.DELIVERY}
     */
    @Override
    public Type getMessageType() {
        return Type.DELIVERY;
    }

    /***
     * Factory class to generate {@link DeliveryMessage} instance
     */
    public static class Factory extends UpstreamMessageFactory {

        /***
         * generate {@link DeliveryMessage} instance by pack of data
         *
         * @param pack contains data for populating {@link DeliveryMessage} instance
         * @return {@link DeliveryMessage} instance
         */
        @Override
        public DeliveryMessage buildMessage(Pack pack) {
            DeliveryMessage message = new DeliveryMessage();
            populateMessage(message, pack);

            message.setOriginalMessageId(
                    pack.getString(Constants.getVal(Constants.F_ORIGINAL_MESSAGE_ID))
            );

            return message;
        }

        /***
         * generate {@link DeliveryMessage} instance by original message ID
         *
         * @param originalMessageId message ID
         * @return {@link DeliveryMessage} instance
         */
        public DeliveryMessage buildMessage(String originalMessageId) {
            DeliveryMessage message = new DeliveryMessage();
            populateMessage(message);
            message.setOriginalMessageId(originalMessageId);
            return message;
        }

        /***
         * generate {@link DeliveryMessage} instance by original message
         *
         * @param originalMessage
         * @return {@link DeliveryMessage} instance
         */
        public DeliveryMessage buildMessage(DownstreamMessage originalMessage) {
            return buildMessage(originalMessage.getMessageId());
        }

    }
}
