package com.pushpole.sdk.internal.log.handlers;

import android.content.Context;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;

import java.util.HashMap;
import java.util.Map;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.GooglePlayServicesHelper;
import com.pushpole.sdk.PushPoleInfo;
import com.pushpole.sdk.SenderInfo;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.device.DeviceIDHelper;
import com.pushpole.sdk.device.DeviceInfoHelper;
import com.pushpole.sdk.internal.log.Log;
import com.pushpole.sdk.internal.log.LogHandler;
import com.pushpole.sdk.internal.log.LogLevel;
import com.pushpole.sdk.internal.log.Sentry;
import com.pushpole.sdk.internal.log.StatsCollector;
import com.pushpole.sdk.util.PushPoleFailedException;


/***
 * This class is registered to Logger.register using log_handlers.json file
 * This class is loaded by reflection and its constructor is called at runtime.
 * Do not mistaken to considering it an unused class.
 */
public class SentryHandler implements LogHandler {
    private Context mContext;
    private boolean isRelatedToPushPole = true;

    public SentryHandler(Context context) {
        mContext = context.getApplicationContext();
    }

    @Override
    public void onLog(Log log) {
        try {
            Sentry.SentryEventBuilder eventBuilder = new Sentry.SentryEventBuilder();
            eventBuilder.setCulprit(log.getCulprit())
                    .setTimestamp(log.getTimestamp())
                    .setLevel(getLevel(log.getLogLevel()))
                    .setRelease(Constants.SENTRY_RELEASE_TRACKING_RELEASE_NAME);

            isRelatedToPushPole = log.isPushPoleError();

            if (log.getMessage() == null || log.getMessage().isEmpty()) {
                eventBuilder.setMessage("<Empty Log>");
            } else if (log.getParams().length > 0) {
                eventBuilder.setMessage(log.getMessage());
            } else {
                eventBuilder.setMessage(log.getMessage());
            }

            PackageInfo packageInfo = getPackageInfo();

            if (log.getException() != null) {
                eventBuilder.setException(log.getException());
            }

            eventBuilder.setTags(getTags(packageInfo));
            eventBuilder.setUser(getUser(packageInfo));

            Map<String, String> extras = log.getLogData().toMap();
            populateExtras(/*packageInfo,*/ extras);
            eventBuilder.setExtra(extras);

            Sentry.captureEvent(eventBuilder);
        }catch (Exception e){
            //to avoid error log when reporting an error!
        }
    }

    private Map<String, String> getTags(PackageInfo packageInfo) {
        Map<String, String> tags = new HashMap<>();
        DeviceInfoHelper infoHelper = new DeviceInfoHelper(mContext);
        tags.put("platform", infoHelper.getDevicePlatform().verbose());
        tags.put("brand", infoHelper.getDeviceBrand());
        tags.put("model", infoHelper.getDeviceModel());
        tags.put("os version", infoHelper.getOSVersion());
        tags.put("media", infoHelper.getDeviceMedia().toString().toLowerCase());
        tags.put("PushPole Version", PushPoleInfo.VERSION_NAME);
        tags.put("PushPole Build Number", String.valueOf(PushPoleInfo.VERSION_CODE));
        DeviceIDHelper idHelper = new DeviceIDHelper(mContext);
        tags.put("Device ID", idHelper.getDeviceId());
        tags.put("GooglePlay Service Status", KeyStore.getInstance(mContext)
                .getString(Constants.getVal(Constants.GOOGLE_PLAY_STATUS_CODE), "Code Unavailable"));
        tags.put("PushPole Error", String.valueOf(isRelatedToPushPole));

        // PushPole.initialize tries to get googlePlayService info, if it fails, it call Logger.fatal
        // and Looger calls getTags which itself tries to read googlePlayService Info
        int goolgePlayStatus = GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(mContext);
        if (goolgePlayStatus != ConnectionResult.SERVICE_MISSING && goolgePlayStatus != ConnectionResult.SERVICE_DISABLED &&
                goolgePlayStatus != ConnectionResult.SERVICE_INVALID) {
            tags.put("google-play version", GooglePlayServicesHelper.getGooglePlayServicesVersionName(mContext));
        }
        if (packageInfo != null) {
            tags.put("Package Name", packageInfo.packageName);
            tags.put("App Version Name", packageInfo.versionName);
            tags.put("App Version Code", String.valueOf(packageInfo.versionCode));
        }
        return tags;
    }

    private Map<String, String> getUser(PackageInfo packageInfo) {
        Map<String, String> user = new HashMap<>();
        DeviceIDHelper idHelper = new DeviceIDHelper(mContext);
        user.put("Device ID", idHelper.getDeviceId());
        user.put("Instance ID", SenderInfo.getInstance(mContext).getInstanceId());
        user.put("Registration Token", SenderInfo.getInstance(mContext).getToken());
        try {
            user.put("Sender Id", SenderInfo.getInstance(mContext).getSenderId());
        } catch (PushPoleFailedException e) {
            user.put("Sender Id", "getSenderId threw exception: " + e.getLocalizedMessage());
            //Logger.debug("getSenderId threw exception",e);
        }
        switch (SenderInfo.getInstance(mContext).getTokenState()) {
            case SenderInfo.NO_TOKEN:
                user.put("Registration State", "No Token");
                break;

            case SenderInfo.TOKEN_RECEIVED:
                user.put("Registration State", "Unregistered token");
                break;

            case SenderInfo.TOKEN_SYNCED:
                user.put("Registration State", "Registration complete");
                break;

            default:
                user.put("Registration State", "Token state is not valid");
                break;
        }

        if (packageInfo != null) {
            user.put("Package Name", packageInfo.packageName);
        }

        // PushPole.initialize tries to get googlePlayService info, if it fails, it call Logger.fatal
        // and Looger calls getTags which itself tries to read googlePlayService Info
        int goolgePlayStatus = GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(mContext);
        if (goolgePlayStatus != ConnectionResult.SERVICE_MISSING && goolgePlayStatus != ConnectionResult.SERVICE_DISABLED &&
                goolgePlayStatus != ConnectionResult.SERVICE_INVALID) {
            user.put("google-play version", GooglePlayServicesHelper.getGooglePlayServicesVersionName(mContext));
        }

        user.put("PushPole Version", PushPoleInfo.VERSION_NAME);
        user.put("PushPole Build Number", String.valueOf(PushPoleInfo.VERSION_CODE));
        return user;
    }


    private void populateExtras(Map<String, String> extras) {
        extras.put("KeyStore Size", String.valueOf(KeyStore.getInstance(mContext).size()));

        extras.put("Sent Mes", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_SENT_MESSAGES)));
        extras.put("Recv Mes", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_RECV_MESSAGES)));
        extras.put("Send Attmpt", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_SEND_ATTEMPTS)));
        extras.put("Sent Delivs", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_SENT_DELIVERIES)));
        extras.put("Recv Resp", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_RECV_RESPONSE)));
        extras.put("Recv Ack", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_ACKED_MESSAGES)));
        extras.put("Crpt Recv", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_BAD_RECV_MESSAGES)));
        extras.put("Sent Errs", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_SENT_ERRORS)));
        extras.put("Delt Mes", String.valueOf(StatsCollector.get(mContext, StatsCollector.STAT_DELETED_MESSAGES)));
    }

    private PackageInfo getPackageInfo() {
        try {
            PackageManager manager = mContext.getPackageManager();
            return manager.getPackageInfo(mContext.getPackageName(), 0);
        } catch (Exception e) {
            return null;
        }
    }

    private Sentry.SentryEventLevel getLevel(LogLevel level) {
        switch (level) {
            case DEBUG:
                return Sentry.SentryEventLevel.DEBUG;
            case INFO:
                return Sentry.SentryEventLevel.INFO;
            case WARN:
                return Sentry.SentryEventLevel.WARNING;
            case ERROR:
                return Sentry.SentryEventLevel.ERROR;
            case FATAL:
                return Sentry.SentryEventLevel.FATAL;
            default:
                return Sentry.SentryEventLevel.ERROR;
        }
    }
}
