package com.pushpole.sdk.receiver;

import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.ConnectivityManager;
import android.net.NetworkInfo;
import android.net.TrafficStats;
import android.net.wifi.WifiInfo;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.PushPole;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.network.SendManager;
import com.pushpole.sdk.util.NetworkHelper;
import com.pushpole.sdk.util.Pack;

/**
 * Created on 16-10-05, 3:07 PM.
 *
 * @author Akram Shokri
 */
public class ConnectivityReceiver extends BroadcastReceiver {
    @Override
    public void onReceive(Context context, Intent intent) {
        try {

            if (intent.getExtras() != null) {
                if (handleConnectionChange(context))
                    return; //network connection type is not changed

            }

            if (!PushPole.isInitialized()) {
                Logger.warning("ConnectivityReceiver: Network connectivity state changed. Initializing PushPole");
                PushPole.initializeInBackground(context);
            }
        } catch (Exception e) {
        }
    }

    public boolean handleConnectionChange(Context context) {
        final ConnectivityManager connectivityManager = (ConnectivityManager) context.getSystemService(Context.CONNECTIVITY_SERVICE);
        final NetworkInfo ni = connectivityManager.getActiveNetworkInfo();

        String wasConnectedTo = null;
        String connectedTo = null;

        //save prev-connection info
        long deviceRx = TrafficStats.getTotalRxBytes();
        long deviceTx = TrafficStats.getTotalTxBytes();
        long appRx = TrafficStats.getUidRxBytes(getPackageUid(context, context.getApplicationContext().getPackageName()));
        long appTx = TrafficStats.getUidTxBytes(getPackageUid(context, context.getApplicationContext().getPackageName()));
        long discTimestamp = System.currentTimeMillis();

        if (ni != null && ni.isConnectedOrConnecting()) {
            wasConnectedTo = KeyStore.getInstance(context).getString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TYPE, null);
            if (!Constants.getVal(Constants.WIFI).equals(wasConnectedTo))
                wasConnectedTo = KeyStore.getInstance(context).getString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE, null);

            connectedTo = ni.getTypeName().toLowerCase();
            if (ni.getType() == 0) //it is a mobile network, so we use its subtype (lte, 3g, etc.)
                connectedTo = ni.getSubtypeName().toLowerCase();
            if (connectedTo != null && connectedTo.equals(wasConnectedTo))
                return true;

            if (connectedTo != null && !connectedTo.equals(wasConnectedTo)) {//connected to a new network
                handleLastConnectionInfo(context, deviceRx, deviceTx, appRx, appTx, discTimestamp);

                //below call must be after 'handleLastConnectionInfo', because it overrides values needed in above code
                saveCurrentlyConnectedNetInfoToKeystore(context, ni, deviceRx, appRx,
                        appTx, deviceTx, discTimestamp); //save currently connected net info
            }
        } else /*if (intent.getBooleanExtra(ConnectivityManager.EXTRA_NO_CONNECTIVITY, Boolean.FALSE))*/ {
            handleLastConnectionInfo(context, deviceRx, deviceTx, appRx, appTx, discTimestamp);
            deleteConnectedNetInfoToKeystore(context);
        }
        return false;
    }

    private void handleLastConnectionInfo(Context context,
                                          long deviceRx, long deviceTx, long appRx, long appTx,
                                          long discTimestamp) {
        long wasConnectedTimestamp = KeyStore.getInstance(context).getLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TIMESTAMP, 0);

        long appRxStart = KeyStore.getInstance(context).getLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_APP_RX_BYTE, 0);
        long appTxStart = KeyStore.getInstance(context).getLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_APP_TX_BYTE, 0);
        long deviceTxStart = KeyStore.getInstance(context).getLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TOTAL_TX_BYTE, 0);
        long deviceRxStart = KeyStore.getInstance(context).getLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TOTAL_RX_BYTE, 0);


        String networkType = KeyStore.getInstance(context).getString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TYPE, null);
        String networkSubtype = KeyStore.getInstance(context).getString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE, null);
        String networkExtra = KeyStore.getInstance(context).getString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_EXTRA, null);
        boolean roaming = KeyStore.getInstance(context).getBoolean(PlainConstants.KEY_STORE_CONNECTED_NETWORK_ROAMING, false);

        if (networkType == null) {
            //if 'networkType == null' means we have no data
            // from when we connected to this network, so do nothing
            return;
        }

        Pack pack = new Pack();
        pack.putString(Constants.getVal(Constants.CONNECTION_INFO_NET_TYPE), networkType);
        if (networkExtra != null)
            pack.putString(Constants.getVal(Constants.CONNECTION_INFO_EXTRA), networkExtra);
        if (!Constants.getVal(Constants.WIFI).equals(networkType.toLowerCase())) {
            pack.putBool(Constants.getVal(Constants.CONNECTION_INFO_ROAMING), roaming);
            pack.putString(Constants.getVal(Constants.CONNECTION_INFO_NETWORK), networkSubtype);
        }

        pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_CONNECT_AT), wasConnectedTimestamp);
        pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DISCONNECT_AT), discTimestamp);
        //threshold is an int, specifying minimum net usage (in KB) which we consider for reporting
        int threshold = KeyStore.getInstance(context).getInt(PlainConstants.KEY_STORE_CONNECTED_NETWORK_THRESHOLD,
                PlainConstants.CONNECTION_INFO_DEFAULT_THRESHOLD);

        if (((deviceRx - deviceRxStart) + (deviceTx - deviceTxStart) / 1000) < threshold) {
            Logger.debug("Network connectivity usage was less than %dKB and dropped", threshold);
            return;
        }
        if ((appRx - appRxStart) > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_APP_DATA_RX), (appRx - appRxStart) / 1000);
        if ((appTx - appTxStart) > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_APP_DATA_TX), (appTx - appTxStart) / 1000);
//        if(mobileRx - mobileRxStart > 0)
//            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_MOBILE_RX), (mobileRx - mobileRxStart)/1000);
//        if(mobileTx - mobileTxStart > 0)
//            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_MOBILE_TX), (mobileTx - mobileTxStart)/1000);
        if ((deviceRx - deviceRxStart) > 0) //total usage minus mobile usage is equal to wifi usage
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_RX), (deviceRx - deviceRxStart) / 1000);
        if ((deviceTx - deviceTxStart) > 0)
            pack.putLong(Constants.getVal(Constants.CONNECTION_INFO_DATA_TX), (deviceTx - deviceTxStart) / 1000);

        SendManager.getInstance(context).send(Constants.getVal(Constants.CONNECTIVITY_INFO_T), pack);
    }

    private void saveCurrentlyConnectedNetInfoToKeystore(Context context, NetworkInfo ni, long deviceRx,
                                                         long appRx, long appTx, long deviceTx,
                                                         long timestamp) {
        KeyStore.getInstance(context).putLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_APP_RX_BYTE, appRx);
        KeyStore.getInstance(context).putLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_APP_TX_BYTE, appTx);
        KeyStore.getInstance(context).putLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TOTAL_TX_BYTE, deviceTx);
        KeyStore.getInstance(context).putLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TOTAL_RX_BYTE, deviceRx);
        KeyStore.getInstance(context).putString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TYPE, ni.getTypeName().toLowerCase());
        if (ni.getTypeName() != null && !Constants.getVal(Constants.WIFI).equals(ni.getTypeName().toLowerCase())) { //its mobile network
            KeyStore.getInstance(context).putString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE, ni.getSubtypeName().toLowerCase());
            KeyStore.getInstance(context).putString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_EXTRA, ni.getExtraInfo().toLowerCase());
            KeyStore.getInstance(context).putBoolean(PlainConstants.KEY_STORE_CONNECTED_NETWORK_ROAMING, ni.isRoaming());
        } else if (ni.getTypeName() != null && Constants.getVal(Constants.WIFI).equals(ni.getTypeName().toLowerCase())) { //its wifi network
            WifiInfo wifiInfo = NetworkHelper.WifiNetwork.connectedNetworkStat(context);
            if (wifiInfo != null) {
                KeyStore.getInstance(context).putString(PlainConstants.KEY_STORE_CONNECTED_NETWORK_EXTRA, wifiInfo.getBSSID());//macAddress of connected network
            }
        }
        KeyStore.getInstance(context).putLong(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TIMESTAMP  , timestamp);
    }

    private void deleteConnectedNetInfoToKeystore(Context context) {
        KeyStore.getInstance(context).delete(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TYPE);
        KeyStore.getInstance(context).delete(PlainConstants.KEY_STORE_CONNECTED_NETWORK_SUBTYPE);
        KeyStore.getInstance(context).delete(PlainConstants.KEY_STORE_CONNECTED_NETWORK_ROAMING);
        KeyStore.getInstance(context).delete(PlainConstants.KEY_STORE_CONNECTED_NETWORK_EXTRA);
        KeyStore.getInstance(context).delete(PlainConstants.KEY_STORE_CONNECTED_NETWORK_TIMESTAMP);
    }

    public static int getPackageUid(Context context, String packageName) {
        PackageManager packageManager = context.getPackageManager();
        int uid = -1;
        try {
            PackageInfo packageInfo = packageManager.getPackageInfo(packageName, PackageManager.GET_META_DATA);
            uid = packageInfo.applicationInfo.uid;
        } catch (PackageManager.NameNotFoundException e) {
        }
        return uid;
    }
}
