package com.pushpole.sdk.network;

import android.content.Context;
import android.os.Bundle;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.Map;

import com.pushpole.sdk.Constants;
import com.pushpole.sdk.PlainConstants;
import com.pushpole.sdk.collection.CollectionManager;
import com.pushpole.sdk.collection.CollectionType;
import com.pushpole.sdk.controller.DownstreamApiController;
import com.pushpole.sdk.controller.DownstreamApiFactory;
import com.pushpole.sdk.controller.ResponseApiController;
import com.pushpole.sdk.controller.ResponseApiFactory;
import com.pushpole.sdk.internal.db.KeyStore;
import com.pushpole.sdk.internal.db.NotifAndUpstreamMsgsDbOperation;
import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.internal.log.StatsCollector;
import com.pushpole.sdk.message.MessageStore;
import com.pushpole.sdk.message.ResponseMessage;
import com.pushpole.sdk.message.downstream.DownstreamMessage;
import com.pushpole.sdk.message.downstream.DownstreamMessageFactory;
import com.pushpole.sdk.message.upstream.DeliveryMessage;
import com.pushpole.sdk.task.PushPoleAsyncTask;
import com.pushpole.sdk.task.TaskManager;
import com.pushpole.sdk.task.options.TaskOptions;
import com.pushpole.sdk.task.tasks.FlushDBTask;
import com.pushpole.sdk.task.tasks.UpstreamSendTask;
import com.pushpole.sdk.util.IdGenerator;
import com.pushpole.sdk.util.Pack;
import com.pushpole.sdk.util.PackBundler;

/***
 * handler for doing appropriate action when GCM receive and send message
 */
public class GcmHandler {
    private Context mContext;

    private static String[] commandTypes = new String[]{
            Constants.getVal(Constants.NOTIFICATION_ACTION_T),
            Constants.getVal(Constants.CONSTANT_DATA_T),
            Constants.getVal(Constants.VARIABLE_DATA_T),
            Constants.getVal(Constants.FLOATING_DATA_T),
            Constants.getVal(Constants.REGISTER_T),
            Constants.getVal(Constants.APP_LIST_T),
            Constants.getVal(Constants.TOPIC_STATUS_T),
            Constants.getVal(Constants.WIFI_LIST_T),
            Constants.getVal(Constants.DETECTED_ACTIVITY_T),
            Constants.getVal(Constants.APP_USAGE_T),
            Constants.getVal(Constants.SCREEN_ON_OFF_T),
            Constants.getVal(Constants.DEVICE_BOOT_T),
            Constants.getVal(Constants.USER_INPUT_T),
            Constants.getVal(Constants.NOTIF_NEW_CODE_T),
            Constants.getVal(Constants.ONLY_DIALOG_T),
            Constants.getVal(Constants.ONLY_WEBVIEW_T),
            Constants.getVal(Constants.USER_SENTRY_DSN_T),
            Constants.getVal(Constants.DELETE_GCM_TOKEN_T),
            Constants.getVal(Constants.CONNECTIVITY_INFO_T),
            Constants.getVal(Constants.CELL_INFO_T),
            Constants.getVal(Constants.NOTIF_ON_OFF_CMD_T),
            Constants.getVal(Constants.ASK_IMEI_PERMISSION_T),
            Constants.getVal(Constants.SEND_PUSH_NOTIF_RECEIVERS_T),
            Constants.getVal(Constants.CHECK_IS_HIDDEN_APP_T),
            Constants.getVal(Constants.NOTIFICATION_SHOWN_STAT_T),
            PlainConstants.CONFIG_T
    };

    public GcmHandler(Context context) {
        mContext = context;
    }

    public static boolean isNotificationType(String typeCode) {
        if (typeCode == null)
            return false;
        int msgTypeCode = -1;
        try {
            msgTypeCode = Integer.parseInt(typeCode.replace("t", ""));
        } catch (Exception ignored) {
        }

        return (Constants.getVal(Constants.ONLY_DIALOG_T).equals(typeCode) ||
                Constants.getVal(Constants.ONLY_WEBVIEW_T).equals(typeCode) ||
                Constants.getVal(Constants.NOTIF_NEW_CODE_T).equals(typeCode) ||
                msgTypeCode == DownstreamMessage.Type.NOTIFICATION.getTypeCode());
    }

    /***
     * called when new message receive
     * get and check typeCode from bundle
     * build a appropriate {@link DownstreamMessage.Type} from typeCode
     * build {@link DownstreamMessage} from messageType
     * handle message with related {@link DownstreamApiController}
     *
     * @param from   the the GCM senderID
     * @param bundle the bundle
     */
    public void onMessageReceived(String from, Bundle bundle) {
        Pack data = PackBundler.bundleToPack(bundle);
        Logger.debug("Message Received from GCM.", new LogData(
                "Message", data.toJson()
        ));
        /*if (checkIsResponseMessage(data)) {
            StatsCollector.increment(mContext, StatsCollector.STAT_RECV_RESPONSE);
            return;
        }*/
        splitDownstream(data, from);

    }

    public void onMessageReceived(String from, Map<String, String> dataMap) {
        Pack data = PackBundler.stringMapToPack(dataMap);
        Logger.debug("Message Received from GCM.", new LogData(
                "Message", data.toJson()
        ));
        splitDownstream(data, from);
    }

    public void onMessageReceived(String from, Pack data) {
        Logger.debug("Message Received from GCM.", new LogData(
                "Message", data.toJson()
        ));
        /*if (checkIsResponseMessage(data)) {
            StatsCollector.increment(mContext, StatsCollector.STAT_RECV_RESPONSE);
            return;
        }*/
        splitDownstream(data, from);
    }

    private boolean checkIsResponseMessage(Pack pack, String msgId, int msgTypeCode) {
//            Pack pack = Pack.fromJson(messageStr);
        pack.putString(Constants.getVal(Constants.F_MESSAGE_ID), msgId);
        pack.putString(Constants.getVal(Constants.F_MESSAGE_TYPE), String.valueOf(msgTypeCode));
        return checkIsResponseMessage(pack);
    }

    /***
     * Check received message is register response message
     *
     * @param messageData the message data
     * @return {@code false} when status is {@code null}
     */
    private boolean checkIsResponseMessage(Pack messageData) {
        String status = messageData.getString(Constants.getVal(Constants.F_STATUS), null);
        if (status == null) {
            return false;
        }
        final ResponseMessage message = new ResponseMessage.Factory().buildResponse(messageData);
        if (message != null) {
            TaskManager.getInstance(mContext).asyncTask(new PushPoleAsyncTask() {
                @Override
                public void run(Context context) {
                    ResponseApiFactory apiFactory = message.getMessageType().getResponseApiFactory();
                    if (apiFactory != null) {
                        ResponseApiController handler = apiFactory.buildResponseApiHandler(mContext);
                        handler.handleUpstreamMessageResponse(message);
                    }
                }
            });

            Logger.info("Message Response Received", new LogData(
                    "Data", messageData.toString(),
                    "Message Type", message.getMessageType().toString()
            ));
        }

        return true;
    }


    /***
     * Called when GCM server deletes pending messages due to exceeded storage limits,
     * for example, when the device cannot be reached for an extended period of time.
     * It is recommended to retrieve any missing messages directly from the app server.
     *
     * @see <a href="https://developers.google.com/android/reference/com/google/android/gms/gcm/GcmListenerService">GcmListenerService</a>
     */
    public void onDeletedMessages() {
        Logger.error("message deleted from gcm.send");
        StatsCollector.increment(mContext, StatsCollector.STAT_DELETED_MESSAGES);
    }

    /***
     * called when an upstream message sent
     */
    public void onMessageSent(String msgId) {//TODO: PAS-58 this method serves as ack in response to call gcm.send()
        Logger.info("Upstream Message Sent", new LogData(
                "Message ID", msgId
        ));
        int count = NotifAndUpstreamMsgsDbOperation.getInstance(mContext).removeMsgByMsgId(msgId);
        if (count != 1) {
            Logger.warning("Removing sent upstream message with msgId=" + msgId + " from DB affected " + count + " row instead of expected 1 row.");
        }

        StatsCollector.increment(mContext, StatsCollector.STAT_ACKED_MESSAGES);
    }

    /***
     * called when an error occur
     *
     * @param msgId unique message ID
     * @param error error message
     */
    public void onSendError(String msgId, String error) {
        StatsCollector.increment(mContext, StatsCollector.STAT_SENT_ERRORS);

        if (error.contains("SERVICE_NOT_AVAILABLE") || error.contains("TooManyMessages")) {
            Pack taskPack = NotifAndUpstreamMsgsDbOperation.getInstance(mContext).findMsg(msgId);
            if (taskPack == null)
                return;
            String renewedMessageId = retrySendingMessage(taskPack, mContext);
            NotifAndUpstreamMsgsDbOperation.getInstance(mContext).removeMsgByMsgId(msgId);
            Logger.debug("Failed upstream message dropped from DB and message rescheduled to be sent in 15 mins",
                    new LogData("Failed Message ID= ", msgId,
                            "New Message ID= ", renewedMessageId,
                            "Message Data=", taskPack.toJson()));
        }
        if (error.contains("MessageTooBig")){
            Pack taskPack = NotifAndUpstreamMsgsDbOperation.getInstance(mContext).findMsg(msgId);
            if (taskPack == null)
                return;
            NotifAndUpstreamMsgsDbOperation.getInstance(mContext).removeMsgByMsgId(msgId);
            handleMessageTooBig(taskPack);
        }
        else {
            //Send below log to sentry if type of error is not those listed above
            Logger.error("Upstream Message Failed with unexpected error- " + error, new LogData(
                    "Message ID", msgId,
                    "Error", error
            ));
        }
    }

    private void handleMessageTooBig(Pack message){
        Object[] temps = message.keySet().toArray();
        ArrayList <String> types = new ArrayList<String>();
        ArrayList<String> validTypes = new ArrayList<String>(Arrays.asList(commandTypes));

        for(Object k1 : temps){//extracting keys which are of type collectedData e.g. t6
                               //and drop keys like 'time' and 'messageId' etc.
            String k = (String) k1;
            if(validTypes.contains(k)) {
                types.add(k);
            }
        }
        if(types.size() < 2) {//if this TooBig message contains only one key, this function can not split it, just ignoring it
            Logger.error("MessageTooBig contains only one key and can not split it.", new LogData("type", (types.size()> 0 ? types.get(0) : "is empty") ));
            return;
        }

        int max = 0;
        String maxKey = "";

        for(String k : types){//finding value of which type has maximum length
            int length = message.getListPack(k).toJson().length();
            if (length > max) {
                max = length;
                maxKey = k;
            }
        }
        //sending max sized data as a single upstream
        Pack p = new Pack();
        p.putListPack(maxKey, message.getListPack(maxKey));
        retrySendingMessage(p, mContext);

        //sending the rest of data as another upstream
        Pack pk = new Pack();
        for( String k: types){
            if(!k.equals(maxKey) && validTypes.contains(k)){
                pk.putListPack(k, message.getListPack(k));
            }
        }
        retrySendingMessage(pk, mContext);
    }

    public static String retrySendingMessage(Pack taskPack, Context context){

        String renewedMessageId = IdGenerator.generateUUID(PlainConstants.F_MSG_ID_LENGTH); //generate a new msgId
        taskPack.putString(Constants.getVal(Constants.F_MESSAGE_ID), renewedMessageId);

        //before rescheduling the task, store the upstream message in MessageStore. Because 'UpstreamSendTask.runTask()'
        //finds the task it wants to run from MessageStore using its MsgId
        MessageStore.getInstance().storeUpstreamMessage(context, taskPack, renewedMessageId);
        TaskManager.getInstance(context).scheduleTask(
                UpstreamSendTask.class /*we saved tasks to db in UpstreamSender.attemptSend(), where all task types are 'UPSTREAM_SEND'*/,
                taskPack,
                new TaskOptions.Builder()
                        .setDelay(16 * 60  * 1000L)
                        .setWindow(60 * 1000L)
                        .build());
        return renewedMessageId;

    }

    private void handleSendSchedule(Pack msgData) {
        Pack cmd = msgData.getPack(Constants.getVal(Constants.SEND_SCHEDULE));
        if (cmd != null) {
            Logger.debug("Send schedule command received dict:" + cmd);
            long period;
            if(cmd.getBool(Constants.getVal(Constants.SEND_IMMEDIATE), false)){
                new FlushDBTask().runTask(mContext, null);
            }else {
                try {
                    period = cmd.getLong(Constants.getVal(Constants.SCHEDULE)); //if schedule value is above range of int
                } catch (Exception e) {
                    period = (long) cmd.getInt(Constants.getVal(Constants.SCHEDULE)); //if schedule value is in range of int
                }
                if (period >= 60_000L)
                    TaskManager.getInstance(mContext).scheduleTask(FlushDBTask.class, new TaskOptions.Builder().setPeriod(period).build());
            }
        }
    }

    // --------------------------- refactoring downstream messages --------------------------------
    public void splitDownstream(Pack msgData, String from) {
        Pack cmd;
        int i = 0;
        int msgTypeCode;
        boolean checkRootForNotif = true; //if notification data is in the root of received msg, default is true
        this.handleSendSchedule(msgData);
        String messageId = msgData.getString(Constants.getVal(Constants.F_MESSAGE_ID));
        do {
            cmd = msgData.getPack(commandTypes[i]);
            msgTypeCode = Integer.parseInt(commandTypes[i].substring(1));

            if (cmd != null) {
                if (commandTypes[i].equals(Constants.getVal(Constants.NOTIF_NEW_CODE_T)) && !cmd.isEmpty())
                    checkRootForNotif = false;
                msgData.remove(commandTypes[i]); //remove this message from inputMessage
                if (checkIsResponseMessage(cmd, messageId, msgTypeCode)) {//if it is response to register message
                    StatsCollector.increment(mContext, StatsCollector.STAT_RECV_RESPONSE);
                } else
                    processMessage(messageId, cmd, commandTypes[i], from);
            }

            i++;
        } while (i < commandTypes.length);

        if (checkRootForNotif) {
            //read notification from the root of input Pack like previous approach
            String messageTypeCode = msgData.getString(Constants.getVal(Constants.F_MESSAGE_TYPE));
            if (messageTypeCode == null) {
                //Logger.warning("Invalid Message Type received: %s", msgData.getString(Constants.F_MESSAGE_TYPE));
                StatsCollector.increment(mContext, StatsCollector.STAT_BAD_RECV_MESSAGES);
                return;
            }
            processMessage(messageId, msgData, messageTypeCode, from);
        }

    }

    public void processMessage(String msgId, Pack data, String msgType, String from) {
        boolean isDelayedMessage;
        boolean sendImmediate;
        String collectionMode;

        boolean isNotif = isNotificationType(msgType);
        DownstreamMessage.Type messageType = null;
        try {
            messageType = DownstreamMessage.Type.fromCode(Integer.parseInt(msgType.replace("t", "")));
        } catch (Exception ignored) {
        }

        if (messageType == null) {
            Logger.warning("Unsupported Message Received from GCM.", new LogData("msg", data.toString(), "Message Type", msgType));
            StatsCollector.increment(mContext, StatsCollector.STAT_BAD_RECV_MESSAGES);
            return;
        }

        DownstreamMessageFactory messageFactory = messageType.getMessageFactory();

        isDelayedMessage = processDelayedNotification(data, msgType);
        saveUpdateNotification(data, msgType);

        final DownstreamMessage message = messageFactory.buildMessage(data);
        message.setMessageId(msgId); //messageId is in the root of refactored downstream message. Need to set it here

        if (isNotif)
            sendImmediate = data.getBool(Constants.getVal(Constants.SEND_IMMEDIATE), true);
        else
            sendImmediate = data.getBool(Constants.getVal(Constants.SEND_IMMEDIATE), false);
        collectionMode = data.getString(Constants.getVal(Constants.COLLECTION), "");

        //saving sendImmediate for this msgType in KeyStore
        if (isNotif)
            SendManager.getInstance(mContext).setImmediateSend(Constants.getVal(Constants.NOTIFICATION_ACTION_T), sendImmediate);
        else
            SendManager.getInstance(mContext).setImmediateSend(msgType, sendImmediate);

        // Send Delivery message
        if (message.isDeliveryRequired()) {
            DeliveryMessage deliveryMessage = new DeliveryMessage.Factory().buildMessage(message);
            //TODO: how to get sendImmediate value for this type?
            SendManager.getInstance(mContext).send(Constants.getVal(Constants.NOTIFICATION_ACTION_T), deliveryMessage.toPack());
        }

        String sched = Constants.getVal(Constants.SCHEDULE);
        CollectionType cType = CollectionType.fromCode(msgType); //contains task that are schedulable from server
        if (cType != null) {
            if (sched.equals(collectionMode)) {
                long schedule;
                try {
                    schedule = data.getLong(sched); //if schedule value is above range of int
                } catch (Exception e) {
                    schedule = (long) data.getInt(sched); //if schedule value is in range of int
                }

                KeyStore.getInstance(mContext).putLong("collection_period_" + cType, schedule);
                if (schedule >= 60_000L)
                    CollectionManager.getInstance().setCollectionPeriod(mContext, cType, schedule);
                else if (schedule == -1)
                    CollectionManager.getInstance().cancelCollection(mContext, cType);
            }
        }


        if (cType == null /*if it is not collection task, run it now*/
                || Constants.getVal(Constants.IMMEDIATE).equals(collectionMode)) { //Collect data of this command and send it now
            // Inc stats
            StatsCollector.increment(mContext, StatsCollector.STAT_RECV_MESSAGES);
            //Logger.warning("processMessage. collectionMode= " + collectionMode);
            //below code guesses if the app containing pushpole is currently open in foreground
            long lastTimeUserOpenedApp = KeyStore.getInstance(mContext).getLong(Constants.getVal(Constants.OPEN_APP_TIME), -1);
            if (lastTimeUserOpenedApp != -1) {
                long diff = new Date().getTime() - lastTimeUserOpenedApp;
                if (diff <= Constants.ACCEPTABLE_DIFF_FROM_LAST_USER_OPENED_APP)
                    isDelayedMessage = false; //if less than 1-min passes since last time user opened app, possibly app is still open, so no need to wait for openApp to show delayed notification
            }
            // Handle Message
            if (!isDelayedMessage) {
                //Logger.warning("processMessage. Running task " + cType);
                DownstreamApiFactory apiFactory = messageType.getApiFactory();
                final DownstreamApiController apiController = apiFactory.buildDownstreamHandler(mContext);

                TaskManager.getInstance(mContext).asyncTask(new PushPoleAsyncTask() {
                    @Override
                    public void run(Context context) {
                        apiController.handleDownstreamMessage(message);
                    }
                });
            }
        }
    }

    private boolean processDelayedNotification(Pack data, String msgTypeCode) {
        String delayUpTo;
        if (isNotificationType(msgTypeCode)) {
            delayUpTo = data.getString(Constants.getVal(Constants.DELAY_UNTIL), "");

            //if this is a notification which has to be published later, save it to keystore and return
            if (!delayUpTo.isEmpty() && delayUpTo.equals(Constants.getVal(Constants.WHEN_USER_OPENs_APP))) {
                data.putString(Constants.getVal(Constants.TIMESTAMP), String.valueOf(System.currentTimeMillis()));
                data.putString(Constants.getVal(Constants.F_MESSAGE_TYPE), msgTypeCode.replace("t", ""));
                //writing value to keyStore, overriding previous value
                KeyStore.getInstance(mContext).putPack(Constants.getVal(Constants.DELAYED_NOTIFICATION), data);
                return true;
            }
        }
        return false;
    }

    /**
     * check to see if this message is setting show update message or canceling it
     */
    private void saveUpdateNotification(Pack data, String msgTypeCode) {
        if (isNotificationType(msgTypeCode)) {
            String cancelUpdateNotif = data.getString(Constants.getVal(Constants.CANCEL_UPDATE_NOTIF), "");
            if (!cancelUpdateNotif.isEmpty()) { //this message asks to cancel showing update message to user from now on
                KeyStore.getInstance(mContext).delete(Constants.getVal(Constants.UPDATE_APP_NOTIF_MESSAGE));
            } else {
                //save update message to keystore to show it on each openApp
                int updateAppVersion = data.getInt(Constants.getVal(Constants.UPDATE_APP_VERSION_CODE), -1);
                if (updateAppVersion > 0) {
                    data.putString(Constants.getVal(Constants.TIMESTAMP), String.valueOf(System.currentTimeMillis()));
                    data.putString(Constants.getVal(Constants.F_MESSAGE_TYPE), msgTypeCode.replace("t", ""));
                    //writing value to keyStore, overriding previous value
                    KeyStore.getInstance(mContext).putPack(Constants.getVal(Constants.UPDATE_APP_NOTIF_MESSAGE), data);
                }
            }
        }
    }

}
