package com.pushpole.sdk.location;

import android.content.Context;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.provider.Settings;

import java.util.Date;

import com.pushpole.sdk.internal.log.LogData;
import com.pushpole.sdk.internal.log.Logger;
import com.pushpole.sdk.util.InsufficientPermissionsException;
import com.pushpole.sdk.util.PermissionChecker;

/**
 * Provides methods for obtaining and requesting the device location
 *
 * @author Hadi Zolfaghari
 * @since 1.0.0
 */
public class GeoManager implements LocationListener {
    private final static String PROVIDER = LocationManager.NETWORK_PROVIDER;
    private final static int BETTER_LOC_TIME_THRESH = 60 * 1000;
    private final static float BETTER_LOC_ACC_THRESH = 3;
    private final static int GOOD_LOC_TIME_THRESH = 60 * 60 * 1000;

    private boolean temporaryUnavailable = false;


    private volatile static GeoManager mInstance;

    private Location mLocation;
    private Context mContext;

    private GeoManager() {

    }

    public static GeoManager getInstance(Context context) {
        if (mInstance == null) {
            synchronized (GeoManager.class) {
                if (mInstance == null) {
                    mInstance = new GeoManager();
                }
            }
        }
        if(context != null){
            mInstance.mContext = context;
        }
        return mInstance;
    }

    /**
     * Gets the current best location available.
     *
     * @return the best location available or null if no location is available
     * @throws InsufficientPermissionsException if neither ACCESS_COURSE_LOCATION nor
     *                                          ACCESS_FINE_LOCATION permissions are given
     */
    public Location getLocation(/*Context context*/) throws InsufficientPermissionsException {
        checkForBetterLocation();
        return mLocation;
    }

    /**
     * Request a location update.
     * <p>
     * Calling this method will request a single location update. If a new location is retrieved it
     * will be compared to the current location and will be kept as the current location if it is
     * considered a better location. The new location could then be obtained using the
     * {@code getLocation} method.
     *
     * @throws InsufficientPermissionsException if neither ACCESS_COURSE_LOCATION nor
     *                                          ACCESS_FINE_LOCATION permissions are given
     */
    public void requestLocation(/*Context context*/) throws InsufficientPermissionsException {
        getLocationManager().requestSingleUpdate(PROVIDER, this, mContext.getMainLooper());
    }

    /**
     * Checks whether there is a good location available.
     *
     * @return true if a good location is available, false otherwise
     * @throws InsufficientPermissionsException if neither ACCESS_COURSE_LOCATION nor
     *                                          ACCESS_FINE_LOCATION permissions are given
     */
    public boolean hasGoodLocation(/*Context context*/) throws InsufficientPermissionsException {
        checkForBetterLocation(/*context*/);

        if (mLocation == null) {
            return false;
        }

        Date now = new Date();
        return now.getTime() - mLocation.getTime() < GOOD_LOC_TIME_THRESH;
    }

    /**
     * Checks if the {@code getLastKnownLocation} method of the {@link LocationManager} class
     * provides a better location than the current known location and updates the current location
     * if so.
     *
     * @throws InsufficientPermissionsException if neither ACCESS_COURSE_LOCATION nor
     *                                          ACCESS_FINE_LOCATION permissions are given
     */
    private void checkForBetterLocation(/*Context context*/) throws InsufficientPermissionsException {
        Location location = getLocationManager(/*mContext*/).getLastKnownLocation(PROVIDER);

        if (isLocationBetter(location, mLocation)) {
            setLocation(location);
        }
    }

    /**
     * Compares to locations with location accuracy and location time as the comparison factors.
     * <p>
     * <ul>
     * <li> If either locations are null the other is considered better. If both locations are null then
     * the first location is considered better.
     * <li> If any location is newer than the other by a certain threshold, that location is
     * considered better
     * <li> If both locations have accuracy and the accuracy of one is greater by a certain threshold
     * than the other, then that location is considered better
     * <li> If neither of the above applies, the location with the newest time is considered better
     *
     * @param loc1 the first location
     * @param loc2 the second location
     * @return true if the first location is a better location than the second, false otherwise
     */
    private boolean isLocationBetter(Location loc1, Location loc2) {
        if (loc2 == null) {
            return true;
        }

        if (loc1 == null) {
            return false;
        }

        if (Math.abs(loc2.getTime() - loc1.getTime()) > BETTER_LOC_TIME_THRESH) {
            return loc1.getTime() > loc2.getTime();
        }

        if (!loc1.hasAccuracy() || !loc2.hasAccuracy()) {
            return loc1.getTime() > loc2.getTime();
        }

        if (loc1.getAccuracy() < loc2.getAccuracy() &&
                loc2.getAccuracy() / loc1.getAccuracy() > BETTER_LOC_ACC_THRESH) {
            return true;
        } else if (loc2.getAccuracy() < loc1.getAccuracy() &&
                loc1.getAccuracy() / loc2.getAccuracy() > BETTER_LOC_ACC_THRESH) {
            return false;
        }

        return loc1.getTime() > loc2.getTime();
    }

    /**
     * @return the android {@link LocationManager}
     * @throws InsufficientPermissionsException if neither ACCESS_COURSE_LOCATION nor
     *                                          ACCESS_FINE_LOCATION permissions are given
     */
    private LocationManager getLocationManager(/*Context context*/) throws InsufficientPermissionsException {
        boolean hasPermission = PermissionChecker.hasPermission(mContext, PermissionChecker.ACCESS_COURSE_LOCATION)
                || PermissionChecker.hasPermission(mContext, PermissionChecker.ACCESS_FINE_LOCATION);

        if (!hasPermission) {
            throw new InsufficientPermissionsException(PermissionChecker.ACCESS_COURSE_LOCATION);
        }

        return (LocationManager) mContext.getSystemService(Context.LOCATION_SERVICE);
    }

    private void setLocation(Location location) {
        mLocation = location;
    }

    /**
     * Checks whether the location permissions exist and the location provider is enabled
     * <p>
     * In order to use {@code GeoManager} either {@code ACCESS_COURSE_LOCATION} or
     * {@code ACCESS_FINE_LOCATION} permission must be defined in the application manifest file.
     *
     * @return true if location is enabled, false otherwise
     */
    public boolean isLocationEnabled(/*Context context*/) {
        try {
            boolean hasPermission = PermissionChecker.hasPermission(mContext, PermissionChecker.ACCESS_COURSE_LOCATION)
                    || PermissionChecker.hasPermission(mContext, PermissionChecker.ACCESS_FINE_LOCATION);
            if (!hasPermission) {
                return false;
            }

            LocationManager manager = (LocationManager) mContext.getSystemService(Context.LOCATION_SERVICE);
            return manager.isProviderEnabled(PROVIDER);
        } catch (SecurityException e) {
            Logger.error("Network Permissions not given to application");
            return false;
        }
    }

    /**
     * Check if user has turned on location access on her/his device
     * @return true if user has enabled its location service, false otherwise
     */
    public boolean isLocationTurnedOn(){
        String locationProviderAllowed = Settings.Secure.getString(mContext.getContentResolver(), Settings.Secure.LOCATION_PROVIDERS_ALLOWED);
        Logger.debug("locationProviderAllowed: " + locationProviderAllowed);
        if(locationProviderAllowed.toLowerCase().contains("network")) {
            return true;
        }
        else
            return false;

    }

    @Override
    public void onLocationChanged(Location location) {
        Logger.debug("Location received", new LogData(
                "Location", location.toString()
        ));
        if (isLocationBetter(location, mLocation)) {
            setLocation(location);
        }
    }

    @Override
    public void onStatusChanged(String s, int i, Bundle bundle) {

    }

    @Override
    public void onProviderEnabled(String s) {
    }

    @Override
    public void onProviderDisabled(String s) {
    }

    public boolean isTemporaryUnavailable() {
        return temporaryUnavailable;
    }
}
