package com.pushchamp.pushchampsdk;

import android.app.NotificationManager;
import android.app.PendingIntent;
import android.content.Context;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import android.media.RingtoneManager;
import android.net.Uri;
import android.os.Bundle;
import android.support.v4.app.NotificationCompat;
import android.util.Log;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.Iterator;
import java.util.Random;

public class PushchampNotification {
    private Context context;
    private boolean isSilent;
    private boolean isShowable;
    private String message;
    private String header;
    private int smallIcon;
    private String largeIcon;
    private Intent pendingIntent;
    private int notification_id;
    private NotificationCompat.Action[] actions;
    private NotificationCompat.Style notification_style;

    private static final String TAG = PushchampNotification.class.getSimpleName();

    public PushchampNotification(Context context, Bundle data){
        this.context = context;
        this.isSilent = data.getBoolean("sl", false);
        this.message = data.getString("mg");
        this.header = data.getString("hd");
        String s_icon = data.getString("si");
        this.smallIcon = this.context.getResources().getIdentifier(s_icon, "drawable", this.context.getPackageName());
        if (this.smallIcon==0){
            Log.e(TAG, "Default icon not found in resources. Unable to process notification.");
            try{
                this.smallIcon = android.R.drawable.ic_dialog_info;
            }catch(Exception e){
                Log.e(TAG, "Default icon not found in resources. Unable to process notification.");
            }
        }

        this.largeIcon = data.getString("li");
        if(this.smallIcon==0 || this.isSilent){
            this.isShowable = false;
        }else{
            this.isShowable = true;
        }
        this.pendingIntent = makePendingIntent(data, "tc", "td");
        this.notification_id = data.getInt("nid", (new Random()).nextInt());
        this.notification_style = makeNotifStyle(data);
        this.actions = addActions(data);
    }

    public void show(){
        Log.i(TAG, "Notification Showable : "+this.isShowable);
        if (this.isShowable){
            Uri defaultSoundUri= RingtoneManager.getDefaultUri(RingtoneManager.TYPE_NOTIFICATION);
            NotificationCompat.Builder notificationBuilder = new NotificationCompat.Builder(this.context)
                    .setSmallIcon(this.smallIcon)
                    .setContentTitle(this.header)
                    .setContentText(this.message)
                    .setAutoCancel(true)
                    .setSound(defaultSoundUri);
            if(this.largeIcon!=null){
                Bitmap lrgIcn = getBitmapFromURL(this.largeIcon);
                notificationBuilder.setLargeIcon(lrgIcn);
            }
            if (this.pendingIntent!=null){
                this.pendingIntent.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
                PendingIntent pi = PendingIntent.getActivity(this.context, 0, this.pendingIntent, PendingIntent.FLAG_ONE_SHOT);
                notificationBuilder.setContentIntent(pi);
            }
            if (this.notification_style!=null){
                notificationBuilder.setStyle(this.notification_style);
            }
            if (this.actions!=null){
                for(int i=0; i<this.actions.length; i++){
                    notificationBuilder.addAction(this.actions[i]);
                }
            }
            NotificationManager notificationManager = (NotificationManager) this.context.getSystemService(Context.NOTIFICATION_SERVICE);

            notificationManager.notify(this.notification_id, notificationBuilder.build());
        }
    }

    public Intent makePendingIntent(Bundle data, String class_key, String data_key){
        Intent pi = null;
        if(data.getString(class_key)==null)
            return null;
        String target_class = data.getString(class_key);
        try{
            pi = new Intent(this.context, Class.forName(target_class));
            String extra_data = data.getString(data_key);
            if (extra_data!=null){
                JSONObject target_data = new JSONObject(data.getString(data_key));
                Iterator<?> keys = target_data.keys();

                while( keys.hasNext() ) {
                    String key = (String)keys.next();
                    if ( target_data.get(key) instanceof String ) {
                        pi.putExtra(key, (String) target_data.get(key));
                    }
                    if ( target_data.get(key) instanceof Integer ) {
                        pi.putExtra(key, (Integer)target_data.get(key));
                    }
                    if ( target_data.get(key) instanceof Boolean ) {
                        pi.putExtra(key, (Boolean)target_data.get(key));
                    }
                }
            }

        }catch(JSONException e) {
            return null;
        }catch(ClassNotFoundException e){
            return null;
        }catch (Exception e){
            return null;
        }
        return pi;
    }

    public NotificationCompat.Action[] addActions(Bundle data){
        NotificationCompat.Action[] action_items = null;
        if(data.getString("ac")==null)
            return null;
        try{
            JSONArray actions = new JSONArray(data.getString("ac"));
            int total_actions = actions.length()>2?3:actions.length();
            action_items = new NotificationCompat.Action[total_actions];
            for(int i = 0 ; i < total_actions; i++){
                JSONObject obj = actions.getJSONObject(i);
                String obj_class = obj.getString("cl");
                String obj_title = obj.getString("tl");
                int obj_icon = 0;
                try{
                    obj_icon = this.context.getResources().getIdentifier(obj.getString("ic"), "drawable", this.context.getPackageName());
                }catch (Exception e){
                    Log.w(TAG, "Action button icon error. Unable to add action button.");
                    continue;
                }

                JSONObject obj_extra = null;
                try{
                    obj_extra = obj.getJSONObject("dt");
                }catch (JSONException e){

                }

                if(obj_class!=null && obj_icon!=0 && obj_title!=null){
                    Intent ai = new Intent(this.context, Class.forName(obj_class));
                    if (obj_extra!=null){
                        Iterator<?> keys = obj_extra.keys();

                        while( keys.hasNext() ) {
                            String key = (String)keys.next();
                            if ( obj_extra.get(key) instanceof String ) {
                                ai.putExtra(key, (String)obj_extra.get(key));
                            }
                            if ( obj_extra.get(key) instanceof Integer ) {
                                ai.putExtra(key, (Integer)obj_extra.get(key));
                            }
                            if ( obj_extra.get(key) instanceof Boolean ) {
                                ai.putExtra(key, (Boolean)obj_extra.get(key));
                            }
                        }
                    }
                    PendingIntent pi = PendingIntent.getActivity(this.context, 0, ai, PendingIntent.FLAG_ONE_SHOT);
                    ai.addFlags(Intent.FLAG_ACTIVITY_CLEAR_TOP);
                    NotificationCompat.Action ac = new NotificationCompat.Action(obj_icon, obj_title, pi);
                    action_items[i] = ac;
                }
            }
        }catch(JSONException e) {
            Log.w(TAG, "Tag JSON error: "+e.toString());
            return null;
        }catch(ClassNotFoundException e){
            Log.w(TAG, "Tag Class error: "+e.toString());
            return null;
        }catch (Exception e){
            Log.w(TAG, "Tag error: "+e.toString());
            return null;
        }
        return action_items;
    }

    public NotificationCompat.Style makeNotifStyle(Bundle data){
        NotificationCompat.Style style = null;
        if(data.getString("st")!=null && data.getString("st").equals("big_picture")){
            String imgUrl = data.getString("bi");
            Bitmap imgBit = getBitmapFromURL(imgUrl);
            if (imgBit!=null) {
                style = new NotificationCompat.BigPictureStyle().bigPicture(imgBit);
            }
        }
        if(data.getString("st")!=null && data.getString("st").equals("big_text")){
            String bigText = data.getString("bt");

            if (bigText!=null) {
                style = new NotificationCompat.BigTextStyle().bigText(bigText);
            }
        }
        return style;
    }

    public static Bitmap getBitmapFromURL(String src) {
        if (src==null) return null;
        try {
            URL url = new URL(src);
            HttpURLConnection connection = (HttpURLConnection) url.openConnection();
            connection.setDoInput(true);
            connection.connect();
            InputStream input = connection.getInputStream();
            Bitmap myBitmap = BitmapFactory.decodeStream(input);
            return myBitmap;
        } catch (IOException e) {
            // Log exception
            return null;
        }
    }
}
